'''
    Utility methods for processing raw sections easily.
    Low level interface, rather not used directly in configurators.

    @organization: ICM UW

    @todo: take a look at backup made by normal section when using raw section
           as a subsection
'''
import os
import re
from subprocess import call

from ucfg_utils import info, error, backupFile, getFile, copyPreserve
from ucfg_main import ConfigUtilities


RAW_SECTION_PATTERN = '.+/(((props|shell|append)/.+)' + \
                    '|' + '(xml/.*(wsrflite.xml|xnjs_legacy.xml|xnjs.xml)))'


def processRawSection(options, config, section):
    '''
        Process raw section.
    '''
    if not re.match(RAW_SECTION_PATTERN, section):
        return

    # spliting section pattern
    sectionTokens = section.split('/', 2)
    component = sectionTokens[0]
    rawType = sectionTokens[1]
    filename = sectionTokens[2]

    paramsDict = __getOnlySectionItems(options, config, section)
    __setEmptyEntriesToNone(paramsDict)

    # apply changes
    cfgUtil = ConfigUtilities(options, component)
    if rawType == 'props':
        cfgUtil.setJavaProperties(filename, paramsDict)
    elif rawType == 'shell':
        cfgUtil.setShellVariables(filename, paramsDict)
    elif rawType == 'xml':
        __processRawXMLSection(options, cfgUtil, filename, paramsDict)
    elif rawType == 'append':
        __processRawAppendSection(options, cfgUtil, filename, paramsDict)
    else:
        # this should never happen
        error('Wrong section raw type')


def __processRawXMLSection(options, cfgUtil, filename, paramsDict):
    if filename.endswith('xnjs.xml') or filename.endswith('xnjs_legacy.xml'):
        cfgUtil.setXNJSProperties(filename, paramsDict)
    elif filename.endswith('wsrflite.xml'):
        cfgUtil.setWSRFLiteProperties(filename, paramsDict)
    else:
        # this should never happen
        error("UNICORE configuration file '" + filename + "' not supported!")


def __processRawAppendSection(options, cfgUtil, filename, paramsDict):
    linePrefixKey = 'line.'
    linesList = []
    for key in sorted(paramsDict):
        if key.startswith(linePrefixKey):
            #cfgUtil.appendTextToFile(filename, paramsDict[key])
            linesList = linesList + [ paramsDict[key] ]
        else:
            info(options, " - unrecognized key '" + key + "'")
    if len(linesList) > 0:
        cfgUtil.appendLinesIfNotAlreadyExist(filename, linesList)


def __setEmptyEntriesToNone(paramsDict):
    for key in paramsDict:
        if paramsDict[key] == '':
            paramsDict[key] = None


def __getOnlySectionItems(options, config, section):
    '''
        Gets only section parameters (removing DEFAULTS keys).
    '''
    paramsList = config.items(section)
    paramsDict = dict(paramsList)
    for p in config.defaults():
        paramsDict.pop(p)
    return paramsDict


def removeUnnecessaryFiles(options, component, filesList):
    '''
        Removes listed files (and moving it to /tmp directory).
        
        @param options:    program options
        @param component:  component name
        @param filesList:  list of files
    '''
    info(options, "Removing unnecessary files")
    backupRootDir = '/tmp/unicore' + options.backupsuffix + '/' + component
    if not os.path.exists(backupRootDir) and options.dry == False:
        os.makedirs(backupRootDir)
    #backupRootDir = tempfile.mkdtemp(prefix = '/tmp/unicore/' + component  )
    for originalFile in filesList:
        filePath = getFile(options, component, originalFile)
        timestampName = backupRootDir + '/' + os.path.basename(originalFile)
        if os.path.isfile(filePath):
            info(options, " - moving file '%s' to '%s'" % (filePath, timestampName))
            copyPreserve(options, filePath, timestampName, True, True)
        else:
            info(options, " - file '" + filePath + "' already not exists")
    info(options, "Finished files clean up")


def copyAdditionalFiles(options, filesDict):
    info(options, "Copying additional files")
    for originalFile in filesDict:
        if re.match('^/|^[a-zA-Z]:', originalFile):
            srcFilePath = originalFile
        else:
            srcFilePath = options.configuratorsPath + '../files/' + originalFile
        destFilePath = filesDict[originalFile]

        if os.path.isfile(destFilePath):
            info(options, " - making backup of '%s'" % destFilePath)
            backupFile(options, destFilePath)
        info(options, " - copying file from '%s' to '%s'" % (originalFile, destFilePath))
        copyPreserve(options, srcFilePath, destFilePath, False, True)
    info(options, "Finished adding new files")

