import os

import ucfg_utils
import ucfg_use


'''
    This class is an entry point for reconfiguration utilities.
'''
class ConfigUtilities(object):

    def __init__(self, options, component):
        self.options = options
        self.component = component
        self.backedup = set([])


    def setJavaProperty(self, filename, propertyName, newValue):
        '''
            Sets a single Java property (subsequent will be commented).
            If the file doesn't exist it will be created
            
            @param   filename:    filename to change
            @param   property:    java property to change
            @param   newValue:    new value of that property
        '''
        self.setJavaProperties(filename, { propertyName : newValue })


    def setJavaProperties(self, filename, propsDict):
        '''
            Sets property, if it finds commented one it replaces the first occurrence.
            The rest, if detected, will be commented out.
            If the file doesn't exist it will be created
            
            @param   filename:    filename to change
            @param  propsDict:    dictionary with java properties to set
        
            @todo: handle keys with spaces
            @todo: handle comments after key & value pair
        '''
        filepath = self._prepare(filename)
        if not os.path.exists(filepath):
            open(filepath, 'w').close()
        ucfg_utils.setJavaProperties(self.options, filepath, propsDict)

    def updateJavaPropertyNames(self, filename, propsDict):
        '''
            Updates property names
            
            @param   filename:    filename to change
            @param  propsDict:    dictionary with java properties to to have names replaced
        
            @todo: handle keys with spaces
            @todo: handle comments after key & value pair
        '''
        filepath = self._prepare(filename)
        if not os.path.exists(filepath):
            open(filepath, 'w').close()
        ucfg_utils.processJavaProperties(self.options, filepath, propsDict, 'key')

    def commentJavaProperties(self, filename, propsDict):
        '''
            Comments properties
            
            @param   filename:    filename to change
            @param  propsDict:    dictionary with java properties to be commented with optional extra clarifications
        
            @todo: handle keys with spaces
        '''
        filepath = self._prepare(filename)
        if not os.path.exists(filepath):
            open(filepath, 'w').close()
        ucfg_utils.processJavaProperties(self.options, filepath, propsDict, 'comment')

    def getJavaProperty(self, filename, name):
        '''
            Gets java property value
            
            @param   filename:    filename to change
            @param  name:    property key
        
            @todo: handle keys with spaces
            @todo: handle comments after key & value pair
        '''
        filepath = self._prepare(filename)
        return ucfg_utils.getJavaProperty(filepath, name)

    def getJavaPropertyKeys(self, filename):
        '''
            Gets java property keys list
            
            @param   filename:    filename 
        
            @todo: handle keys with spaces
            @todo: handle comments after key & value pair
        '''
        filepath = self._prepare(filename)
        return ucfg_utils.getJavaPropertyKeys(filepath)

    def setShellVariable(self, filename, variable, newValue):
        '''
            Sets a single shell configuration variable (subsequent will be commented).
        
            @param   filename:    filename to change
            @param   variable:    java property to change
            @param   newValue:    new value of that property
        '''
        self.setShellVariables(filename, { variable : newValue })


    def setShellVariables(self, filename, varsDict):
        '''
           Sets multiple shell configuration variables. Subsequent occurrences of those being set 
           will be commented.
             
           @param   filename:    filename to change
           @param  propsDict:    dictionary with shell variables to set    
            
           @todo: handle mutlilines (when line ends with backslash)
        '''
        filepath = self._prepare(filename)
        ucfg_utils.setShellVariables(self.options, filepath, varsDict)


    def setXMLAttribute(self, filename, xpath, name, value):
        '''
            Sets xml attribute value
        '''
        filepath = self._prepare(filename)
        if not os.path.exists(filepath):
            return
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        ucfg_utils.setXMLElementAttribute(self.options, doc, xpath, name, value)
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)
        

    def removeXPathElements(self, filename, xPathExpression, nsPrefixMap = {}):
        '''
            Removes elements pointed by an XPath expression. 
            
            @param            file:    file to change  
            @param xPathExpression:    XPath expression 
            @param     nsPrefixMap:    prefix map of namespaces used in XPath  
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        ucfg_utils.removeXPathElements(self.options,
                                       doc, xPathExpression, nsPrefixMap)
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)



    def setWSRFLiteProperty(self, filename, name, value):
        '''
            Sets property element in WSRFLite configuration file.
            
            @param        file:    file to change  
            @param        name:    property's name
            @param       value:    property's value
        '''
        self.setWSRFLiteProperties(filename, {name : value})



    def setWSRFLiteProperties(self, filename, valsDict):
        '''
            Sets multiple properties in WSRFLite configuration file.
            
            @param    filename:    file to change  
            @param    valsDict:    properties
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        for p in sorted(valsDict):
            ucfg_use.setWSRFLiteProperty(self.options, doc, p, valsDict[p], 'value')
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)

    def setWSRFLitePropertyKeys(self, filename, valsDict):
        '''
            Updates keys of multiple properties in WSRFLite configuration file.
            
            @param    filename:    file to change  
            @param    valsDict:    properties
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        for p in sorted(valsDict):
            ucfg_use.setWSRFLiteProperty(self.options, doc, p, valsDict[p], 'key')
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)

    def commentWSRFLiteProperties(self, filename, valsDict):
        '''
            Comments properties in WSRFLite configuration file.
            
            @param    filename:    file to change  
            @param    valsDict:    properties
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        for p in sorted(valsDict):
            ucfg_use.setWSRFLiteProperty(self.options, doc, p, valsDict[p], 'comment')
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)

    def getWSRFLiteProperties(self, filename):
        '''
            Returns all properties defined in WSRFLite configuration file.
            
            @param    filename:    file  
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        return doc.xpath("/services/property")

    def setXNJSProperty(self, filename, name, value):
        '''
            Sets property element in XNJS configuration file.
        
            @param        file:    file to change  
            @param        name:    property's name
            @param       value:    property's value
        '''
        self.setXNJSProperties(filename, {name : value})



    def setXNJSProperties(self, filename, valsDict):
        '''
            Sets multiple properties in XNJS configuration file.
        
            @param    filename:    file to change  
            @param    valsDict:    properties
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        for prop in sorted(valsDict):
            ucfg_use.setXNJSEngineProperty(self.options, doc, prop, valsDict[prop])
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)


    def toggleWSRFLiteService(self, filename, serviceName, enabled = True):
        '''
            Sets enabled attribute in WSRFLite configuration file.
            
            @param        filename:    file
            @param  serviceName:    service name 
            @param        enabled:    if service is enabled (default: True)
        '''
        self.toggleWSRFLiteServices(filename, {serviceName: enabled})



    def toggleWSRFLiteServices(self, filename, servicesDict):
        '''
            Sets enabled attribute in WSRFLite configuration file.
            
            @param        filename: file
            @param   servicesDict: pairs, service name, Boolean 
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        for prop in sorted(servicesDict):
            ucfg_use.toggleWSRFLiteService(self.options,
					doc, prop, servicesDict[prop])
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)


    def appendXNJSProcessor(self, filename, processorClass):
        '''
            Appends processor to the JSDL Processing Chain.
        '''
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        ucfg_use.appendToJSDLProcessingChain(self.options, doc, processorClass)
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)

        
    def addXMLStringElementAfterXPath(self, filename, xmlString2Add, xPathPosition = None, nsPrefixMap = {}):
        filepath = self._prepare(filename)
        doc = ucfg_utils.loadXMLDocumentFromFile(self.options, filepath)
        ucfg_utils.addAdditionalXML(self.options, doc, xmlString2Add, xPathPosition, nsPrefixMap)
        ucfg_utils.writeXMLDocument(self.options, doc, filepath)


    def appendLinesIfNotAlreadyExist(self, filename, linesList):
        '''
            Appends lines to the file, but only if a line did not exist so far. 
        '''
        filepath = self._prepare(filename)
        ucfg_utils.appendLinesIfNotAlreadyExist(self.options, filepath, linesList)



    def removeLinesIfExist(self, filename, linesList):
        '''
            Removes existing lines. 
        '''
        filepath = self._prepare(filename)
        ucfg_utils.removeLinesIfExist(self.options, filepath, linesList)



    def replaceFileLines(self, filename, linesDict):
        '''
            Replaces existing lines.
        '''
        filepath = self._prepare(filename)
        ucfg_utils.replaceFileLines(self.options, filepath, linesDict)



    def _prepare(self, filename):
        '''
            Prepares backup if needed and returns recolved full path.
        '''
        filepath = ucfg_utils.getFile(self.options, self.component, filename);
        if not(filepath in self.backedup) and os.path.isfile(filepath):
            ucfg_utils.backupFile(self.options, filepath)
            self.backedup.add(filepath)
        return filepath

