""" 
    Base class of all configurators, must be extended.
"""

class UConfiguratorBase(object):
    def getParams(self):
        """
            Dictionary with known variables, need to be overridden.
        """
        pass


    def run(self, options, config, section, params):
        """
            Needs to be overriden to provide an actual implementation. 
            
            Parameters: <parsed command line options> <parsed configuration> <configuration section of this module>
        """
        pass

    def getTemplate(self, section):
        """
            Returns a template of configurator's configuration with all the
            available parameters.
        """
        ret = "[" + section + "]\n\n"
        sortedKeys = sorted(self.getParams().keys())
        for param in sortedKeys:
            value = self.getParams().get(param)
            if value == None:
                value = ''
            splitVal = value.split("##", 1)
            if len(splitVal) > 1:
                value = splitVal[0]
                ret += "# "
                if value == "__REQUIRED":
                    ret += "(REQUIRED) "
                ret += splitVal[1] + "\n"
            if value == '__NONE' or value == "__REQUIRED":
                param = "#"+param
                value = ''
            
            ret += param + "=" + value + "\n\n"
        return ret

    #Useful methods for (nearly) all configurators

    def prepareCredAndTrustDict(self, prefix, params):
        secProps = {
            prefix+'.credential.path'             : params[prefix+'.credential.path'],
            prefix+'.credential.password'        : params[prefix+'.credential.password'],
            prefix+'.credential.keyPath'        : params[prefix+'.credential.keyPath'],
            prefix+'.truststore.type'             : params[prefix+'.truststore.type'],
            prefix+'.truststore.allowProxy'        : params[prefix+'.truststore.allowProxy'],
        }

        tsType = params[prefix+'.truststore.type']
        if tsType == 'keystore':
            secProps[prefix+'.truststore.keystorePath'] = params[prefix+'.truststore.location']
            secProps[prefix+'.truststore.keystorePassword'] = params[prefix+'.truststore.keystorePassword']
            secProps[prefix+'.truststore.crlLocations.1'] = params[prefix+'.truststore.crlLocation']
        elif tsType == 'openssl':
            secProps[prefix+'.truststore.opensslPath'] = params[prefix+'.truststore.location']
        elif tsType == 'directory':
            secProps[prefix+'.truststore.directoryLocations.1'] = params[prefix+'.truststore.location']
            secProps[prefix+'.truststore.crlLocations.1'] = params[prefix+'.truststore.crlLocation']
        return secProps

    def prepareStartupDict(self, prefix, params):            
        return {
            'JAVA'      : params[prefix+'.javaCommand'],
            'MEM'       : '"-Xmx' + params[prefix+'.memory'] + '"',
            'JMX_PORT'  : '"' + params[prefix+'.jmxPort'] + '"'
        }

    def addCredAndTrustParams(self, prefix, params):
        params[prefix+'.credential.path'] = "__REQUIRED##Path to the file with the credential"
        params[prefix+'.credential.password'] =  "__NONE##Password of the credential (or leave undefined if not encrypted)"
        params[prefix+'.credential.keyPath'] = "__REQUIRED##Path to the pem file with the private key, used if the credential was given as a PEM certificate"
            
        params[prefix+'.truststore.type'] =  "__REQUIRED##Type of truststore: keystore, openssl or directory"
        params[prefix+'.truststore.location'] =  "__REQUIRED##Location of the truststore dependent on its type: path to the keystore file, path to openssl directory or wildcard expression with CAs locations"
        params[prefix+'.truststore.crlLocation'] = "##Location of CRL files given as wildcard expression. Not used for openssl truststore."
        params[prefix+'.truststore.keystorePassword'] = "##Password of the keystore if it is used as a truststore"
        params[prefix+'.truststore.allowProxy'] = 'ALLOW##Whether to accept proxy certificates (ALLOW or DENY)'
        return params

    def addStartupParams(self, prefix, params, mem):
        params[prefix+'.jmxPort'] = '##JMX command port or leave empty to disable JMX'
        params[prefix+'.memory'] = mem+'m##Maximum Java memory to be used for the process'
        params[prefix+'.javaCommand'] = 'java##Path to java command'
        return params

    # the rest is used internally, no need to bother
    def loadParams(self, options, config, section):
        paramsDict = self._loadParamsNoCheck(options, config, section)
        self._stripValuesExternalQuotationMarks(paramsDict)
        self._checkParams(paramsDict, section)
        return paramsDict

    def _stripValuesExternalQuotationMarks(self, paramsDict):
        for paramKey in paramsDict:
            paramVal = paramsDict[paramKey].strip()
            if len(paramVal) > 2 and paramVal.startswith('"') and paramVal.endswith('"'):
                paramVal = paramVal[1:-1]
            paramsDict[paramKey] = paramVal

    def _loadParamsNoCheck(self, options, config, section):
        paramsList = config.items(section)
        paramsDict = dict(paramsList)
        return paramsDict

    def _checkParams(self, paramsDict, section):
        from ConfigParser import NoOptionError
        for param in self.getParams().keys():
            value = self.getParams().get(param)
            if value.startswith("__REQUIRED") and not param in paramsDict.keys():
                raise NoOptionError(param, section)

        return paramsDict
