#!/usr/bin/env python
'''
	Script allows for reconfiguration of UNICORE component, using a simple to
	maintain and update files.

	@organization: ICM UW
	@author: K. Benedyczak @: golbi AT icm.edu.pl
'''
import sys
import os
import glob
import fnmatch
import re

from datetime import datetime
from ConfigParser import SafeConfigParser, MissingSectionHeaderError
from optparse import OptionParser


def detectInstallationType(options):
    mypath = os.path.abspath(__file__)
    if mypath.startswith("/usr/bin") | mypath.startswith("/usr/local/bin"):
        options.systemInstall = True
    else:
        options.systemInstall = False

    if options.systemInstall:
        options.configuratorsPath = "/usr/share/unicore/configurator/configurator.d/"
        sys.path.append("/usr/share/unicore/configurator/modules/")
        options.configsRoot = "/etc/unicore/"
    else:
        options.configuratorsPath = os.path.dirname(mypath) + "/configurator.d/"
        options.configsRoot = os.path.dirname(mypath) + "/"
        sys.path.append(os.path.dirname(mypath) + "/modules/")
    sys.path.append(options.configuratorsPath)


def listConfigurators(options):
    print "Configurators available in the directory " + options.configuratorsPath + ":\n"
    dirList = glob.glob(options.configuratorsPath + "*.py")
    for fname in dirList:
        name = os.path.basename(fname)[:-len(".py")]
        if not name.startswith("_"):
            print os.path.basename(fname)[:-len(".py")]


def processConfigFile(options, filename):
    info(options, "Processing configuration file: " + filename)
    #read configuration file
    config = SafeConfigParser()
    #make the parser case-sensitive
    config.optionxform = str
    try:
        config.readfp(open(filename))
        sections = config.sections()
        #needs to sort the sections because of unordered dictionary in python 2.4 :/
        for section in sorted(sections):
            if section.startswith('_'):
                info(options, "  Hidden section '" + section[1:] + "'. Skipping it.")
            else:
                processConfigurator(options, config, section)
    except MissingSectionHeaderError, inst:
        error("Error processing file '" + filename + "'. Reason: " + str(inst))
    except IOError, (errno, strerror):
        error("Processing configuration " + filename + ": " + "I/O error(%s): %s" % (errno, strerror))


def processConfigurator(options, config, section):
    info(options, "  Processing configuration for: " + section)
    options.indent = "    "
    from ucfg_raw import RAW_SECTION_PATTERN
    try:
        if re.match(RAW_SECTION_PATTERN, section):
            from ucfg_raw import processRawSection
            processRawSection(options, config, section)
        else:
            module = __import__(section)
            configuratorObj = module.UConfigurator()
            sectionParams = configuratorObj.loadParams(options, config, section)
            params = configuratorObj.getParams().copy()
            for key in params.keys():
            	val = params.get(key)
            	if val == None:
            		continue
            	split = val.split("##")
            	if len(split) > 1:
            		params[key] = split[0]
            	else:
            		params[key] = ""
            params.update(sectionParams)
            configuratorObj.run(options, config, section, params)
    except ImportError, inst:
        error("Importing '" + section + "' failed. Skipping it... Reason: " + str(inst))
    except Exception, inst:
        error("Processing '" + section + "' failed. Skipping it... Reason: " + str(inst))
        import traceback
        traceback.print_exc()

    options.indent = ""
    info(options, "  ------------DONE: " + section + "------------ ")


def printConfiguratorTemplate(options, configurator):
    print "# ---------- Template configuration for: " + configurator
    try:
        module = __import__(configurator)
        configuratorObj = module.UConfigurator()
        print configuratorObj.getTemplate(configurator)
    except ImportError:
        error("Importing '" + configurator + "' failed. Skipping it...")


def revert(options, suffix):
    if options.manualConfigurationRoot != None:
        dir = options.manualConfigurationRoot
    else:
        dir = options.configsRoot
    info(options, "Reverting configuration files under " + dir + " to backups with suffix " + suffix)
    for root, dirnames, filenames in os.walk(dir):
        for filename in fnmatch.filter(filenames, '*' + suffix):
            full = os.path.join(root, filename)
            stripped = full[:-len(suffix)]
            info(options, filename + " -> " + stripped)
            copyPreserve(options, full, stripped, True, False)



usage = "This program configures UNICORE components using a simple and easy to maintain files with parameters.\n\nInvocation: %prog [options] [configurationFile1 ...]"
parser = OptionParser(usage = usage)
parser.add_option("-q", "--quiet",
		action = "store_true", dest = "quiet", default = False,
		help = "Disable any diagnostic output, only errors will be printed")
parser.add_option("-d", "--dry-run",
		action = "store_true", dest = "dry", default = False,
		help = "Simulates configuration change but nothing is written to disk")
parser.add_option("-n", "--no-backup",
		action = "store_false", dest = "backup", default = True,
		help = "Disables creation of backup files")
parser.add_option("-B", "--always-backup",
		action = "store_true", dest = "backupAlways", default = False,
		help = "Creates a backup always, even if one for the changed file exists. The backup has a timestamp appended.")
parser.add_option("-l", "--list",
		action = "store_true", dest = "listConfigurators", default = False,
		help = "Instead of processing configuration files, the list of available configurators is printed.")
parser.add_option("-t", "--template",
		action = "store_true", dest = "printTemplate", default = False,
		help = "Instead of processing configuration files, a reference with all available parameters is printed in a form of a template configuration.")
parser.add_option("-c", "--configurationRoot",
		dest = "manualConfigurationRoot", default = None,
		help = "If specified the components configuration directory will not be autodetected, it will be assumed that all files to be reconfigured are rooted in the provided directory. Note that this option makes sense usually only when configuring a single component.")
parser.add_option("-r", "--revert",
		dest = "revert", default = None,
		help = "If specified, must contain a suffix which was added to backed files. All configuration files in ALL components will be reverted to backups with the given suffix. If you want to limit the directories reverted use -c.")

(options, args) = parser.parse_args()
options.indent = ""
options.parser = None
options.backupsuffix = datetime.now().strftime("_%Y%m%d-%H%M%S")

detectInstallationType(options)
from ucfg_utils import info, error, getConfigRoot, copyPreserve

if options.listConfigurators:
    listConfigurators(options)
    sys.exit(0)

if options.revert:
    revert(options, options.revert)
    sys.exit(0)

if len(args) == 0:
    if options.printTemplate:
        print "ERROR: You must provide configurator name for which the template should be printed\n"
    else:
        print "ERROR: You must provide configuration file name(s) as argument(s)\n"
    parser.print_help()
    sys.exit(2)

if options.manualConfigurationRoot != None:
    if not options.manualConfigurationRoot.endswith("/"):
        options.manualConfigurationRoot = options.manualConfigurationRoot + "/"

for config in args:
    if options.printTemplate:
        printConfiguratorTemplate(options, config)
    else:
        processConfigFile(options, config)


