#!/usr/bin/env python
'''
	Script allows for update of configuration syntax of UNICORE component.
	It does not change the configuration. 

	@organization: ICM UW
	@author: K. Benedyczak @: golbi AT icm.edu.pl
'''
import sys
import os
import glob
import fnmatch
import re

from datetime import datetime
from ConfigParser import SafeConfigParser, MissingSectionHeaderError
from optparse import OptionParser


def detectInstallationType(options):
    mypath = os.path.abspath(__file__)
    if mypath.startswith("/usr/bin") | mypath.startswith("/usr/local/bin"):
        options.systemInstall = True
    else:
        options.systemInstall = False

    if options.systemInstall:
        options.updatersPath = "/usr/share/unicore/configurator/updater.d/"
        sys.path.append("/usr/share/unicore/configurator/modules/")
        options.configsRoot = "/etc/unicore/"
    else:
        options.updatersPath = os.path.dirname(mypath) + "/updater.d/"
        options.configsRoot = os.path.dirname(mypath) + "/"
        sys.path.append(os.path.dirname(mypath) + "/modules/")
    sys.path.append(options.updatersPath)


def listUpdaters(options):
    print "Updaters available in the directory " + options.updatersPath + ":\n"
    dirList = glob.glob(options.updatersPath + "*.py")
    options.indent="   "
    for fname in dirList:
        name = os.path.basename(fname)[:-len(".py")]
        if not name.startswith("_"):
            try:
                module = __import__(name)
                updaterObj = module.UUpdater()
                print(name)
                updaterObj.printDescription(options)
            except ImportError, inst:
                error("Importing '" + name + "' failed. Skipping it... Reason: " + str(inst))
            except Exception, inst:
                error("Processing '" + name + "' failed. Skipping it... Reason: " + str(inst))
                import traceback
                traceback.print_exc()
    options.indent=""


def processUpdate(options, updater):
    info(options, "NOTE This tool is updating configuration files syntax on a best basis.")
    info(options, "You should:")
    info(options, " - carefully examine all messages, especially those with WARNINGs:")
    info(options, " - be aware that many of the currently commented properties won't be updated")
    info(options, " - be aware that some, advanced and not default properties won't be updated and must be fixed manually")
    info(options, " - be aware that none of the property descriptions which are in comments in the configuration files will be updated, and so the descriptions might remain outdated")
    info(options, "")
    
    info(options, "  Invoking updater: " + updater)
    options.indent = "    "
    try:
        module = __import__(updater)
        updaterObj = module.UUpdater()
        updaterObj.run(options)
    except ImportError, inst:
        error("Importing '" + updater + "' failed. Skipping it... Reason: " + str(inst))
    except Exception, inst:
        error("Processing '" + updater + "' failed. Skipping it... Reason: " + str(inst))
        import traceback
        traceback.print_exc()

    options.indent = ""
    info(options, "  ------------DONE: " + updater + "------------ ")


def printConfiguratorTemplate(options, configurator):
    print "# ---------- Template configuration for: " + configurator
    try:
        module = __import__(configurator)
        configuratorObj = module.UConfigurator()
        print configuratorObj.getTemplate(configurator)
    except ImportError:
        error("Importing '" + configurator + "' failed. Skipping it...")


def revert(options, suffix):
    if options.manualConfigurationRoot != None:
        dir = options.manualConfigurationRoot
    else:
        dir = options.configsRoot
    info(options, "Reverting configuration files under " + dir + " to backups with suffix " + suffix)
    for root, dirnames, filenames in os.walk(dir):
        for filename in fnmatch.filter(filenames, '*' + suffix):
            full = os.path.join(root, filename)
            stripped = full[:-len(suffix)]
            info(options, filename + " -> " + stripped)
            copyPreserve(options, full, stripped, True, False)



usage = "This program updates UNICORE components configuration to a newer syntax. It does not change any configuration values.\n\nInvocation: %prog [options] [updaterName ...]"
parser = OptionParser(usage = usage)
parser.add_option("-q", "--quiet",
		action = "store_true", dest = "quiet", default = False,
		help = "Disable any diagnostic output, only errors will be printed")
parser.add_option("-d", "--dry-run",
		action = "store_true", dest = "dry", default = False,
		help = "Simulates configuration change but nothing is written to disk")
parser.add_option("-n", "--no-backup",
		action = "store_false", dest = "backup", default = True,
		help = "Disables creation of backup files")
parser.add_option("-B", "--always-backup",
		action = "store_true", dest = "backupAlways", default = False,
		help = "Creates a backup always, even if one for the changed file exists. The backup has a timestamp appended.")
parser.add_option("-l", "--list",
		action = "store_true", dest = "listUpdaters", default = False,
		help = "Instead of performing update, the list of available updaters is printed.")
parser.add_option("-c", "--configurationRoot",
		dest = "manualConfigurationRoot", default = None,
		help = "If specified the components configuration directory will not be autodetected, it will be assumed that all files to be reconfigured are rooted in the provided directory. Note that this option makes sense usually only when configuring a single component.")
parser.add_option("-r", "--revert",
		dest = "revert", default = None,
		help = "If specified, must contain a suffix which was added to backed files. All configuration files in ALL components will be reverted to backups with the given suffix. If you want to limit the directories reverted use -c.")

(options, args) = parser.parse_args()
options.indent = ""
options.parser = None
options.backupsuffix = datetime.now().strftime("_%Y%m%d-%H%M%S")

detectInstallationType(options)
from ucfg_utils import info, error, getConfigRoot, copyPreserve

if options.listUpdaters:
    listUpdaters(options)
    sys.exit(0)

if options.revert:
    revert(options, options.revert)
    sys.exit(0)

if len(args) == 0:
    print "ERROR: You must provide updater name(s) as argument(s)\n"
    parser.print_help()
    sys.exit(2)

if options.manualConfigurationRoot != None:
    if not options.manualConfigurationRoot.endswith("/"):
        options.manualConfigurationRoot = options.manualConfigurationRoot + "/"

for updater in args:
    processUpdate(options, updater)


