import arc, os, pipes, shutil
import nagutils, vomsutils
from arcnagios import persistence
from arcnagios.jobplugins import load_jobplugin
from arcutils import ParseError

def write_jsdl(jsdl_path, script_path, *script_args, **kwargs):
    """Write a JSDL job description for executing ``script_path`` with
    arguments ``script_args``, using the additional job attributes specified
    in ``kwargs``::

	job_name		- The job name.
	application_name	- The Application name.
	output			- File in which to store stdout.
	error			- File in which to store stderr.
	wall_time_limit		- Soft limit on the execution time.
	memory_limit		- The max. amount of memory used by the job.
	staged_inputs		- A list of triples specifying staged inputs.
	staged_outputs		- A list of triples specifying staged outputs.
	runtime_environments	- A list of runtime environments to request.
    """

    fh = open(jsdl_path, 'w')
    script_name = os.path.basename(script_path)
    _command = '<Executable>%s</Executable>'%script_name \
	+ ''.join(['\n\t<Argument>%s</Argument>'%arg for arg in script_args])
    kwargs.update(_command = _command)
    if not 'output' in kwargs:
	kwargs['output'] = 'stdout.txt'
    if not 'error' in kwargs:
	kwargs['error'] = 'stderr.txt'
    kwargs['script_name'] = script_name
    kwargs['script_path'] = script_path
    fh.write("""\
<?xml version="1.0" encoding="UTF-8"?>
<JobDefinition xmlns="http://schemas.ggf.org/jsdl/2005/11/jsdl"
	       xmlns:jsdl-arc="http://www.nordugrid.org/ws/schemas/jsdl-arc">
  <JobDescription>
    <JobIdentification>
      <JobName>%(job_name)s</JobName>
    </JobIdentification>
    <Application>
      <ApplicationName>%(application_name)s</ApplicationName>
      <POSIXApplication xmlns="http://schemas.ggf.org/jsdl/2005/11/jsdl-posix">
	%(_command)s
        <Output>%(output)s</Output>
        <Error>%(error)s</Error>
	<WallTimeLimit>%(wall_time_limit)d</WallTimeLimit>
	<MemoryLimit>%(memory_limit)d</MemoryLimit>
      </POSIXApplication>
    </Application>
    <DataStaging>
      <FileName>%(script_name)s</FileName>
      <Source><URI>file:%(script_path)s</URI></Source>
      <CreationFlag>overwrite</CreationFlag>
    </DataStaging>\n""" % kwargs)
    for filename, url, urloptions in kwargs.get('staged_inputs', []):
	fh.write("""\
    <DataStaging>
      <FileName>%s</FileName>
      <Source><URI>%s</URI>%s</Source>
      <DeleteOnTermination>false</DeleteOnTermination>
      <CreationFlag>overwrite</CreationFlag>
    </DataStaging>\n"""
	    % (filename, url,
	       ''.join(["<URIOption>%s</URIOption>"%o for o in urloptions])))
    for filename, url, urloptions in kwargs.get('staged_outputs', []):
	if not url is None:
	    opts = ''.join(["<URIOption>%s</URIOption>"%o for o in urloptions])
	    target = '\n<Target><URI>%s</URI>%s</Target>'%(url, opts)
	else:
	    target = ''
	fh.write("""\
    <DataStaging>
      <FileName>%s</FileName>%s
      <DeleteOnTermination>false</DeleteOnTermination>
      <CreationFlag>overwrite</CreationFlag>
    </DataStaging>\n"""%(filename, target))
    runtime_environments = kwargs.get('runtime_environments', [])
    queue_name = kwargs.get('queue_name', None)
    if runtime_environments or queue_name:
	fh.write("    <Resources>\n")
    if queue_name:
	fh.write("      <jsdl-arc:QueueName>%s</jsdl-arc:QueueName>\n"
		 % queue_name)
    if runtime_environments:
	fh.write("""\
      <jsdl-arc:RunTimeEnvironment>\n""")
	for rte in runtime_environments:
	    fh.write("""\
	<jsdl-arc:Software>
	  <jsdl-arc:Name>%s</jsdl-arc:Name>
	</jsdl-arc:Software>
"""%rte)
	fh.write("""\
      </jsdl-arc:RunTimeEnvironment>\n""")
    if runtime_environments or queue_name:
	fh.write("    </Resources>\n")
    fh.write("""\
  </JobDescription>
</JobDefinition>
""")
    fh.close()

class JobInfo(persistence.PersistentObject):
    persistent_attributes = {
	'submission_time':	persistence.pt_int,
	'host':			persistence.pt_str,
	'job_tag':		persistence.pt_str_opt,
	'termination_service':	persistence.pt_str_opt,
	'job_id':		persistence.pt_str,
	'job_state':		persistence.pt_str,
	'check_time':		persistence.pt_int_opt,
	'fetch_attempts':	persistence.pt_int_opt,
	'stored_urls':		persistence.pt_str_list,
	'tests':		persistence.pt_str_list,
    }

# COMPAT: To load old job descriptions. Only needed for one release to convert
# existing descriptions.
def compat_load_job_info(path):
    active_job_attrs = [
	(int, 'submission_time'),
	(str, 'host'),
	(str, 'job_tag'),
	(str, 'termination_service'),
	(str, 'job_id'),
	(str, 'job_state'),
	(int, 'check_time'),
	(int, 'fetch_attempts'),
    ]

    fh = open(path)
    vs = [ln.strip() for ln in fh]
    fh.close()

    # COMPAT: Addition of job_tag. Remove later.
    if len(vs) == 8:
	vs = vs[0:2] + ['__none'] + vs[2:]

    if len(active_job_attrs) + 1 != len(vs):
	msg = 'Malformed job info %s, expecting %d, not %d elements.' \
		% (path, len(active_job_attrs) + 1, len(vs))
	raise ParseError(msg)
    stored_urls = vs.pop().split()
    d = {}
    for (t, k), v in zip(active_job_attrs, vs):
	if v == '__none':
	    d[k] = None
	else:
	    try:
		d[k] = t(v)
	    except ValueError:
		raise ParseError('Bad value %s for %s in job info %s'
				 %(v, k, path))
    return JobInfo(stored_urls = stored_urls, **d)

def parse_staging(spec):
    """Parse the arguments to the --stage-input and --stage-output options."""

    if ';' in spec:
	xs = spec.split(';')
	spec, urloptions = xs[0], xs[1:]
    else:
	urloptions = []
    if '=' in spec:
	filename, url = spec.split('=', 1)
    else:
	filename, url = os.path.basename(spec), spec
    return (filename, url, urloptions)

def key_value(s):
    kv = s.split('=', 1)
    if len(kv) != 2:
	raise ValueError('Expecting an argument of the form KEY=VALUE.')
    return kv

class JobNagiosPlugin(nagutils.NagiosPlugin, vomsutils.NagiosPluginVomsMixin):
    """Nagios probe to test ARC CEs.  The probe has two sub-commands
    implemented by `check_submit` and `check_monitor`.  The former is run on
    all CEs, while the latter is run to collect submitted jobs."""

    probe_name = 'ARCCE'
    main_config_section = ['arcce']

    JOBID_FILENAME = 'active.jobid'
    JSDL_FILENAME = 'job.jsdl'
    JOB_SCRIPT_FILENAME = 'job.sh'
    JOB_OUTPUT_DIRNAME = 'job_output'
    ACTIVE_JOB_FILENAME = 'active.map'

    _arc_bindir = None

    prev_status = None

    def __init__(self, **kwargs):
	nagutils.NagiosPlugin.__init__(self, **kwargs)
	ap = self.argparser
	ap.add_argument('--fqan', dest = 'fqan')
	ap.add_argument('--top-workdir', dest = 'top_workdir',
		help = 'Parent directory of per-VO probe working directories.')
	ap.add_argument('-O', dest = 'jobtest_options',
		action = 'append', type = key_value, default = [],
		help = 'Given a value of the form VAR=VALUE, binds VAR to '
		       'VALUE in the environment of the job tests.')
	self._user_config = arc.UserConfig()

    def parse_args(self, args):
	nagutils.NagiosPlugin.parse_args(self, args)

    @property
    def top_workdir(self):
	templ = self.opts.top_workdir
	if templ is None:
	    templ = os.path.join(self.opts.plugins_spooldir, 'arcce/%(voms)s')
	return templ % vars(self.opts)

    def workdir_for(self, host, job_tag):
	if job_tag:
	    return os.path.join(self.top_workdir, host + '#' + job_tag)
	else:
	    return os.path.join(self.top_workdir, host)

    def _cleanup_job_state(self, host, job_tag):
	self.log.debug('Cleaning up job files for %s.'%host)
	workdir = self.workdir_for(host, job_tag)
	for filename in [self.ACTIVE_JOB_FILENAME, self.JOBID_FILENAME]:
	    try:
		os.unlink(os.path.join(workdir, filename))
	    except StandardError:
		pass
	try:
	    job_output_dir = os.path.join(workdir, self.JOB_OUTPUT_DIRNAME)
	    if os.path.exists(job_output_dir) and os.listdir(job_output_dir):
		last_dir = job_output_dir + '.LAST'
		shutil.rmtree(last_dir, ignore_errors = True)
		os.rename(job_output_dir, last_dir)
	except StandardError, xc:
	    self.log.warn('Error clearing %s: %s'%(job_output_dir, xc))

    def run_arc_cmd(self, prog, *args, **kwargs):
	if self._arc_bindir:
	    prog = os.path.join(self._arc_bindir, prog)
	cmd = prog + ' ' + ' '.join([pipes.quote(str(arg)) for arg in args])
	self.log.debug('Exec: %s'%cmd)
	fh = os.popen(cmd + ' 2>&1')
	output = fh.read()
	err = fh.close()
	return err or 0, output

    def require_voms_proxy(self):
	proxy_path = vomsutils.NagiosPluginVomsMixin.require_voms_proxy(self)
#	self._user_config.KeyPath(self.opts.user_key)
#	self._user_config.CertificatePath(self.opts.user_cert)
	if proxy_path:
	    self._user_config.ProxyPath(proxy_path)
	try:
	    self._user_config.InitializeCredentials() # old API
	except TypeError:
	    self._user_config.InitializeCredentials(
		    arc.initializeCredentialsType(
			arc.initializeCredentialsType.RequireCredentials))

    def load_active_job(self, host, job_tag):
	"""Load information about the current job on `host : str` tagged with
	`job_tag : str`, or `None` if no information is found."""

	workdir = self.workdir_for(host, job_tag)
	ajf = os.path.join(workdir, self.ACTIVE_JOB_FILENAME)
	if os.path.exists(ajf):
	    self.log.debug('Loading job info from %s.'%ajf)
	    # FIXME: Lock.
	    try:
		jobinfo = JobInfo()
		jobinfo.persistent_load(ajf)
		return jobinfo
	    except ParseError, xc:
		try:
		    jobinfo = compat_load_job_info(ajf)
		    jobinfo.persistent_save(ajf) # Update to new format.
		    return jobinfo
		except Exception:
		    self.log.error('Ignoring invalid job info %s: %s'%(ajf, xc))

    def save_active_job(self, jobinfo, host, job_tag):
	"""Save information about the current job running on `host : str`
	tagged with `job_tag : str`."""

	workdir = self.workdir_for(host, job_tag)
	ajf = os.path.join(workdir, self.ACTIVE_JOB_FILENAME)
	self.log.debug('Saving active job info.')
	# FIXME: Lock.
	jobinfo.persistent_save(ajf)

    def discard_job(self, jobinfo):
	"""Discard the job described by `jobinfo : JobInfo`."""

	job_id = jobinfo.job_id
	self.log.debug('Cancelling old job %s' % job_id)
	rc, output = self.run_arc_cmd('arckill', job_id)
	if rc == 0:
	    return
	self.log.warning('Failed to kill old job, trying to clean instead.')
	rc, output = self.run_arc_cmd('arcclean', job_id)
	if rc == 0:
	    return
	self.log.warning('Failed to clean old job, carrying on.')

    def load_jobtest(self, jobtest_name, **env):
	"""Load a plugin-based job-test from the section of the configuration
	specified by `jobtest_name`.  The result is an instance of `JobPlugin`
	subclass specified by the ``jobplugin`` variable of the given
	section."""

	env.update(self.opts.jobtest_options)

	jobplugin_section = 'arcce.%s'%jobtest_name
	if not self.config.has_section(jobplugin_section):
	    if self.config.has_section('arc-ce.%s'%jobtest_name):
		self.log.warn('The section arc-ce.%s is deprecated, please use %s.'
			      % (jobtest_name, jobplugin_section))
		jobplugin_section = 'arc-ce.%s'%jobtest_name
	    else:
		raise nagutils.ServiceUNKNOWN(
			'Missing configuration section %s for '
			'job-plugin test.' % jobplugin_section)
	jobplugin_name = self.config.get(jobplugin_section, 'jobplugin')
	jobplugin_cls = load_jobplugin(jobplugin_name)
	return jobplugin_cls(jobplugin_name, self.config, jobplugin_section,
			     self.log, env)
