/*
 * Decompiled with CFR 0.152.
 */
package org.glite.security.util;

import java.io.File;
import java.io.IOException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.X509CRL;
import java.security.cert.X509CRLEntry;
import java.security.cert.X509Certificate;
import java.util.Date;
import java.util.Set;
import java.util.Vector;
import org.apache.log4j.Logger;
import org.bouncycastle.asn1.ASN1Object;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x509.IssuingDistributionPoint;
import org.bouncycastle.asn1.x509.X509Extensions;
import org.glite.security.util.CaseInsensitiveProperties;
import org.glite.security.util.CertificateRevokedException;
import org.glite.security.util.DN;
import org.glite.security.util.DNHandler;
import org.glite.security.util.FileCertReader;
import org.glite.security.util.RevocationChecker;

public class FileCRLChecker
extends RevocationChecker {
    private static final Logger LOGGER = Logger.getLogger(FileCRLChecker.class);
    private X509CRL m_crl = null;
    private X509Certificate m_caCert;
    private String m_caBaseFilename;
    private String m_crlFilename;
    private int m_caNumber;
    private long m_crlModified = -255L;
    private boolean m_crlRequired = true;
    private String failureString = null;
    static FileCertReader s_certReader;

    public FileCRLChecker(X509Certificate caCert, String caBaseFilename, int caNumber, CaseInsensitiveProperties props) {
        super(caCert, caBaseFilename, caNumber, props);
        this.m_caCert = caCert;
        this.m_caBaseFilename = caBaseFilename;
        this.m_caNumber = caNumber;
        this.m_crlFilename = this.m_caBaseFilename + ".r" + this.m_caNumber;
        String crlReqText = null;
        if (props != null) {
            crlReqText = props.getProperty("crlRequired");
        }
        if ((crlReqText = crlReqText != null ? crlReqText.trim().toLowerCase() : "true").startsWith("f") || crlReqText.startsWith("n")) {
            this.m_crlRequired = false;
        }
        this.checkUpdate();
    }

    private void checkCrl() throws CertificateException, IOException {
        if (this.m_crl == null) {
            throw new IllegalArgumentException();
        }
        Set<String> criticalOIDs = this.m_crl.getCriticalExtensionOIDs();
        if (criticalOIDs != null) {
            for (String oid : criticalOIDs) {
                this.checkExtension(oid);
            }
        }
        try {
            this.m_crl.verify(this.m_caCert.getPublicKey());
        }
        catch (SignatureException e) {
            LOGGER.debug((Object)("The crl " + this.m_crlFilename + " is not signed properly by CA " + DNHandler.getSubject(this.m_caCert) + "."));
            throw new CertificateException("The crl " + this.m_crlFilename + " is not signed properly by CA " + DNHandler.getSubject(this.m_caCert) + ".");
        }
        catch (Exception e) {
            LOGGER.debug((Object)("The verification of crl " + this.m_crlFilename + " failed: " + e.getMessage()));
            throw new CertificateException("The verification of crl " + this.m_crlFilename + " failed: " + e.getMessage());
        }
    }

    public void checkUpdate() {
        try {
            File crlFile = new File(this.m_crlFilename);
            long lastModified = crlFile.lastModified();
            if (lastModified != this.m_crlModified) {
                LOGGER.debug((Object)("CRL file changed, reloading it: " + crlFile.getName()));
                this.loadCRL(crlFile.getAbsolutePath());
                this.failureString = null;
            }
        }
        catch (CertificateException e) {
            if (this.m_crlRequired && this.failureString == null) {
                LOGGER.warn((Object)("CRL loading for CA file " + this.m_caBaseFilename + "." + this.m_caNumber + " failed, the certificates from the CA " + DNHandler.getSubject(this.m_caCert).getRFCDN() + " will be refused. Error was: " + e.getClass() + ": " + e.getMessage()));
            }
            this.failureString = e.getMessage();
        }
        catch (IOException e) {
            if (this.m_crlRequired && this.failureString == null) {
                LOGGER.warn((Object)("CRL loading for CA file " + this.m_caBaseFilename + "." + this.m_caNumber + " failed, the certificates from the CA " + DNHandler.getSubject(this.m_caCert).getRFCDN() + " will be refused. Error was: " + e.getClass() + ": " + e.getMessage()));
            }
            this.failureString = e.getMessage();
        }
    }

    private void loadCRL(String absolutePath) throws CertificateException, IOException {
        Vector certs = s_certReader.readCRLs(absolutePath);
        if (certs == null || certs.isEmpty()) {
            throw new IOException("No CRL found in " + absolutePath + ".");
        }
        this.m_crl = (X509CRL)certs.get(0);
        File file = new File(absolutePath);
        this.m_crlModified = file.lastModified();
        this.checkCrl();
    }

    public void check(X509Certificate cert) throws IOException, CertificateException, CertificateRevokedException {
        X509CRLEntry entry;
        Date thisUpdate;
        DN subDN = DNHandler.getSubject(cert);
        DN caDN = DNHandler.getIssuer(cert);
        if (this.failureString != null) {
            throw new CertificateException("CRL checking failed, CRL loading had failed: " + this.failureString);
        }
        Date now = new Date(System.currentTimeMillis());
        Date nextUpdate = this.m_crl.getNextUpdate();
        if (nextUpdate.before(now)) {
            if (this.m_crlRequired) {
                LOGGER.info((Object)("The certificate " + subDN + " is not in the CRL of " + caDN + ", but the CRL has expired on " + nextUpdate + ", so rejecting this certificate."));
                throw new CertificateRevokedException("The certificate " + subDN + " is not in the CRL of " + caDN + ", but the CRL has expired on " + nextUpdate + ", so rejecting this certificate.");
            }
            LOGGER.warn((Object)("The CRL of " + caDN + " has expired on " + nextUpdate + ", but as CRLs are not required, the cert is not rejected."));
        }
        if ((thisUpdate = this.m_crl.getThisUpdate()).after(now)) {
            if (this.m_crlRequired) {
                LOGGER.info((Object)("The certificate " + subDN + " is not in the CRL of " + caDN + ", but the CRL is not yet valid (valid from " + thisUpdate + "), so rejecting this certificate."));
                throw new CertificateRevokedException("The certificate " + subDN + " is not in the CRL of " + caDN + ", but the CRL is not yet valid (valid from " + thisUpdate + "), so rejecting this certificate.");
            }
            LOGGER.warn((Object)("The CRL of " + caDN + " is not yet valid (valid from " + thisUpdate + "), but as CRLs are not required, the cert is not rejected."));
        }
        if ((entry = this.m_crl.getRevokedCertificate(cert.getSerialNumber())) != null) {
            throw new CertificateRevokedException("The certificate " + subDN + " is revoked by CA " + caDN + " on " + entry.getRevocationDate() + ".");
        }
    }

    private void checkExtension(String oid) throws CertificateException, IOException {
        if (oid.equals(X509Extensions.DeltaCRLIndicator.getId())) {
            LOGGER.debug((Object)"Found DeltaCRLIndicator extension that can't be supported.");
            throw new CertificateException("Usupported critical extension in CRL: DeltaCRLIndicator");
        }
        if (oid.equals(X509Extensions.IssuingDistributionPoint.toString())) {
            this.checkIssuinDistributionPoint();
            return;
        }
        throw new CertificateException("Unrecognized critical extension in CRL: " + oid);
    }

    private void checkIssuinDistributionPoint() throws CertificateException, IOException {
        byte[] extensionBytes = this.m_crl.getExtensionValue(X509Extensions.IssuingDistributionPoint.toString());
        ASN1Object object = ASN1Object.fromByteArray((byte[])extensionBytes);
        if (!(object instanceof DEROctetString)) {
            throw new CertificateException("Invalid data in IssuingDistributionPoint extension, not DEROctetString");
        }
        DEROctetString string = (DEROctetString)object;
        if (!((object = ASN1Object.fromByteArray((byte[])string.getOctets())) instanceof ASN1Sequence)) {
            throw new CertificateException("Invalid data in IssuingDistributionPoint extension, not ASN1Sequence");
        }
        IssuingDistributionPoint issuingDistributionPoint = new IssuingDistributionPoint((ASN1Sequence)object);
        if (issuingDistributionPoint.onlyContainsAttributeCerts()) {
            throw new CertificateException("CRL only contains attribute certs, not useful for authentication.");
        }
        if (issuingDistributionPoint.getOnlySomeReasons() != null) {
            throw new CertificateException("CRL only contains some reasons of revocations, can't trust the certificates without other complementing CRL(s), which is not supported.");
        }
    }

    static {
        try {
            s_certReader = new FileCertReader();
        }
        catch (CertificateException e) {
            throw new RuntimeException("Security provider initialization failed: " + e.getMessage(), e);
        }
    }
}

