/**
 *  Copyright (c) Members of the EGEE Collaboration. 2004-2008.
 *  See http://www.eu-egee.org/partners/ for details on the copyright
 *  holders.
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *  GLite Data Management - Simple file fts API
 *  Authors: Gabor Gombas <Gabor.Gombas@cern.ch>
 *           Zoltan Farkas <Zoltan.Farkas@cern.ch>
 *           Paolo Tedesco <paolo.tedesco@cern.ch>
 *  Version info: $Id: fts-simple.h,v 1.21 2009/07/02 15:28:18 szamsu Exp $
 *  Release: $Name:  $
 */

#ifndef GLITE_DATA_TRANSFER_FTS_SIMPLE_H
#define GLITE_DATA_TRANSFER_FTS_SIMPLE_H

#ifdef __cplusplus
extern "C" {
#endif

#include <glite/data/transfer/c/transfer-simple.h>

/**********************************************************************
 * Constants
 */

/* SURL generation pattern types */
typedef enum
{
    GLITE_FTS_SURL_LFN,
    GLITE_FTS_SURL_DAY,
    GLITE_FTS_SURL_HOUR,
    GLITE_FTS_SURL_FLAT
} glite_fts_surl_pattern;

/* Default SURL generation pattern */
#define GLITE_FTS_SURL_DEFAULT      GLITE_FTS_SURL_LFN

/* Job parameter key for gridftp */
#define GLITE_FTS_PARAM_GRIDFTP     "gridftp"

/* Job parameter key for MyProxy */
#define GLITE_FTS_PARAM_MYPROXY     "myproxy"

/* Job parameter key for DelegationID */
#define GLITE_FTS_PARAM_DELEGATIONID    "delegationid"

/* Job parameter for storageClass */
#define GLITE_FTS_PARAM_SPACETOKEN   "spacetoken"
#define GLITE_FTS_PARAM_SPACETOKEN_SOURCE   "source_spacetoken"

#define GLITE_FTS_PARAM_COPY_PIN_LIFETIME "copy_pin_lifetime"
#define GLITE_FTS_PARAM_LAN_CONNECTION "lan_connection"
#define GLITE_FTS_PARAM_FAIL_NEARLINE "fail_nearline"

/* SRM 2.2 overwrite flag */
#define GLITE_FTS_PARAM_OVERWRITEFLAG   "overwrite"

#define GLITE_FTS_PARAM_CHECKSUM_METHOD "checksum_method"

/* Service Discovery parameter keys */
#define GLITE_FTS_SD_PATTERN        "surl_pattern"
#define GLITE_FTS_SD_SURL_PREFIX    "surl_prefix"
#define GLITE_FTS_SD_SRM_ENDPOINT   "srm_endpoint"

/* Service type for Service Discovery */
#define GLITE_FTS_SD_TYPE       "org.glite.FileTransfer"

/* Service type of Channel Agent for Service Discovery */
#define GLITE_CA_SD_TYPE        "org.glite.ChannelAgent"

/* Environment variable to override default service type */
#define GLITE_FTS_SD_ENV        "GLITE_SD_FTS_TYPE"


/**********************************************************************
 * General guidelines:
 * - Functions that return a pointer return NULL when there is an error.
 * - Functions that return 'int' return 0 when successful and -1 in case
 *   of an error.
 * - Any objects returned by a function is owned by the caller and has to
 *   be deallocated by the caller.
 */

/**********************************************************************
 * Function prototypes - library management functions
 */

/* Return the current FileTransfer endpoint used by the transfer context */
const char *glite_fts_get_endpoint(glite_transfer_ctx *ctx);

#if 0 /* These need more research */
char *glite_fts_generate_surl(glite_transfer_ctx *ctx, const char *src);

int glite_fts_set_srm_endpoint(glite_transfer_ctx *ctx, const char *uri);

int glite_fts_set_surl_prefix(glite_transfer_ctx *ctx, const char *prefix);

int glite_fts_set_surl_pattern(glite_transfer_ctx *ctx,
    glite_fts_surl_pattern pattern);
#endif 

/**********************************************************************
 * Function prototypes - org.glite.data.transfer.ServiceBase
 */

/**
 * @defgroup transfer_ServiceBase org.glite.data.transfer.ServiceBase
 * @{
 */

/**
 * Get the version of the remote endpoint.
 * @param ctx The transfer context.
 * @return The version string, or NULL in case of an error. 
 */
char *glite_fts_getVersion(glite_transfer_ctx *ctx);

/**
 * Get the schema version of the remote endpoint.
 * @param ctx The transfer context.
 * @return The schema version string, or NULL in case of an error. 
 */
char *glite_fts_getSchemaVersion(glite_transfer_ctx *ctx);

/**
 * Get the interface version of the remote endpoint.
 * @param ctx The transfer context.
 * @return The interface version string, or NULL in case of an error. 
 */
char *glite_fts_getInterfaceVersion(glite_transfer_ctx *ctx);

/**
 * Get the service metadata for a given key.
 * @param ctx The transfer context.
 * @param key The key of which the metadata we want to get. Must not be NULL.
 * @return The metadata string, or NULL in case of an error. 
 */
char *glite_fts_getServiceMetadata(glite_transfer_ctx *ctx, const char *key);

/**
 * @}
 */

/**********************************************************************
 * Function prototypes - org.glite.data.transfer.fts.FileTransfer
 */

/**
 * @defgroup FilePlacement org.glite.data.transfer.fts.FileTransfer
 * @{
 */

/**
 * Submit a FileTransfer job.
 * This funcion is for backward compatibility here, so clients with a version
 * number can use services with smaller version number. Only the minor and
 * patch versions can differ!
 * @param ctx The transfer context.
 * @param job The object containing all the informations related to the job.
 * @return The requestID, or NULL in case of an error.
 */
char *glite_fts_submit(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJob *job);

/**
 * Submit a FileTransfer job with optional checskum information.
 * @param ctx The transfer context.
 * @param job The object containing all the informations related to the job.
 * @param withDelegation If the job should use delegated credentials.
   @param withChecksums  If the job has checksum specified by the client.
 * @return The requestID, or NULL in case of an error.
 */
char *glite_fts_transferSubmit(glite_transfer_ctx *ctx,
    const glite_transfer_TransferJob *job, 
    int withDelegation, int withChecksums);

/**
 * Submit a FilePlacement job.
 * @param ctx The transfer context.
 * @param job The object containing all the informations related to the job.
 * @return The requestID, or NULL in case of an error.
 */
char *glite_fts_placementSubmit(glite_transfer_ctx *ctx,
    const glite_transfer_PlacementJob *job, int withDelegation);

/**
 * Get a list of jobIDs matching the given states on the given channel.
 * @param ctx The transfer context.
 * @param numStates The number of items in the states array.
 * @param states Array of states we are interested in.
 * @param channelName The name of the channel.
 * @param resultCount [OUT] The number of items in the result array.
 * @return An array of jobIDs.
 */
glite_transfer_JobStatus **glite_fts_listRequests(glite_transfer_ctx *ctx,
    int numStates, const char * const states[], const char *channelName,
    int *resultCount);

/**
 * Get a list of jobIDs matching the given states on the given channel.
 * @param ctx The transfer context.
 * @param numStates The number of items in the states array.
 * @param states Array of states we are interested in.
 * @param channelName The name of the channel.
 * @param restrict_dn Restrict to this DN
 * @param restrict_vo Restrict to this VO
 * @param resultCount [OUT] The number of items in the result array.
 * @return An array of jobIDs.
 */
glite_transfer_JobStatus **glite_fts_listRequests2(glite_transfer_ctx *ctx,
    int numStates, const char * const states[], const char *channelName,
   const char* restrict_dn, const char* restrict_vo,
    int *resultCount);

/**
 * Get the statuses of all the files with the given requestID.
 * @param ctx The transfer context.
 * @param requestId The ID of the request.
 * @param offset Number of results to skip.
 * @param limit Number of results to return.
 * @param resultCount Number of items in the result array.
 * @return Array of file transfer status objects, or NULL in case of an error.
 */
glite_transfer_FileTransferStatus **glite_fts_getFileStatus(glite_transfer_ctx *ctx,
    const char *requestId, int offset, int limit, int *resultCount);

/**
 * Get the summary state for the whole job. Possible answers: SUBMITTED, PENDING, ACTIVE, CANCELLING, CANCELLED, DONE, FINISHED, TRANSFERFAILED.
 * @param ctx The transfer context.
 * @param requestId The ID of the request.
 * @return The job status.
 */
glite_transfer_JobStatus *glite_fts_getTransferJobStatus(glite_transfer_ctx *ctx,
    const char *requestId);

/**
 * Get a summary status of the job. Including the job state and some summary statistics.
 * @param ctx The transfer context.
 * @param requestId The ID of the request.
 * @return An object containing the status informations.
 */
glite_transfer_TransferJobSummary *glite_fts_getTransferJobSummary(glite_transfer_ctx *ctx,
    const char *requestId);
glite_transfer_TransferJobSummary2 *glite_fts_getTransferJobSummary2(glite_transfer_ctx *ctx,
    const char *requestId);


/**
 * Attempt to cancel the given job on the service.
 * The only semantics defined are 'best effort':
 * - no roll back. Any file already transferred will be kept and not deleted;
 * - any file currently transferring will complete and will not be deleted.
 * - Any files pending will be cancelled and will not be transferred.
 * @param ctx The transfer context.
 * @param nitem Number of job IDs passed in requestIds
 * @param requestIds The IDs of the request.
 * @return 0 when successful and -1 in case of an error. 
 */
int glite_fts_cancel(glite_transfer_ctx *ctx, int nitems, const char **requestIds);

/**
 * Add a manager to the given VO.
 * @param ctx The transfer context.
 * @param VOname The name of the VO.
 * @param principal The principal to add.
 * @return 0 when successful and -1 in case of an error. 
 */
int glite_fts_addVOManager(glite_transfer_ctx *ctx, const char *VOname,
        const char *principal);

/**
 * Remove a manager from the given VO.
 * @param ctx The transfer context.
 * @param VOname The name of the VO.
 * @param principal The principal to add.
 * @return 0 when successful and -1 in case of an error. 
 */
int glite_fts_removeVOManager(glite_transfer_ctx *ctx, const char *VOname,
        const char *principal);

/**
 * Lists managers of a given VO.
 * @param ctx The transfer context.
 * @param VOname The name of the VO.
 * @param resultcount [OUT] Number of returned items.
 * @return Array of managers, or NULL in case of an error. 
 */
char **glite_fts_listVOManagers(glite_transfer_ctx *ctx, const char *VOname,
        int *resultcount);

/**
 * Get roles of the client on this service.
 * @param ctx The transfer context.
 * @return Roles object. 
 */
glite_transfer_Roles *glite_fts_getRoles(glite_transfer_ctx *ctx);

/**
 * Get roles of the specified user on this service.
 * @param ctx The transfer context.
 * @return Roles object.
 */
glite_transfer_Roles *glite_fts_getRolesOf(glite_transfer_ctx *ctx, const char *otherDN);

/**
 * Set priority of a job.
 * @param ctx The transfer context.
 * @param requestID ID of the transfer job.
 * @param priority The new priority of the job.
 * @return Roles object. 
 */
int glite_fts_setJobPriority(glite_transfer_ctx *ctx,
        const char *requestID, const int priority);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* GLITE_DATA_TRANSFER_FTS_SIMPLE_H */
