/*     Schema creation for AMGA 2.0
  Setup as follows:
  su postgres
  createuser -S -d arda
  createdb -O arda metadata
  createlang plpgsql metadata
  psql -Uarda metadata < createInitialPG.sql_2_0
*/

/* Main table */
CREATE TABLE masterindex (  
        "id" integer unique,
        "directory" varchar(1024) PRIMARY KEY,
        "table_name" varchar(64),
        "flags" smallint,
        "main_column" varchar(64),
        "master" varchar(64),
        "owner_name" varchar(64),
        "permissions" char(3),
        "acls" varchar(2048),
        "rep_groups" varchar(2048),
        "rep_xid" bigint,
        "info" varchar(2048)
);

INSERT INTO masterindex ("id", "directory", "table_name", "flags", "master", "owner_name", "permissions", "acls", "rep_groups" )
	         VALUES (1, '/', '', 4096, '', 'root', 'rwx', '|system:anyuser rx|', '');

/* User management */
CREATE TABLE groups (
    "name" varchar(64) PRIMARY KEY,
    "owner" varchar(64),
    "members" varchar(1024)
);

CREATE TABLE users (
    "name" varchar(64) PRIMARY KEY,
    "password" varchar(64),
    "capabilities" VARCHAR(256)
);

CREATE TABLE certificates (
    "subject" varchar(256),
    "name" varchar(64) REFERENCES users ON DELETE CASCADE,
    PRIMARY KEY ("subject", "name")
);

CREATE TABLE proxy (
    "directory" varchar(1024),
    "mnt_master" varchar(64) ,
    PRIMARY KEY ("directory", "mnt_master")
);

CREATE TABLE voms_user (
    "subject" varchar(256),
    "name" varchar(64) ,
    PRIMARY KEY ("subject")
);

CREATE TABLE voms_group (
    "subject" varchar(256),
    "name" varchar(64) ,
    PRIMARY KEY ("subject")
);


/* Tables for a file catalogue */
CREATE TABLE guids (
      "guid" bytea,
      "si" bytea,
      "reference" char(32),
      "req_delete" bytea,
      "req_migrate" bytea,
      "table_name" char(65),
      PRIMARY KEY ("guid")
);

CREATE OR REPLACE FUNCTION newguid() RETURNS TRIGGER AS '
DECLARE
    temp varchar;
BEGIN
--    RAISE NOTICE ''GUID INSERT TRIGGER CALLED'';
    IF NOT NEW.guid IS NULL THEN
        temp = TG_RELNAME || TG_ARGV[0];
        IF TG_ARGV[0] = ''_s'' THEN
            temp = temp || NEW.dir;
        END IF;
        INSERT INTO guids ("guid", "si", "reference") VALUES (NEW.guid, '''', temp);
    END IF;
    RETURN NULL;
END;
' LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION delguid() RETURNS TRIGGER AS '
BEGIN
    DELETE FROM guids WHERE "guid"=OLD.guid;
    RETURN NULL;
END;
' LANGUAGE plpgsql;

CREATE TABLE replicas (
  "guid" bytea REFERENCES guids ON DELETE CASCADE,
  "last_access" TIMESTAMP WITHOUT TIME ZONE,
  "expires" TIMESTAMP WITHOUT TIME ZONE,
  "surl" VARCHAR(256)
);

CREATE TABLE sites (
  "id" INTEGER,
  "name" VARCHAR(64) UNIQUE NOT NULL,
  "hostname" VARCHAR(64) NOT NULL,
  "port" INTEGER NOT NULL,
  "login" VARCHAR(64) NOT NULL DEFAULT '',
  "password" VARCHAR(32),
  "use_ssl" SMALLINT NOT NULL DEFAULT 0,
  "authenticate_with_certificate" SMALLINT NOT NULL DEFAULT 0,
  "cert_file" VARCHAR(128),
  "key_file" VARCHAR(128),
  "use_grid_proxy" SMALLINT NOT NULL DEFAULT 0,
  "verify_server_cert" SMALLINT NOT NULL DEFAULT 0,
  "trusted_cert_dir" VARCHAR(128),
  "require_data_encryption" SMALLINT NOT NULL DEFAULT 0,
  "cert_password" VARCHAR(32),
  PRIMARY KEY("id")
);

/* Replication: tables used as a master */

/* The log_cwd field is null for transactions */
CREATE TABLE logs (
	"log_xid" BIGSERIAL PRIMARY KEY,
	"log_directory" varchar(256) NOT NULL,
	"log_user" varchar(64) NOT NULL,
	"log_cwd" varchar(256),
	"log_permissions"  char(3) NOT NULL,
	"log_group_rights" char(3) NOT NULL,
	"log_command" TEXT NOT NULL
);

/* The tr_id column gives logs a serial id so they can be ordered */
CREATE TABLE transactions (
    "tr_id"             BIGSERIAL,
    "tr_transaction_id" BIGINT REFERENCES logs("log_xid") ON DELETE CASCADE,
    "tr_log_directory"  VARCHAR(256),
    "tr_log_cwd"        VARCHAR(256),
    "tr_log_command"    TEXT NOT NULL
);

CREATE INDEX transactions_tr_transaction_id_idx
    ON transactions(tr_transaction_id);

CREATE TABLE subscribers (
	"sub_id" VARCHAR(64) PRIMARY KEY,
	"sub_rep_users" smallint NOT NULL,
	"sub_lease_duration" TIMESTAMP,
	"sub_cur_xid" BIGINT
);

CREATE TABLE subscriptions (
	"sub_subscriber" VARCHAR(64) REFERENCES subscribers("sub_id") ON DELETE CASCADE,
	"sub_directory" VARCHAR(256),
    "sub_rep_permissions" smallint NOT NULL,
	PRIMARY KEY ("sub_subscriber", "sub_directory")
);

/* Replication: tables used as a slave */
CREATE TABLE masters (
    "mst_id" VARCHAR(64) PRIMARY KEY,
    "mst_active" SMALLINT NOT NULL,
    "mst_last_xid" BIGINT
);

CREATE TABLE mounts (
    "mnt_master" VARCHAR(64) REFERENCES masters("mst_id") ON DELETE CASCADE,
    "mnt_directory" VARCHAR(256) PRIMARY KEY,
    "mnt_state" VARCHAR(16) NOT NULL,
    "mnt_rep_permissions" smallint NOT NULL
);

CREATE TABLE constraints (
  "table_id"   integer references masterindex ("id") ON DELETE CASCADE,
  "name"       varchar(64),
  "type"       char(1),
  "column"     varchar(64),
  "constraint" varchar(1024),
  PRIMARY KEY ("table_id", "name")
);

/* For generating unique numbers global to this AMGA installation.*/
CREATE SEQUENCE sequencer CYCLE;


/* Table for holding system information
   - Log tuples is used for counting the number of tuples in the log table 
   - rep_users masters and xid keep track of whether the local node is replicating
   user and group information from some other node.
   */
CREATE TABLE amga (
	"amga_version" varchar(8) NOT NULL,
	"amga_node_id" varchar(64),
	"log_table_tuples" integer,
	"rep_users_master" VARCHAR(64) REFERENCES masters("mst_id") ON DELETE SET NULL,
	"rep_users_xid" bigint
);

INSERT INTO amga ("amga_version", "log_table_tuples", "rep_users_master", "rep_users_xid")
    VALUES ('2.4.0', 0, NULL, NULL);

/* Trigger to update the row count of the logs table */
CREATE OR REPLACE FUNCTION logs_count_trig()
RETURNS TRIGGER AS
'
    DECLARE
    BEGIN
    IF TG_OP = ''INSERT'' THEN
        EXECUTE ''UPDATE amga SET log_table_tuples = log_table_tuples+1'';
        RETURN NEW;
    ELSIF TG_OP = ''DELETE'' THEN
        EXECUTE ''UPDATE amga SET log_table_tuples = log_table_tuples-1'';
        RETURN OLD;
    END IF;
    END;
'
LANGUAGE 'plpgsql';

CREATE TRIGGER logs_count
    BEFORE INSERT OR DELETE
    ON logs FOR EACH ROW EXECUTE PROCEDURE logs_count_trig();

--
-- instr functions that mimic Oracle's counterpart
-- Syntax: instr(string1, string2, [n], [m]) where [] denotes optional parameters.
--
-- Searches string1 beginning at the nth character for the mth occurrence
-- of string2.  If n is negative, search backwards.  If m is not passed,
-- assume 1 (search starts at first character).
--

CREATE FUNCTION instr(varchar, varchar) RETURNS integer AS '
DECLARE
    pos integer;
BEGIN
    pos:= instr($1, $2, 1);
    RETURN pos;
END;
' LANGUAGE plpgsql;


CREATE FUNCTION instr(varchar, varchar, integer) RETURNS integer AS '
DECLARE
    string ALIAS FOR $1;
    string_to_search ALIAS FOR $2;
    beg_index ALIAS FOR $3;
    pos integer NOT NULL DEFAULT 0;
    temp_str varchar;
    beg integer;
    length integer;
    ss_length integer;
BEGIN
    IF beg_index > 0 THEN
        temp_str := substring(string FROM beg_index);
        pos := position(string_to_search IN temp_str);

        IF pos = 0 THEN
            RETURN 0;
        ELSE
            RETURN pos + beg_index - 1;
        END IF;
    ELSE
        ss_length := char_length(string_to_search);
        length := char_length(string);
        beg := length + beg_index - ss_length + 2;

        WHILE beg > 0 LOOP
            temp_str := substring(string FROM beg FOR ss_length);
            pos := position(string_to_search IN temp_str);

            IF pos > 0 THEN
                RETURN beg;
            END IF;

            beg := beg - 1;
        END LOOP;

        RETURN 0;
    END IF;
END;
' LANGUAGE plpgsql;



CREATE FUNCTION instr(varchar, varchar, integer, integer) RETURNS integer AS '
DECLARE
    string ALIAS FOR $1;
    string_to_search ALIAS FOR $2;
    beg_index ALIAS FOR $3;
    occur_index ALIAS FOR $4;
    pos integer NOT NULL DEFAULT 0;
    occur_number integer NOT NULL DEFAULT 0;
    temp_str varchar;
    beg integer;
    i integer;
    length integer;
    ss_length integer;
BEGIN
    IF beg_index > 0 THEN
        beg := beg_index;
        temp_str := substring(string FROM beg_index);

        FOR i IN 1..occur_index LOOP
            pos := position(string_to_search IN temp_str);

            IF i = 1 THEN
                beg := beg + pos - 1;
            ELSE
                beg := beg + pos;
            END IF;

            temp_str := substring(string FROM beg + 1);
        END LOOP;

        IF pos = 0 THEN
            RETURN 0;
        ELSE
            RETURN beg;
        END IF;
    ELSE
        ss_length := char_length(string_to_search);
        length := char_length(string);
        beg := length + beg_index - ss_length + 2;

        WHILE beg > 0 LOOP
            temp_str := substring(string FROM beg FOR ss_length);
            pos := position(string_to_search IN temp_str);

            IF pos > 0 THEN
                occur_number := occur_number + 1;

                IF occur_number = occur_index THEN
                    RETURN beg;
                END IF;
            END IF;

            beg := beg - 1;
        END LOOP;

        RETURN 0;
    END IF;
END;
' LANGUAGE plpgsql;

--
-- Tests whether an entry is accessible according to its ACLs
--
CREATE OR REPLACE FUNCTION acl_test(varchar, varchar, varchar) RETURNS integer AS '
DECLARE
    acls ALIAS FOR $1;
    groups ALIAS FOR $2;
    right ALIAS FOR $3;
    group varchar;
    pos1 integer;
    pos2 integer;
    groupstart integer;
BEGIN
    IF acls IS NULL THEN
	RETURN NULL;
    END IF;

--    RAISE NOTICE ''ACLS are >%<'', acls;
    pos1 := 1;
    LOOP
        group := NULL;

        pos2 := instr(groups,'' '', pos1 + 1);
        IF pos2 = 0 THEN
	    pos2 = LENGTH(groups) + 1;
        END IF;

	EXIT WHEN pos1 = pos2;

        group := substr(groups, pos1, pos2 - pos1);
--	RAISE NOTICE ''Group is >%<'', group;

	groupstart := instr(acls, ''|'' || group || '' '');
	IF groupstart != 0 THEN
	    groupstart := groupstart + LENGTH(group) + 2;
	    LOOP
--		RAISE NOTICE ''CHAR >%<'', substr(acls, groupstart, 1);
		EXIT WHEN substr(acls, groupstart, 1) = ''|'';
		IF substr(acls, groupstart, 1) = right THEN
		    RETURN 1;
		END IF;
		groupstart = groupstart + 1;
	    END LOOP;
	END IF;

        pos1 := pos2 + 1;
	EXIT WHEN pos1 >= LENGTH(groups);

    END LOOP;
    RETURN NULL;
END;
' LANGUAGE plpgsql;
