/*
  Copyright (c) Members of the EGEE Collaboration. 2004. 
  See http://www.eu-egee.org/partners/ for details on the copyright
  holders.  

  Licensed under the Apache License, Version 2.0 (the "License"); 
  you may not use this file except in compliance with the License. 
  You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0 

  Unless required by applicable law or agreed to in writing, software 
  distributed under the License is distributed on an "AS IS" BASIS, 
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
  See the License for the specific language governing permissions and 
  limitations under the License.
*/


// -*- Mode:c++ -*-
// $Id: MDClient.h,v 1.16 2011/06/08 01:39:04 asunil Exp $
/** MDClient is a class which provides client functionality for an
    application which wants to run as a client to the metatada
    server. MDClient controls an IP connection to a server providing a
    socket for metadata queries and responses
**/

#ifndef MDCLIENT_H
#define MDCLIENT_H

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <string>
#include <list>
#include <stdint.h>

#include "AMGAService.h"

#define MD_DEFAULT_PORT 8822

class MDClient;
class ConfigParser;
class TCPSocket;
class MDFed;

int serverCertVerifyCallback(int ok, X509_STORE_CTX *store);

void int_sig_handler(int sig);
MDClient *registerClient(MDClient * client);

class MDClient {
  friend void int_sig_handler(int);
  friend MDClient *registerClient(MDClient *);
  friend void pthreads_locking_callback(int, int, const char *, int);
 public:
  MDClient(const std::string &host="", int port=0,
	   bool persistent = false,
	   const std::string &configFile = "", bool dbg=false);

  ~MDClient();

  /// Sets server for next connectToServer() call
  void useServer(const std::string &ipAddr, int port=0);

  /// Picks next configured server from list of BDII and configuration file
  int nextServer();

  // Connects to the previously set server
  int connectToServer();

  /// Returns true if the MDClient holds a TCP connection an AMGA server
  bool isConnected() const { return connected; };

  int disconnect(bool saveSession = false);

  int setTimeout(int seconds);

  int execute(const std::string &command);

  int execNoWait(const std::string &command);

  int fetchRow(std::string &row, bool readAhead=true);

  std::string getError() { std::string err;
                           error.swap(err); return err; };

  bool eot();

  /// Returns the greeting string the server sent on connecting
  const std::string &getGreeting();

  /// Sets the send-buffer when streaming data to the server
  void setSendBuffer(unsigned int s) { sendBufferMax=s; };

  /// Sets the session number to allow several sessions for a user
  void setSessionNumber(int n) { sessionNumber = n; }

  /// Gets the session number of the user
  int getSessionNumber() { return sessionNumber; }

  int cancel();

  /// Returns the name of the server the client is connected to
  /// (in this session)
  const std::string &serverName() const {
    return sessionHost;
  }

  /// Returns port number of the server the client is connected to
  /// (in this session)
  int serverPort() const {
    return sessionPort;
  }

  std::string currentServiceName() const {
    if( currentService != services.end() )
      return currentService->dn;
    return "";
  }

  bool getDebug(){ return debug; };
  void setDebug(bool d){ debug = d; };

  static char certKey[256];

  bool endOfTransmission;
  int resultCols;
  int fetchData();
  void setError(std::string err);

  // federation
  MDFed* mdfed;

 private:
  bool debug;

  class MDClientLock {
    friend void pthreads_locking_callback(int, int, const char *, int);
  public:
    MDClientLock();
    ~MDClientLock();
  private:
    pthread_mutex_t *locks;
  };
  void sslErrors();

  int getBDIIInfo();
  int doServiceDiscovery();
  int getConfiguredServer();

  int prepareExec();
  int setServer(const std::string &host="", int port=0);
  int retrieveResult();

  int tentativeRead();
  int doRead();

  int establishContext();
  int initConnection(bool &resumeSession);
  int initSSLConnection(bool resumeSession);
  int sendLine(const std::string &c, bool flush=true);
  int initSSL();
  int initSSLSession();
  int readConfiguration(std::list<std::string> &locations);
  int loadCertificate();

  int saveSessionData();
  int restoreSessionData();
  

  static MDClientLock sslLock;

  std::string response;
  int timeout;
  TCPSocket *sock;   // Client socket
  SSL_CTX *ctx;    // SSL context
  bool useSSL;
  bool requireSSL;

  std::string error;
  std::string greeting;           // The greeting string of the server
  unsigned int protocolVersion;   // The protocol version of the server

  bool connected;
  ConfigParser *config;

  int usedPort;

  std::string sendBuffer;
  unsigned int sendBufferMax;
  volatile bool cancellationRequested;
  static MDClient *clientRegisteredForInt;
  bool wasCancelled;

  SSL_SESSION *sslSessionData;
  uint64_t sessionID;
  std::string sessionHost;
  int sessionPort;
  int sessionFile;
  std::string sessionFileName;
  bool persistentSessions;
  int sessionNumber;
  bool verifyServerCert;

  std::list< AMGA::Service > services;
  std::list< AMGA::Service >::const_iterator currentService;



};

#endif
