/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#include <dlfcn.h>
#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "eef_plugindl_s.h"
#include "plugin_manager.h"
#include "eval_manager.h"
#include "eef/eef_library.h"
#include "eef/eef_log.h"
#include "eef/eef_plugin_typedefs.h"


#define MAX_ERR_BUF	256


/************************************************************************
 * Global variables
 ************************************************************************/

static pthread_key_t           _plugin_key;
static pthread_key_t           _plugin_name_key;
static pthread_once_t  	       _plugin_key_once = PTHREAD_ONCE_INIT;

static const char *            _global_running_plugin_name;
static eef_plugindl_t *        _global_running_plugin;
static int                     _is_threading;

static struct eef_plugindl_s * _plugin_list;


/************************************************************************
 * Private prototypes
 ************************************************************************/

/* Creates thread-local storage key */
static void plugin_manager_make_key(void);

/* Internal method to keep track of which plugin is currently executing */
static void set_running_plugin(eef_plugindl_t * plugin);
/* Sets the name of the currently running plugin */
static void set_running_plugin_name(const char* plugin_name);

/* Loads and initializes (plugin_initialize) given plugin */
static EES_RC initialize_plugin(eef_plugindl_t* plugin);
/* Runs (plugin_run) given plugin */
static EES_PL_RC run_plugin(eef_plugindl_t* plugin);
/* Terminates (plugin_term) given plugin */
static EES_PL_RC term_plugin(char* plugin_name);

/* Create new plugin structure to add to the list */
static eef_plugindl_t * create_plugin_struct(char** argv, const char* name);
/* Wrapper for dlsym with error logging */
static eef_proc_t get_procsymbol(void* handle, char* symname);

/* Run specified policy */
static EES_RC run_policy(policy_t *policy);

/* Cleans up list of plugins, incl. dlclose */
static EES_RC clean_plugin_list(eef_plugindl_t * list);


/************************************************************************
 * Public functions
 ************************************************************************/

/**
 * Starts the plugin manager (only initializes values to NULL)
 */
EES_RC start_plugin_manager(){
    _plugin_list = NULL;
    _global_running_plugin = NULL;
    if (pthread_once(&_plugin_key_once, plugin_manager_make_key) != 0)
	return EES_FAILURE;
    
    return EES_SUCCESS;
}

/**
 * Stops the plugin manager: terminating plugins and calling clean_plugin_list
 */
EES_RC stop_plugin_manager(){
    /* Call term_plugins() */
    EEF_log(LOG_DEBUG, "%s: Terminating plugins...\n", __func__);
    if (term_plugins() != EES_SUCCESS )	{
	EEF_log(LOG_ERR, "%s: term_plugins() failed\n", __func__);
	return EES_FAILURE;
    }
    /* Call clean_plugin_list() */
    EEF_log(LOG_DEBUG, "%s: Cleaning plugins...\n", __func__);
    if( clean_plugin_list(_plugin_list) != EES_SUCCESS )    {
	EEF_log(LOG_ERR, "%s: clean_plugin_list() failed\n", __func__);
        return EES_FAILURE;
    }
    return EES_SUCCESS;
}

/**
 * sets threading flag
 */
void plugin_manager_start_threading(void){
    _is_threading = 1;
}

/**
 * Calls create_plugin_struct() to create a plugin struct and adds it to the
 * plugin_list. Name is strdupped, argv is copied.
 */
EES_RC add_plugin_struct(char* argv[], const char* name){
    eef_plugindl_t *last_plugin = NULL, *current_plugin = NULL;

    /* create_plugin_struct returns NULL if plugin failed to load. */
    if((current_plugin = create_plugin_struct(argv, name)) == NULL)
	return EES_FAILURE;

    /* append plugin to the end of the list */
    if(_plugin_list == NULL)
	_plugin_list = current_plugin;
    else {
	last_plugin = _plugin_list;
	/* TODO don't iterate list */
	while(last_plugin->next != NULL)
	    last_plugin = last_plugin->next;
	last_plugin->next = current_plugin;
    }
    return EES_SUCCESS;
}

/**
 * Initializes all the plugins
 * Returns EES_SUCCESS or EES_FAILURE if a plugin failed to load.
 */ 
EES_RC initialize_plugins(){
    eef_plugindl_t* node = _plugin_list;

    EEF_log(LOG_DEBUG, "Initializing plugins\n");
    while(node != NULL){
        set_running_plugin_name(node->name);
        if(initialize_plugin(node) == EES_FAILURE)
            return EES_FAILURE;
        node = node->next;
    }
    return EES_SUCCESS;
}

/**
 * Runs all the plugins
 * Returns EES_SUCCESS if a plugin branch executed successfully. Returns
 * EES_FAILURE if all plugin branches were exhausted unsuccessfully
 */ 
EES_RC run_plugins(char *policy_name[]){
    policy_t *policy, *first_policy=get_policies();
    int i;

    /* If we specified a array of policy names, use it */
    if (policy_name)	{
	for (i=0; policy_name[i]; i++)	{
	    /* Find named policy */
	    policy=first_policy;
	    while (policy && strcmp(policy->name, policy_name[i])!=0)
		policy = policy->next;
	    /* Check it exists */
	    if (!policy)    {
		EEF_log(LOG_ERR, "%s: specified policy %s does not exist\n",
			__func__, policy_name[i]);
		continue;
	    }
	    /* Try policy */
	    if (run_policy(policy)==EES_SUCCESS)
		return EES_SUCCESS;
	}
    } else {
	/* No name(s) specified, try all */
	policy=first_policy;
	while (policy){
	    if (run_policy(policy)==EES_SUCCESS)
		return EES_SUCCESS;
	    policy = policy->next;
	}
    }

    return EES_FAILURE;
}

/**
 * Terminates all the plugins
 * Returns EES_SUCCESS if all plugins terminated successfully.
 */
EES_RC term_plugins(void){
    eef_plugindl_t *node = _plugin_list;
    EES_RC retval = EES_SUCCESS;

    while(node != NULL){
        if(term_plugin(node->name) == EES_PL_FAILURE){
            retval = EES_FAILURE;
        }
        node = node->next;
    }
    return retval;
}

/**
 * Returns a plugin from the list by name, or NULL if not found
 */
eef_plugindl_t* get_plugin(const char* plugin_name){
    eef_plugindl_t * current_plugin = _plugin_list;

    /* iterate plugin_list  until plugin is found */
    while(current_plugin != NULL){
	if(strcmp(current_plugin->name, plugin_name) == 0)
	    return current_plugin;
	current_plugin = current_plugin->next;
    }
    return NULL;
}

/**
 * Keep track of which plugin is currently executing (used by AOS to determine
 * ownership of data)
 */
eef_plugindl_t * get_running_plugin(){
    return _global_running_plugin;
}

/**
 * returns name of currently running plugin
 */
const char* get_running_plugin_name(void){
    const char* name = NULL;

    if(!_is_threading){
        return _global_running_plugin_name;
    }

    /*  Try to get TLS */
    name = pthread_getspecific(_plugin_name_key);
    if (name == NULL)
        pthread_setspecific(_plugin_name_key, name);

    return name;
}


/************************************************************************
 * Private functions
 ************************************************************************/

/**
 * Creates thread-local storage key
 */
static void plugin_manager_make_key(void){
    pthread_key_create(&_plugin_key, NULL);
    pthread_key_create(&_plugin_name_key, NULL);
}

/**
 * Internal method to keep track of which plugin is currently executing
 */
static void set_running_plugin(eef_plugindl_t * plugin){
    if(!_is_threading)
        _global_running_plugin = plugin;
    else
        pthread_setspecific(_plugin_key, plugin);
}

/**
 * Sets the name of the currently running plugin
 */
static void set_running_plugin_name(const char* plugin_name){
    if(!_is_threading)
        _global_running_plugin_name = plugin_name;
    else
        pthread_setspecific(_plugin_name_key, plugin_name);
}

/**
 * Prepares the supplied plugin by name, initializes the struct with these
 * values. argv is copied, not duplicated, name is duplicated.
 * \post the plugin node is ready to be passed to the initialize_plugin function
 */
static eef_plugindl_t * create_plugin_struct(char** argv, const char* name){
    eef_plugindl_t * current_plugin = NULL;
    int argc = 0;

    /* basic input parsing */
    if (argv==NULL || name==NULL)
	return NULL;

    /* allocate memory */
    if( !(current_plugin = calloc(1,sizeof(eef_plugindl_t))) )	{
	EEF_log(LOG_ERR,"%s: out of memory\n", __func__);
	free(current_plugin);
	return NULL;
    }

    /* Copy in basic stuff */
    current_plugin->init_argv = argv;
    strncpy(current_plugin->name, name, FILENAME_MAX);
    current_plugin->next = NULL;
   
    /* Log input and count arguments */
    EEF_log(LOG_DEBUG, "Plugin definition for: %s\n", current_plugin->name);
    for(argc = 0; argv[argc]; argc++)
	EEF_log(LOG_DEBUG, "    argv[%d] = %s\n",
		argc, argv[argc]);
    current_plugin->init_argc = argc;

    /* return plugin struct */
    return current_plugin;
}

/**
 * Wrapper for dlsym with error logging. Adapted from lcmaps_pluginmanager.c
 * get_procsymbol
 */
static eef_proc_t get_procsymbol(void* handle, char* symname){
    eef_proc_t symhandle;
    char* errstring;

    /* dlsym returns void pointer (data pointer) whereas we need to cast to a
     * function pointer.
     * This is a known problem. See:
     * http://www.opengroup.org/onlinepubs/009695399/functions/dlsym.html#tag_03_112_08
     * Although gcc will produce a warning this seems to work on most platforms
     */
    dlerror();
    symhandle = dlsym(handle,symname);
    if ( (errstring = dlerror()) != NULL ){
	EEF_log(LOG_ERR, "%s: dlsym error %s\n", __func__, errstring);
	return NULL;
    }

    return symhandle;
}

/**
 * Initializes the supplied plugin node, which already holds the plugin name and
 * arguments
 * \post the plugin node is initialized and its functions are linked and
 * accessible
 */
static EES_RC initialize_plugin(eef_plugindl_t* plugin) {
    plugin_init_t *init_fnc_ptr;
    const char *local_module_dir;
    char *abs_plugin_path;
    char *short_name=plugin->init_argv[0];
    size_t len;
    char errbuf[MAX_ERR_BUF];
    EES_RC retval=EES_SUCCESS;
    int sn_rc;
    char* errstring;

    EEF_log(LOG_DEBUG, "Initializing plugin: %s\n", plugin->name);

    /* Check if we specified an absolute or relative path */
    if (short_name[0]!='/')	{
	/* Get plugin dir from the framework */
	local_module_dir = EEF_get_modules_path();
	if (local_module_dir==NULL) /* Use default */
	    local_module_dir=EES_MOD_HOME;

	/* Malloc sufficient memory for both versions, we might alloc 1 byte too
	 * many, for the sake of simplicity. */
	len=strlen(local_module_dir)+1+strlen(short_name)+1;
	/* Malloc memory */
	if ( (abs_plugin_path=malloc(len))==NULL )	{
	    EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return EES_FAILURE;
	}
	/* Create full path: len must fit */
	if (local_module_dir[strlen(local_module_dir)-1]!='/')
	    /* Need to add / in the middle */
	    sn_rc=snprintf(abs_plugin_path, len, "%s/%s",
			   local_module_dir, short_name);
	else
	    /* Can just concatenate */
	    sn_rc=snprintf(abs_plugin_path, len, "%s%s",
			   local_module_dir, short_name);
	/* Check return value, only <0: too long is not possible since we
	 * calculated it beforehand */
	if (sn_rc<0)    {
	    EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	    EEF_log(LOG_ERR,
		    "%s: Error prefixing plugin name with module dir: %s\n",
		    __func__,errbuf);
	    return EES_FAILURE;
	}
    } else
	/* plugin name is already absolute */
	abs_plugin_path=short_name;

    EEF_log(LOG_DEBUG, "Loading: %s\n", abs_plugin_path);

    /* load plugin handle */
    dlerror();	/* clear any pre-existing error */
    plugin->handle = dlopen(abs_plugin_path, RTLD_LAZY);
    if( (errstring = dlerror()) != NULL ){
        EEF_log(LOG_ERR, "%s: Failed to acquire handle on plugin: %s\n",
		__func__, abs_plugin_path);
        EEF_log(LOG_ERR, "%s\n", errstring);
        retval=EES_FAILURE;
	goto initialize_plugin_cleanup;
    }

    EEF_log(LOG_DEBUG, "Getting symbols from: %s\n", abs_plugin_path);

    plugin->procs[INITPROC] = NULL;
    plugin->procs[RUNPROC] = NULL;
    plugin->procs[TERMPROC] = NULL;

    /* link function pointers */
    plugin->procs[INITPROC]=get_procsymbol(plugin->handle,"plugin_initialize");
    if(plugin->procs[INITPROC] == NULL){
        EEF_log(LOG_ERR,"%s: plugin %s missing plugin_initialize()\n",
		__func__, plugin->name);
        retval=EES_FAILURE;
	goto initialize_plugin_cleanup;
    }

    plugin->procs[RUNPROC] = get_procsymbol(plugin->handle, "plugin_run");
    if(plugin->procs[RUNPROC] == NULL){
        EEF_log(LOG_ERR, "%s: plugin %s missing plugin_run()\n",
		__func__, plugin->name);
        retval=EES_FAILURE;
	goto initialize_plugin_cleanup;
    }

    plugin->procs[TERMPROC] = get_procsymbol(plugin->handle, "plugin_terminate");
    if(plugin->procs[TERMPROC] == NULL){
        EEF_log(LOG_ERR, "%s: plugin %s missing plugin_terminate()\n",
		__func__, plugin->name);
        retval=EES_FAILURE;
	goto initialize_plugin_cleanup;
    }

    set_running_plugin(plugin);
    EEF_log(LOG_DEBUG, "Running plugin_initialize() for plugin: %s\n",
	    plugin->name);
    init_fnc_ptr = (plugin_init_t*)plugin->procs[INITPROC];

    /* call function pointer */
    if(init_fnc_ptr(plugin->init_argc, plugin->init_argv) != EES_PL_SUCCESS){
        EEF_log(LOG_ERR, "%s: Failed to initialize plugin %s\n",
		__func__, abs_plugin_path);
	retval=EES_FAILURE;
    }

initialize_plugin_cleanup:
    /* Free memory when needed */
    if (abs_plugin_path!=short_name)
	free(abs_plugin_path);

    return retval;
}

/**
 * Calls the plugin's run method
 */
static EES_PL_RC run_plugin(eef_plugindl_t* plugin){
    EES_PL_RC rc;
    plugin_run_t *run_fnc_ptr;

    run_fnc_ptr = (plugin_run_t*) plugin->procs[RUNPROC];
    EEF_log(LOG_DEBUG, "Run method for %s linked at %p\n",
	    plugin->init_argv[0], (void*)run_fnc_ptr);
    EEF_log(LOG_DEBUG, "Running plugin %s\n", plugin->init_argv[0]);
    set_running_plugin(plugin);
    rc=run_fnc_ptr();
    EEF_log(LOG_DEBUG, "Plugin %s %s\n",
	    plugin->init_argv[0], rc==EES_PL_SUCCESS ? "succeeded" : "failed");
    return rc;
}

/**
 * Calls the plugin's term method
 */
static EES_PL_RC term_plugin(char* plugin_name){
    plugin_term_t *term_fnc_ptr;
    eef_plugindl_t * current_plugin;

    if((current_plugin = get_plugin(plugin_name))){
        if((term_fnc_ptr = (plugin_term_t*)current_plugin->procs[TERMPROC])){
            EEF_log(LOG_DEBUG, "Calling term for plugin: %s (address %p)\n",
		    plugin_name, (void*)term_fnc_ptr);
            set_running_plugin(current_plugin);
            if(term_fnc_ptr() != EES_PL_SUCCESS)
                return EES_PL_FAILURE;
        }
    }
    return EES_PL_SUCCESS;
}

/**
 * Cleans the list of plugins, unlinking all the dlopened modules and freeing
 * the structs
 * \post The plugin list is empty
 */
static EES_RC clean_plugin_list(eef_plugindl_t * list){
    eef_plugindl_t * plugin_entry   = list;
    eef_plugindl_t * plugin_next    = NULL;
    EES_RC           retval         = EES_SUCCESS;
    int              i;

    /* iterate plugin_list */
    while(plugin_entry!=NULL){
	EEF_log(LOG_DEBUG, "Cleaning plugin: %s\n", plugin_entry->name);
        if(plugin_entry->handle != NULL){
	    /* Valgrind can't trace modules past their dlclose-ing. See:
	     * http://valgrind.org/docs/manual/faq.html#faq.unhelpful */
            if(!getenv("VALGRIND")){
		dlerror();
                if((dlclose(plugin_entry->handle))){
                    EEF_log(LOG_ERR,
			    "dlclose error %s while cleaning up plugin list\n",
			    dlerror());
                    retval = EES_FAILURE;
                }
            } else {
                EEF_log(LOG_DEBUG,
			"%s: Running in valgrind, not dlclose'ing plugins\n",
			__func__);
            }
        }

        /* free argv array */
        for(i = 0; plugin_entry->init_argv[i]; i++)
            free(plugin_entry->init_argv[i]);
	free(plugin_entry->init_argv);

        /* free current struct and move to next plugin */
        plugin_next = plugin_entry->next;
        free(plugin_entry);
        plugin_entry = plugin_next;
    }
    return retval;
}

/**
 * Run specified policy
 */
static EES_RC run_policy(policy_t *policy) {
    rule_t* temp_rule;

    EEF_log(LOG_INFO, "Running policy: %s\n", policy->name);
    /* evaluate rules */
    temp_rule = policy->rules;
    while(temp_rule){
	EEF_log(LOG_DEBUG, "Evaluating rule: %s\n", temp_rule->name);
	if ( (temp_rule->plugin) == NULL ) {
	    /* plugin not linked to current rule - This should never happen
	     * */
	    EEF_log(LOG_WARNING,
		    "%s: Rule %s at line %i is not linked to a plugin!\n",
		    __func__, temp_rule->name, temp_rule->lineno);
	    return EES_FAILURE;
	}
	/* Run current plugin */
	set_running_plugin_name(temp_rule->plugin->name);
	if(run_plugin(temp_rule->plugin) == EES_PL_SUCCESS){
	    /* Plugin succeeded */
	    if (temp_rule->true_branch) {
		/* we have a true branch: go there */
		EEF_log(LOG_DEBUG, "Progressing to true branch: %s\n",
			temp_rule->true_branch->name);
		temp_rule = temp_rule->true_branch;
		continue;
	    } else {
		/* We don't have a true branch: we're done. */
		/* All rules were exhausted and a successful state was
		 * reached */
		EEF_log(LOG_DEBUG,
			"Executed policy %s successfully, ended in rule %s\n",
			policy->name, temp_rule->name);
		return EES_SUCCESS;
	    }
	} else {
	    /* Plugin failed */
	    if(temp_rule->false_branch){
		/* We have a false branch: go there */
		EEF_log(LOG_DEBUG, "Progressing to false branch: %s\n",
			temp_rule->false_branch->name);
		temp_rule = temp_rule->false_branch;
		continue;
	    } else {
		/* We don't have a false branch: go to the next policy. */
		/* All rules were exhausted and no successful state was
		 * reached */
		EEF_log(LOG_DEBUG,
			"Policy %s ended in failure, ended in rule %s\n",
			policy->name, temp_rule->name);
		break;
	    }
	}
    }

    return EES_FAILURE;
}

