/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE 500

#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif

#include "ees_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/syscall.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include "eics/eics_http.h"

#include "eef/eef_library.h"
#include "eef/eef_log.h"

/************************************************************************
 * Defines
 ************************************************************************/

#ifndef EES_PID_FILE
# define EES_PID_FILE   "/var/run/ees/ees.pid"
#endif


/************************************************************************
 * Global variables
 ************************************************************************/

static int signal_has_been_received = 0;/* signal handler state/lock */
static int caught_signal = 0;		/* signal being caught */
static long signal_thread;		/* thread in which signal was caught */

/************************************************************************
 * Private prototypes
 ************************************************************************/

static EES_RC ees_start(void);          /* initializes EEF */
static EES_RC ees_stop(void);           /* terminates EEF */
static int ees_loop(int debug_mode);	/* main loop - calls
					   start_xacml_http_server (http eic)
					   and pauses */

static int parse_cmdline_options(int argc, char *argv[], int *debug_mode);
static void usage (void);
static void version(void);
static void signal_handler(int sig);	/* main signal handler */
static void setup_signal_handlers(void);
static int write_pidfile(void);


/************************************************************************
 * Main
 ************************************************************************/

/**
 * Main
 */
int main (int argc, char* argv[]){
#if ENABLE_DEBUG
    int debug_mode=1;
#else
    int debug_mode=0;
#endif
    int rc;

    /* Parse command-line options */
    if (parse_cmdline_options(argc, argv, &debug_mode) == -1)
	return EXIT_FAILURE;

    /* Setup signal handlers */
    setup_signal_handlers();

    /* Now daemonize before starting the EES */
    if (!debug_mode)
	EEF_daemonize();

    /* Write the PID file */
    if (write_pidfile() == -1)
	return EXIT_FAILURE;

    /* initialize EEF: AOS, plugin manager and parse the config file, also start
     * threading */
    if(ees_start() != EES_SUCCESS)  {
	EEF_closelog();
	return EXIT_FAILURE;
    }

    /* Make sure the XACML server will be in threading mode */
    setenv("XACML_THREAD_MODEL","pthread",1);

    /* start xacml server loop */
    rc=ees_loop(debug_mode);

    if (rc==EXIT_FAILURE)
	EEF_closelog();

    return rc;
}


/************************************************************************
 * Private functions
 ************************************************************************/

static void usage (void){
    printf("Usage: ees [OPTION] ...\n");
    printf("Run the EES (Execution Environment Service).\n");
    printf("Normally the EES would be run using the init-script.\n\n");
    printf("Valid options:\n");
    printf("  -h, --help                  print this help text and exit\n");
    printf("  -v, --version               print EES version and exit\n");
    printf("  -d, --debug                 run in debug mode (foreground and debug level 5)\n");
    printf("  -l, --level <loglevel>      set log level [0-5], default 4\n");
    printf("  -f, --facility <facility>   set syslog facility, default LOG_DAEMON\n");
    printf("  -L, --logfile <file>        set log file, default empty=syslog\n");
    printf("  -p, --policies <policies>   set list of policies, policy(,policy...).\n");
    printf("                              Default: all policies\n");
    printf("  -P, --path <path>           set modules dir.\n");
    printf("                              Default %s\n", EES_MOD_HOME);
    printf("  -c, --conf <path>           Set a configuration file.\n");
    printf("                              Default: %s\n", EES_CONFIG_FILE);
    printf("\nReport bugs to grid-mw-security-support@nikhef.nl\n");
}

static void version(void)   {
    printf("EES %s\n", VERSION);
}

/**
 * Parse cmdline options, returns 0 on success, -1 on error
 */ 
static int parse_cmdline_options(int argc, char *argv[], int *debug_mode)    {
    int option_index = 0, c = 0;    /* getopt */
    int value;
    char *illval=NULL;
    static struct option long_options[] = /* options */
    {
        {"help",        no_argument,       NULL, 'h'},
	{"version",     no_argument,       NULL, 'v'},
        {"conf" ,       required_argument, NULL, 'c'},
        {"debug",       no_argument,       NULL, 'd'},
        {"level",       required_argument, NULL, 'l'},
	{"facility",    required_argument, NULL, 'f'},
        {"logfile",     required_argument, NULL, 'L'},
        {"policies",    required_argument, NULL, 'p'},
        {"path",        required_argument, NULL, 'P'},
	{NULL,          0,                 NULL, 0  }
    };

    opterr=1;
    optind=0;
    
    /* default config file path */
    EEF_set_config_file(EES_CONFIG_FILE);

    /* parse options */
    while(1){
        c = getopt_long_only(argc, argv, "hvc:dl:f:L:p:P:", long_options, &option_index);
        if(c == -1){
            break;
        }
        switch(c){
            case 'h':
                usage();
                exit(EXIT_SUCCESS);
	    case 'v':
		version();
		exit(EXIT_SUCCESS);
            case 'c':
		EEF_set_config_file(optarg);
                break;
            case 'd':
		*debug_mode = 1;
		EEF_set_debug_mode(1);
		EEF_set_conf_log_level(5);
		break;
	    case 'l':
		if (*debug_mode)    {
		    fprintf(stderr,
			    "%s: Ignoring setting of loglevel in debug mode, "
			    "%s will be 5\n",
			    argv[0], long_options[option_index].name);
		    EEF_log(LOG_INFO,
			    "%s: Ignoring setting of loglevel in debug mode, "
			    "%s will be 5\n", __func__, long_options[option_index].name);
		} else {
		    errno=0;
		    /* Note: we also might have something like -c: (*illval) will
		     * then become 'c' */
		    value=(int)strtol(optarg, &illval, 10);
		    if (errno || *illval!='\0' ||
			EEF_set_conf_log_level(value)!=EES_SUCCESS)
		    {
			fprintf(stderr,
				"%s: Cannot parse %s as valid loglevel\n",
				argv[0], optarg);
			EEF_log(LOG_ERR, "%s: Cannot parse %s as valid loglevel\n",
				__func__, optarg);
			return -1;
		    }
		}
		break;
	    case 'f':
		if (EEF_set_conf_log_facility(optarg) != EES_SUCCESS)   {
		    fprintf(stderr,
			    "%s: Cannot parse %s as valid log facility\n",
			    argv[0],optarg);
		    EEF_log(LOG_ERR,
			    "%s: Cannot parse %s as valid log facility\n",
			    __func__, optarg);
		    return -1;
		}
		break;
	    case 'L':
		EEF_set_conf_log_file(optarg);
		break;
	    case 'p':
		if (EEF_set_conf_policy_names(optarg) != EES_SUCCESS)	{
		    fprintf(stderr,
			    "%s: Converting %s to list of policies failed\n",
			    argv[0],optarg);
		    EEF_log(LOG_ERR,
			    "%s: Converting %s to list of policies failed\n",
			    argv[0],optarg);
		    return -1;
		}
		break;
	    case 'P':
		EEF_set_conf_modules_path(optarg);
		break;
            case '?':
		/* Note: opterr==1 makes the error printed to stderr */
                EEF_log(LOG_ERR, "%s: Error with option %s\n",
			__func__, argv[optind-1]);
                return -1;
	    default:
		EEF_log(LOG_ERR, "%s: Unknown option \"%c\"\n", __func__, c);
		return -1;
        }
    }

    return 0;
}


/**
 * Prepares logging, initializes EEF and puts the EEF in threading mode
 */
static EES_RC ees_start(void){
    /* reuse this part when implementing SIGHUP */
    if(EEF_init() == EES_FAILURE){
        EEF_log(LOG_ERR, "%s: Failed to initialize EEF\n", __func__);
        return EES_FAILURE;
    }

    /* Set threading flag in AOS and plugin manager */
    EEF_startThreading();

    /* Initialize threading for the XACML HTTP EIC */
    init_xacml_http_threading();

    EEF_log(LOG_DEBUG, "%s: EES initialized.\n", __func__);

    return EES_SUCCESS;
}

/**
 * Terminates EES
 */
static EES_RC ees_stop(void){
    /* Stop xacml server EIC */
    stop_xacml_http_server();

    /* Stop EEF library - evaluation manager, plugin manager and the AOS */
    if(EEF_term() == EES_SUCCESS)
        EEF_log(LOG_INFO, "%s: EEF terminated\n", __func__);
    else
        EEF_log(LOG_WARNING, "%s: Error while terminating EEF\n", __func__);

    fflush(stdout);
    fflush(stderr);
    EEF_log(LOG_NOTICE, "EES Stopped\n");
    EEF_closelog();

    return EES_SUCCESS;
}

/**
 * Main EES loop. Starts xacml server and pauses.
 */
static int ees_loop(int debug_mode){
    int cont=1, restart=1;
    int send_timeout=-1, recv_timeout=-1, acpt_timeout=-1;

    /* this is to prevent falling out of the loop when a signal gets handled */
    while(cont) {
        signal_has_been_received = 0;
	if (restart)	{
	    /* Initialize the HTTP EIC before starting it, do within the loop as
	     * we might have changed something in the config, and we certainly
	     * need to re-read the policy_names if we have received a SIGHUP */
	    EEF_get_http_timeouts(&send_timeout, &recv_timeout, &acpt_timeout);
	    set_xacml_http_timeouts(send_timeout, recv_timeout, acpt_timeout);
	    set_xacml_http_port(EEF_get_http_port());
	    set_xacml_http_policy_names(EEF_get_policy_names());

	    /* Start xacml server EIC */
	    if (start_xacml_http_server() != EES_SUCCESS) {
		ees_stop();
		return EXIT_FAILURE;
	    }
	}

	/*printf("Paused.\n");*/
	if (!signal_has_been_received)	{
	    EEF_log(LOG_NOTICE,"EES is ready to handle requests\n");
	    pause();
	}
	/* reset alarm */
	alarm(0);
	EEF_log(LOG_INFO,"%s: EES caught signal %i in thread %ld\n",
		__func__, caught_signal, signal_thread);

	switch(caught_signal) {
	    case SIGINT:
		EEF_log(LOG_NOTICE,
			"%s: Got SIGINT - shutting down\n", __func__);
		ees_stop();
		cont=0;
		break;
	    case SIGTERM:
		EEF_log(LOG_NOTICE,
			"%s: Got SIGTERM - shutting down\n", __func__);
		ees_stop();
		cont=0;
		break;
	    case SIGHUP:
		EEF_log(LOG_NOTICE,
			"%s: Got SIGHUP - restart\n", __func__);
		ees_stop();
		if (!debug_mode)
		    EEF_daemonize();

		/* Write the PID file */
		if (write_pidfile() == -1)
		    return EXIT_FAILURE;

		/* Now we are the new process, since the parent has quit
		 * in the EEF_daemonize() */
		if (ees_start() != EES_SUCCESS) {
		    fprintf(stderr,
			    "%s: Unable to restart - check logs.\n", __func__);
		    return EXIT_FAILURE;
		}
		restart=1;
		break;
	    case SIGUSR1:
		if(EEF_reopenlog() != EES_SUCCESS)  {
		    /* Can probably not log this, but in any case bail out */
		    ees_stop();
		    return EXIT_FAILURE;
		}
		restart=0;
		break;
	    default:
		restart=0;
		break;
	}
    }

    return EXIT_SUCCESS;
}

/**
 * Handles incoming signals
 */
static void signal_handler(int sig){
    if (signal_has_been_received == 0)	{
        signal_has_been_received = 1;
	caught_signal=sig;
#if defined(SYS_gettid)
	signal_thread=(long)syscall(SYS_gettid);
#elif defined(SYS_lwp_self)
	signal_thread=(long)syscall(SYS_lwp_self);
#else
	signal_thread=0L;
#endif
	/* set alarm to prevent we get stuck in pause() */
	alarm(1);
    }
}


/**
 * Sets up the signal handlers
 */
static void setup_signal_handlers(void)	{
    struct sigaction sa;

    if (sigemptyset(&sa.sa_mask))
        EEF_log(LOG_ERR, "%s: Error initializing signal handlers: %s\n",
		__func__, strerror(errno));

    /* catch hangup and term signals */
    sa.sa_handler = signal_handler;
    sa.sa_flags = SA_SIGINFO | SA_RESTART;

    if(sigaction(SIGHUP, &sa, NULL))
        EEF_log(LOG_ERR, "%s: Unable to add signal handler for SIGHUP: %s\n",
		__func__, strerror(errno));

    if(sigaction(SIGINT, &sa, NULL))
        EEF_log(LOG_ERR, "%s: Unable to add signal handler for SIGINT: %s\n",
		__func__, strerror(errno));

    if(sigaction(SIGTERM, &sa, NULL))
        EEF_log(LOG_ERR, "%s: Unable to add signal handler for SIGHUP: %s\n",
		__func__, strerror(errno));

    if(sigaction(SIGUSR1, &sa, NULL))
        EEF_log(LOG_ERR, "%s: Unable to add signal handler for SIGUSR1: %s\n",
		__func__, strerror(errno));
}


/*
 * Opens (truncates/creates) pidfile, writes current pid in it and closes it
 * again. It uses the location EES_PID_FILE.
 * Return -1 on error, 0 on success.
 */
static int write_pidfile(void)	{
    const char *pidfile=EES_PID_FILE;
    FILE *pidfilep;

    /* Open file */
    if ( (pidfilep=fopen(pidfile,"w"))==NULL)	{
	EEF_log(LOG_ERR,"%s: Cannot open pidfile %s: %s\n",
		__func__, pidfile,strerror(errno));
	return -1;
    }

    /* Write file */
    if (fprintf(pidfilep,"%ld",(long)getpid())<0)	{
	EEF_log(LOG_ERR,"%s: Writing pidfile %s failed: %s\n",
		__func__, pidfile,strerror(errno));
	return -1;
    }
    
    /* Close file */
    if (fclose(pidfilep)!=0)	{
	EEF_log(LOG_ERR,"%s: Closing pidfile %s failed: %s\n",
		__func__, pidfile,strerror(errno));
	return -1;
    }

    return 0;
}


