/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

%option noyywrap

%{
#include <string.h>

#include "eval_manager.h"
#include "pdl_yacc.h"
#include "eef/eef_log.h"

/* If we are running flex, we are creating our own prototype for the lexer
 * called pdl_yylex(). Flex will use that, bison will still call yylex() so we
 * create (for flex only) yylex as a wrapper around pdl_yylex(). This is all
 * needed to prevent warnings about  missing or redundant prototype. */
#ifdef FLEX_SCANNER
#ifndef YY_DECL
#define YY_DECL	static int pdl_yylex(void)
/* Declare prototype */
static int pdl_yylex(void);
/* Note: we don't need to declare a prototype for yylex() since we have that
 * already declared in pdl.h. flex will not declare it flex since we use
 * pdl_yylex() instead */
int yylex(void) {
    return pdl_yylex();
}
#endif
#endif

#ifndef _XOPEN_SOURCE
/* _XOPEN_SOURCE is defined by flex on supporting platforms, hence we are on a
 * non-supported one: The compiler might still have defined macros to get a
 * prototype for strdup and fileno. As long as they are not defined as macros,
 * the worst thing is a duplicate prototype. So check they aren't defined as
 * macro. */
#ifndef strdup
char *strdup(const char *s);
#endif /* strdup */
#ifndef fileno
int fileno(FILE *stream);
#endif /* fileno */
#endif /* _XOPEN_SOURCE */

#ifndef YY_FATAL_ERROR
#define YY_FATAL_ERROR(msg) \
    EEF_log(LOG_ERR, "%s: Fatal parsing error: %s\n", __func__, (msg));
#endif
%}

ws           [\t ]+
nl           \n
numval       [0-9]+
logfacility  LOG_[A-Z0-9]+
term         [a-zA-Z_\.0-9/]+
label        {term}:
policies     {term}(,{term})*
trans        "->"
comment      #.*\n
tokens       [\~=\|]
pvar         "path"
portvar      "port"
sendvar      "send_timeout"
receivevar   "receive_timeout"
acceptvar    "accept_timeout"
levelvar     "log_level"
facilityvar  "log_facility"
logvar       "log"
policyvar    "policies"
pdlstr       \"[^\"\n]*[\"\n]


%x path mq
%%

{ws}         ;
{pvar}       {	BEGIN path; return PVAR; }
{logvar}     {	BEGIN path; return LOGVAR; }
<path>{tokens}  { return yytext[0]; }
<path>{ws}      ;
<path>{term} {	if (!(yylval.record = (record_t *)malloc(sizeof(record_t))) ||
		    !((yylval.record)->string = strdup(yytext)))
		{
		    EEF_log(LOG_ERR, "%s: out of memory.\n", __func__);
		    return 0;
	        }
		(yylval.record)->lineno = lineno;
		return PATH;
             }
<path>\n        { ++lineno; BEGIN INITIAL; }
<path>{comment} { ++lineno; BEGIN INITIAL; }
<path>.       {	EEF_log(LOG_ERR, "%s: unrecognized tokens in path name.\n",
				 __func__); }
{levelvar}    {	return LEVELVAR; }
{portvar}     {	return PORTVAR; }
{sendvar}     {	return SENDVAR; }
{receivevar}  {	return RECEIVEVAR; }
{acceptvar}   {	return ACCEPTVAR; }
{facilityvar} { return FACILITYVAR; }
{policyvar}   { return POLICYVAR; }
{numval}      {	if (!(yylval.record = (record_t*)malloc(sizeof(record_t)))) {
		    EEF_log(LOG_ERR, "%s: out of memory.\n", __func__);
		    return 0;
		}
		if (sscanf(yytext,"%u",&((yylval.record)->uval))!=1)	{
		    EEF_log(LOG_ERR, "%s: %s is not a valid unsigned integer.\n",
			    __func__, yytext);
		    free(yylval.record);
		    yylval.record=NULL;
		    return 0;
		}
		(yylval.record)->lineno = lineno;
		return NUMVAL;
	      }
{logfacility} { if (!(yylval.record = (record_t*)malloc(sizeof(record_t))) ||
		    !((yylval.record)->string = strdup(yytext)))
		{
		    EEF_log(LOG_ERR, "%s: out of memory.\n",__func__);
		    return 0;
		}
		(yylval.record)->lineno = lineno;
		return LOGFACILITY;
              }
{term}	      { if (!(yylval.record = (record_t*)malloc(sizeof(record_t))) ||
		    !((yylval.record)->string = strdup(yytext)))
		{
		    EEF_log(LOG_ERR, "%s: out of memory.\n",__func__);
		    return 0;
		}
		(yylval.record)->lineno = lineno;
		return TERM;
              }
{label}	      { yytext[yyleng-1] = '\0';   /* remove trailing ':' */
		if (!(yylval.record = (record_t*)malloc(sizeof(record_t))) ||
		    !((yylval.record)->string = strdup(yytext)))
		{
		   EEF_log(LOG_ERR, "%s: out of memory.\n",__func__);
		   return 0; 
		}
		(yylval.record)->lineno = lineno;
		return LABEL;
              }
{policies}    { if (!(yylval.record = (record_t*)malloc(sizeof(record_t))) ||
		    !((yylval.record)->string = strdup(yytext)))
		{
		    EEF_log(LOG_ERR, "%s: out of memory.\n",__func__);
		    return 0;
		}
		(yylval.record)->lineno = lineno;
		return POLICIES;
              }
{nl}          { ++lineno;          }
{tokens}      { return yytext[0];  }
{trans}       { return TRANS;      }
{comment}     { ++lineno; BEGIN INITIAL ; }
{pdlstr}      { if ((yylval.record = (record_t*)malloc(sizeof(record_t)))) {
                 if (yytext[yyleng-2] == '\\') {
                   if (((yylval.record)->string = strdup(yytext+1))) {
                     (yylval.record)->string[yyleng-2] = '\0';
                     (yylval.record)->string[yyleng-3] = '\"';
                     (yylval.record)->lineno = lineno;
                     yyless(yyleng-1);  /*  make the next string start with "  */
                     BEGIN mq;
                     return STRING;
                   }
                 } else {
                   if (yytext[yyleng-1] == '\"') {
                     if (((yylval.record)->string = strdup(yytext+1))) {
                       (yylval.record)->string[yyleng-2] = '\0';
                       (yylval.record)->lineno = lineno;
                       return STRING;
                     }
                   }
                 }
                 EEF_log(LOG_ERR, "%s: out of memory.\n", __func__);
                 return 0;
               } else
                 EEF_log(LOG_ERR, "%s: string contains a newline.\n", __func__);
              }
<mq>{pdlstr}  { if ((yylval.record = (record_t*)malloc(sizeof(record_t)))) {
                 if (yytext[yyleng-2] == '\\') {
                   if (((yylval.record)->string = strdup(yytext+1))) {
                     (yylval.record)->string[yyleng-2] = '\0';
                     (yylval.record)->string[yyleng-3] = '\"';
                     (yylval.record)->lineno = lineno;
                     yyless(yyleng-1);  /*  make the next string start with "  */
                     BEGIN INITIAL;
                     return STRING;
                   }
                 } else {
                   EEF_log(LOG_ERR, "%s: Unbalanced quotes.\n", __func__);
                   return 0;
                 }
               }
               EEF_log(LOG_ERR, "%s: out of memory.\n", __func__);
               return 0;
             }
.            { EEF_log(LOG_WARNING, "%s: unrecognized token '%c'.\n",
		       __func__, yytext[0]);
               return yytext[0];
             }
%%

/**
 * Function to tidy up (most of) the memory, to be called after we're done with
 * parsing
 */
void pdl_lex_cleanup(void)	{
#ifdef FLEX_SCANNER
#if (YY_FLEX_MAJOR_VERSION>2) || \
    (YY_FLEX_MAJOR_VERSION==2 && YY_FLEX_MINOR_VERSION>5) || \
    (YY_FLEX_MAJOR_VERSION==2 && YY_FLEX_MINOR_VERSION==5 && \
     defined(YY_FLEX_SUBMINOR_VERSION) && YY_FLEX_SUBMINOR_VERSION>=9)
    yylex_destroy();
#else /* flex >= 2.5.9 */
    /* delete buffer */
    yy_delete_buffer(YY_CURRENT_BUFFER);
    /* make sure to put it to zero: note that on newer flex we should use
     * YY_CURRENT_BUFFER_LVALUE here instead of YY_CURRENT_BUFFER itself */
    YY_CURRENT_BUFFER=NULL;
    /* set init to 1 meaning we need to initialize */
    yy_init=1;
#endif /* flex >= 2.5.9 */
#endif /* FLEX_SCANNER */
}
