/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#include "ees_config.h"

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "eval_manager.h"
#include "plugin_manager.h"
#include "eef/eef_library.h"
#include "eef/eef_log.h"

/* Define format string to be used by add_rule for an unknown variable */
#define UNKNOWN_VAR_FORMAT_STRING   "%s: Unknown variable %s at line %i in config file %s"


/************************************************************************
 * Type definitions
 ************************************************************************/

/**
 * \struct pdl_config_t
 * A struct that holds the initial config options, note: the combination of
 * these plus what is obtained from the config file will be in config (a struct
 * pdl_config_s)
 */
typedef struct pdl_init_config_s {
    /* general */
    const char*	config_file_s;
    /* plugin modules path */
    const char*	modules_path;
    /* HTTP EIC values */
    int         port; 
    int		send_timeout;
    int         receive_timeout;
    int         accept_timeout;
    /* Logging values */
    int         log_level;	/* 0-5 */
    const char*	log_facility;	/* LOG_... */
    const char*	log_file;
    const char* policy_string; /* policy1,policy2 */
} pdl_init_config_t;


/**
 * \struct pdl_config_t
 * A struct that holds the actual config options
 */
typedef struct pdl_config_s {
    /* general */
    FILE*       config_file_fp; /* only used in config */
    /* plugin modules path */
    char*	modules_path;
    /* HTTP EIC values */
    int         port;
    int		send_timeout;
    int         receive_timeout;
    int         accept_timeout;
    /* Logging values */
    int         log_level;	/* 0-5 */
    char*	log_facility;	/* LOG_... */
    char*	log_file;
    char*	*policy_names;	/* {"policy1","policy2",NULL} */
} pdl_config_t;

/**
 * \struct pdl_data_s
 * A struct that holds internal data, and some error flags.
 */
typedef struct pdl_data_s {
    int         recursion_was_created;
    int         unknown_variable_was_referenced;
    int         starting_state_was_reused;
    int         parse_errors_detected;

    var_t*      variables_list;
    var_t*      variables_list_last;
    var_t*      current_variable;

    rule_t*     rules_list;
    rule_t*     rules_list_last;

    policy_t*   policies_list;
    policy_t*   policies_list_last;
} pdl_data_t;

/***********************************************************************
 * Global variables
 ***********************************************************************/
#if ENABLE_DEBUG
extern int yy_flex_debug;
#endif

/* Contains all the config entries used before parsing config file: these are
 * used to initialize config itself */
static pdl_init_config_t init_config =
    {	NULL,	/* config_file_s: only used in init_config */
	NULL,	/* modules_path */
	-1,	/* port */
	-1,	/* send_timeout */
    	-1,	/* receive_timeout */
    	-1,	/* accept_timeout */
	-1,	/* log_level */
	NULL,	/* log_facility */
	NULL,	/* log_file */
	NULL    /* policy-names string */
    };

/* Contains all the config entries used after parsing cmdline and config file */
static pdl_config_t config =
    {	NULL,	/* config_file_fp: only used in config */
	NULL,	/* modules_path */
	-1,	/* port */
	-1,	/* send_timeout */
	-1,	/* receive_timeout */
	-1,	/* accept_timeout */
	-1,	/* log_level */
	NULL,	/* log_facility */
	NULL,	/* log_file */
	NULL	/* policies */
    };

/* Contains all the internal PDL and config parsing data */
static pdl_data_t data =
    {	0,	/* recursion_was_created */
	0,	/* unknown_variable_was_referenced */
	0,	/* starting_state_was_reused */
	0,	/* parse_errors_detected */
	NULL,	/* variables_list */
	NULL,	/* variables_list_last */
	NULL,	/* current_variable */
	NULL,	/* rules_list */
	NULL,	/* rules_list_last */
	NULL,	/* policies_list */
	NULL	/* policies_list_last */
    };

int lineno;		    /* used by lex/yacc */


/***********************************************************************
 * Private prototypes
 ***********************************************************************/

static EES_RC	    parse_config_file(void);

/* Parsing helper functions */
static EES_RC	    add_plugin_structs(void);
static rule_t*	    check_for_recursion(rule_t*, rule_t*);
static EES_RC	    remove_unreachable_rules_in_policies(policy_t* policies);
static rule_t*	    remove_unreachable_rules(rule_t* rules, EES_RC*);
static void	    link_dead_end_rules_in_policies(policy_t* policy);
static void	    link_dead_end_rules(policy_t* policy, rule_t* rule);
static rule_t*	    get_left_hand_rule(rule_t*, const char*);
static rule_t*	    get_right_hand_rule(rule_t*, const char*);
static EES_RC	    link_rules_to_plugins(policy_t* policies);
static int	    link_rule_to_plugin(policy_t *policy, rule_t *rule);
static policy_t*    reduce_policies(policy_t* policies,
				    int number_of_policies,
				    char* names_of_policies[]);
/* Cleanup functions */
static policy_t*    remove_policy_by_name(policy_t* policies, char* policy);
static EES_RC	    clean_variables_list(void);
static EES_RC	    clean_policies_list(policy_t* policies);
static rule_t*	    clean_rules_list(rule_t*);
static rule_t*	    clean_rules_tree(rule_t*);
static void         clean_policy_name_list(void);
/* Helper functions */
static var_t*	    get_variable_by_name(char*);
static char **      _var_to_argv(const char*, const char *);
static char *       parse_modules_path(const char *path);

/* debug functions */
static void         print_policies(policy_t* policies);
static void         print_rules(policy_t* policy, rule_t* rules);


/* Initialize the config structure with the init_config structure:
 * the former is filled by the pdl parser, the latter can be filled using e.g.
 * command line options and survives a restart of the service.
 */
static EES_RC	    initialize_config(void);
/* Clean data in configuration options */
static void clean_config_data(void);

/***********************************************************************
 * Public functions
 ***********************************************************************/

/*! Sets the config file */
void EEF_set_config_file(const char *file)	{
    init_config.config_file_s=file;
}

/*! Sets the initial config and config module path */
void EEF_set_conf_modules_path(const char *path) {
    if (config.modules_path)
	free(config.modules_path);

    /* Parse path */
    config.modules_path=parse_modules_path(path);
    if ( config.modules_path )
	init_config.modules_path=path;
    else {
	init_config.modules_path=NULL;
	EEF_log(LOG_WARNING, "%s: Error copying path\n", __func__);
    }
}

/*! Sets the initial config and config port to be used the HTTP EIC */
EES_RC EEF_set_conf_http_port(int port)    {
    if (port<1 || port>65535)   {
	EEF_log(LOG_ERR,"%s: Invalid port number: %d\n", __func__, port);
	return EES_FAILURE;
    }
    init_config.port=config.port=port;
    return EES_SUCCESS;
}

/*! Sets the initial config and config timeouts to be used by the HTTP EIC */
EES_RC EEF_set_conf_http_timeouts(int send_timeout,
				  int rcve_timeout,
				  int acpt_timeout)    {
    if (send_timeout>=0)
	init_config.send_timeout=config.send_timeout=send_timeout;
    else if (send_timeout!=-1)
	return EES_FAILURE;

    if (rcve_timeout>=0)
	init_config.receive_timeout=config.receive_timeout=rcve_timeout;
    else if (rcve_timeout!=-1)
	return EES_FAILURE;

    if (acpt_timeout>=0)
	init_config.accept_timeout=config.accept_timeout=acpt_timeout;
    else if (acpt_timeout!=-1)
	return EES_FAILURE;

    return EES_SUCCESS;
}


/*! Sets the initial config, config and actual log level */
EES_RC EEF_set_conf_log_level(int level)	{
    /* Set log level */
    EES_RC rc=EEF_set_log_level(level);

    /* Set init loglevel */
    if (rc==EES_SUCCESS)
	init_config.log_level=config.log_level=level;
    
    return rc;
}

/*! Sets the initial config, config and actual log facility */
EES_RC EEF_set_conf_log_facility(const char *facility)	{
    /* Set log facility */
    EES_RC rc=EEF_set_log_facility(facility);

    /* Set init log facility */
    if (rc==EES_SUCCESS)    {
	init_config.log_facility=facility;
	if (config.log_facility)
	    free(config.log_facility);
	if ( (config.log_facility=strdup(init_config.log_facility))==NULL)
	    EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
    }

    return rc;
}

/*! Sets the initial config, config and actual log file, also opens it */
EES_RC EEF_set_conf_log_file(const char *file)	{
    /* Set log file (also opens it) */
    EES_RC rc=EEF_set_log_file(file);

    if (rc==EES_SUCCESS)    {
	init_config.log_file=file;
	if (config.log_file)
	    free(config.log_file);
	if ( (config.log_file=strdup(init_config.log_file))==NULL)
	    EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
    }

    return rc;
}

/*! Sets the initial list of policies, config and actual list of policies */
EES_RC EEF_set_conf_policy_names(const char *policy_string) {
    if (config.policy_names)
	clean_policy_name_list();

    /* Convert to array of policy names */
    if (policy_string)	{
        config.policy_names=_var_to_argv(policy_string, ",");
	if (config.policy_names==NULL)	{
	    init_config.policy_string=NULL;
	    return EES_FAILURE;
	}
    }
    /* Store the initial value */
    init_config.policy_string=policy_string;

    return EES_SUCCESS;
}


/*! Return the config file */
const char *EEF_get_config_file(void)	{
    return init_config.config_file_s;
}

/*! Return the modules path */
const char *EEF_get_modules_path(void){
    return config.modules_path;
}

/*! Returns the port for the HTTP EIC */
int EEF_get_http_port(void)    {
    return config.port;
}

/*! Returns the timeouts for the HTTP EIC */
void EEF_get_http_timeouts(int *send_timeout,
			   int *rcve_timeout,
			   int *acpt_timeout)    {
    if (*send_timeout)
	*send_timeout=config.send_timeout;
    if (*rcve_timeout)
	*rcve_timeout=config.receive_timeout;
    if (*acpt_timeout)
	*acpt_timeout=config.accept_timeout;
}

/*! Returns the policy_names array */
char **EEF_get_policy_names(void)   {
    return config.policy_names;
}


/* This function does everything parser related.
 * After running yacc through parse_config_file(), the created list is reduced
 * to those in the array of named policies, if number_of_policies is greater
 * than 0. The policies are checked for unreachable rules. After the parser has
 * created a list of policies containing lists of rules, these rules are
 * transformed to a tree by the link_dead_end_rules_in_policies() function.
 * Finally, when a list of policies containing rules trees was created, the
 * plug-ins are linked to each rule in the node. */
EES_RC start_eval_manager(int number_of_policies,
			  char* names_of_policies[]){
    /* (re)initialize the config struct: note that we can change with e.g.
     * cmdline arguments the init_config values. */
    if ( (initialize_config()==EES_FAILURE) )
	return EES_FAILURE;

    EEF_log(LOG_INFO, "%s: Parsing config file %s\n",
	    __func__, init_config.config_file_s);

    /* Note: all the log-related entries are executed as soon as they are found
     * in the parsing. This includes opening the log file */
    if(parse_config_file() != EES_SUCCESS){
	/* Something went wrong */
	EEF_log(LOG_ERR, "%s: Problem with config file: %s\n",
		__func__, init_config.config_file_s);
	if(data.recursion_was_created)
	    EEF_log(LOG_ERR,
		    "  The loaded configuration file defines recursive rules. "
		    "Please check your configuration file.\n");
	if(data.unknown_variable_was_referenced)
	    EEF_log(LOG_ERR,
		    "  The loaded configuration file references undefined "
		    "variables. Please check your configuration file.\n");
	if(data.starting_state_was_reused)
	    EEF_log(LOG_ERR,
		    "  The loaded configuration file reuses a starting state. "
		    "Please check your configuration file.\n");
	if(data.parse_errors_detected)
	    EEF_log(LOG_ERR,
		    "  The configuration was not parsed correctly. "
		    "Please check your configuration file.\n");

	return EES_FAILURE;
    }

    /* We need to start the plugin manager before adding the plugin structures,
     * but we also need to have the plugin structures before checking the
     * link_rules_to_plugins, so start plugin manager here */

    /* Start the plugin manager */
    if ( start_plugin_manager() != EES_SUCCESS )    {
	EEF_log(LOG_ERR, "%s: Failed to start plugin manager\n", __func__);
	return EES_FAILURE;
    }
    EEF_log(LOG_DEBUG, "%s: Plugin manager successfully started\n", __func__);

    /* Now add all the plugin definitions from the pdl */
    if (add_plugin_structs() != EES_SUCCESS)	{
	EEF_log(LOG_ERR,
		"%s: Failed to load plug-ins from policy config file %s\n",
		__func__, init_config.config_file_s);
	return EES_FAILURE;
    }

    /* strip policies that are not explicitly passed in the
     * names_of_policies array */
    if (number_of_policies) {
	data.policies_list =
	    reduce_policies(data.policies_list,
			    number_of_policies,
			    names_of_policies);
    }

    if (remove_unreachable_rules_in_policies(data.policies_list)
	    != EES_SUCCESS) {
	EEF_log(LOG_ERR,
		"%s: The loaded configuration file defines unreachable rules. "
		"Please check your configuration file.\n", __func__);
	return EES_FAILURE;
    }

    link_dead_end_rules_in_policies(data.policies_list);
    if (link_rules_to_plugins(data.policies_list)!=EES_SUCCESS)	{
	EEF_log(LOG_ERR, "%s: Linking rules to plugins failed.\n", __func__);
	return EES_FAILURE;
    }

    print_policies(data.policies_list);

    return EES_SUCCESS;
}

/*! Terminates the parser and tries to free all used memory */
void stop_eval_manager(void){
    clean_variables_list();
    clean_policies_list(data.policies_list);

    /* Clean the config data */
    clean_config_data();

    /* Clean policy name list */
    clean_policy_name_list();

    /* Reset data struct */
    memset(&data,0,sizeof(pdl_data_t));

    pdl_lex_cleanup(); 
}

/*! Returns configured policies */
policy_t* get_policies(){
    return data.policies_list;
}

/*! Handles a variable representing a plug-in; creates a structure containing
 * its values and adds it to the list of var_t structs variables_list */
void add_variable(record_t* name, record_t* value){
    /* allocate struct and populate fields */
    if (!name || !name->string || !value || !value->string)
	return;

    if ((data.current_variable = calloc(1,sizeof(var_t)))==NULL || 
	(data.current_variable->name = strdup(name->string))==NULL ||
	(data.current_variable->value = strdup(value->string))==NULL)
    {
	EEF_log(LOG_ERR, "%s: Out of memory!\n", __func__);
	free(data.current_variable);
	return;
    }
    data.current_variable->lineno = name->lineno;
    data.current_variable->next = NULL;

    /* append to the end of the list */
    if(data.variables_list){
        data.variables_list_last->next = data.current_variable;
    } else {
        data.variables_list = data.current_variable;
    }
    data.variables_list_last = data.current_variable;

    /* clean up */
    free(name->string);
    free(name);
    free(value->string);
    free(value);
}

/*! Adds a new policy_t structure to the policies_list which will hold the
 * current rules_list. rules_list is nulled after creating a policy so a new
 * list can be created for the next policy */
void add_policy(record_t* policy, rule_t* top_rule){
    policy_t* new_policy = NULL;

    if (!policy || !policy->string)
	return;

    if ((new_policy = calloc(1, sizeof(policy_t)))==NULL || 
	(new_policy->name = strdup(policy->string))==NULL)
    {
	EEF_log(LOG_ERR, "%s: Out of memory!\n", __func__);
	free(new_policy);
	return;
    }
    new_policy->lineno = policy->lineno;
    new_policy->rules = top_rule;
    new_policy->rules_list_transformed_to_tree = 0;
    new_policy->next = NULL;

    /* append to the end of the list */
    if(data.policies_list)
        data.policies_list_last->next = new_policy;
    else
        data.policies_list = new_policy;
    data.policies_list_last = new_policy;

    /* start a new rules list */
    data.rules_list = NULL;

    free(policy->string);
    policy->string = NULL;
    free(policy);
    policy = NULL;
}

/* TODO I think this function is a bit of a kludge and could probably be
 * rewritten to be more efficient and more legible - I think it might do too
 * much, so extracting some functions might help */
/*! Appends a rule to the global rules_list, which is to be added to a policy
 * when add_policy is called.
 * Will return NULL and log an error when:
 - adding the rule to the list will create a recursion in the tree of rules
 - an unknown variable is referenced
*/
rule_t* add_rule(record_t* state, record_t* true_branch, record_t* false_branch){
    rule_t *new_rule = NULL, *new_false_branch = NULL,
                  *new_true_branch = NULL, *recursive_rule = NULL;
    var_t  *temp_var = NULL;

    if (!state)
	return NULL;

    if((new_rule = get_left_hand_rule(data.rules_list, state->string))){
        EEF_log(LOG_WARNING,
		"%s: State %s at line %i is already in use at line %i.\n",
		__func__, state->string, state->lineno, new_rule->lineno);
        data.starting_state_was_reused = 1;
    } else {
        /*find variables for rule */
        temp_var = get_variable_by_name(state->string);
        if(temp_var == NULL){
	    /* Errorous state - variable referenced in rule not previously
	     * defined */
            EEF_log(LOG_ERR, UNKNOWN_VAR_FORMAT_STRING, __func__,
		    state->string, state->lineno, init_config.config_file_s);
            data.unknown_variable_was_referenced = 1;
            new_rule = clean_rules_tree(new_rule);
            goto cleanup;
        }
        if(temp_var != NULL){
            if ((new_rule = calloc(1, sizeof(rule_t)))==NULL ||
		(new_rule->name = strdup(state->string))==NULL)
	    {
		EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
		free(new_rule); new_rule=NULL;
		goto cleanup;
	    }
	    /* populate fields of current state */
	    new_rule->lineno = state->lineno;

	    /* populate fields of branches */
	    if(false_branch){
		temp_var = get_variable_by_name(false_branch->string);
		if(temp_var == NULL){
		    /* Errorous state - variable referenced in rule not
		     * previously defined */
		    EEF_log(LOG_ERR, UNKNOWN_VAR_FORMAT_STRING, __func__,
			    false_branch->string,
			    false_branch->lineno,
			    init_config.config_file_s);
		    data.unknown_variable_was_referenced = 1;
		    new_rule = clean_rules_tree(new_rule);
		    goto cleanup;
		} else {
		    if ((new_false_branch = calloc(1, sizeof(rule_t)))==NULL ||
			(new_false_branch->name = strdup(false_branch->string))==NULL)
		    {
			EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
			free(new_false_branch); new_false_branch=NULL;
			goto cleanup;
		    }
		    new_false_branch->lineno = false_branch->lineno;
		    new_rule->false_branch = new_false_branch;
		}
	    }

	    if(true_branch){
		temp_var = get_variable_by_name(true_branch->string);
		if(temp_var == NULL){
		    /* Errorous state - variable referenced in rule not
		     * previously defined */
		    EEF_log(LOG_ERR, UNKNOWN_VAR_FORMAT_STRING, __func__,
			    true_branch->string,
			    true_branch->lineno, init_config.config_file_s);
		    data.unknown_variable_was_referenced = 1;
		    new_rule = clean_rules_tree(new_rule);
		    goto cleanup;
		} else {
		    if ((new_true_branch = calloc(1, sizeof(rule_t)))==NULL ||
			(new_true_branch->name = strdup(true_branch->string))==NULL)
		    {
			EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
			free(new_true_branch); new_true_branch=NULL;
			goto cleanup;
		    }
		    new_true_branch->lineno = true_branch->lineno;
		    new_rule->true_branch = new_true_branch;
		}
	    }

	    /* check for recursion */
	    if((recursive_rule = check_for_recursion(data.rules_list, new_rule))){
		EEF_log(LOG_WARNING,
			"%s: Rule %s at line %i leads to recursion into state %s\n",
			__func__,
			new_rule->name, new_rule->lineno, recursive_rule->name);
		new_rule = clean_rules_tree(new_rule);
		data.recursion_was_created = 1;
		goto cleanup;
	    } else {
		/* add new rule at the end of the rules list */
		if(data.rules_list){
		    data.rules_list_last->next = new_rule;
		} else {
		    data.rules_list = new_rule;
		}
		data.rules_list_last = new_rule;

		EEF_log(LOG_DEBUG, "Added a new rule: %s\n", new_rule->name);
	    }
        }
    }

cleanup:

    /* clean up */
    if(state != NULL){
        free(state->string);
        free(state);
    }
    if(true_branch != NULL){
        free(true_branch->string);
        free(true_branch);
    }
    if(false_branch != NULL){
        free(false_branch->string);
        free(false_branch);
    }

    return new_rule;
}

/*! Removes a policy from the list of policies */
void remove_policy(record_t* policy){
    policy_t *temp_policy = data.policies_list, *next_policy = NULL;
    EEF_log(LOG_DEBUG, "Deleted policy: %s\n", policy->string);

    while(temp_policy){
        if(strcmp(temp_policy->name, policy->string) == 0){
            /* target policy was found */
            next_policy = temp_policy->next;

            /* Clean policy fields */
            if(temp_policy->rules_list_transformed_to_tree){
                temp_policy->rules = clean_rules_tree(temp_policy->rules);
            } else {
                temp_policy->rules = clean_rules_list(temp_policy->rules);
            }
            free(temp_policy->name);
            temp_policy->name = NULL;
            free(temp_policy);
            temp_policy = NULL;
        }

        /* Move to next policy */
        temp_policy = next_policy;
    }

    free(policy->string);
    free(policy);
}

/*! sets the path to modules directory */
void set_pdl_modules_path(record_t* path){
    if(!path || !path->string)
        return;

    /* First log what we found */
    EEF_log(LOG_DEBUG, "Found modules path: %s\n", path->string);

    /* is it already set? */
    if (config.modules_path)  {
	EEF_log(LOG_INFO,
		"modules path already initialized to %s, "
		"ignoring setting in config file on line %d\n",
		config.modules_path, path->lineno);
    } else {
	/* Parse and set modules_path */
	config.modules_path=parse_modules_path(path->string);
	if ( config.modules_path == NULL)
	    EEF_log(LOG_WARNING, "%s: Error copying path\n", __func__);
    }

    free(path->string);
    free(path);
}


/*! sets the path to the logfile: both in config and also in logging */
void set_pdl_log_file(record_t* path){
    if (!path || !path->string)
        return;

    if (config.log_file)  {
	EEF_log(LOG_INFO,
		"log_file already initialized to %s, "
		"ignoring setting in config file on line %d\n",
		config.log_file, path->lineno);
    } else  {
	/* Need to first copy into config.log_file */
	if ( (config.log_file=strdup(path->string)) == NULL )
	    EEF_log(LOG_WARNING, "%s: Error: out of memory\n", __func__);
	else    {
	    if (EEF_set_log_file(config.log_file)!=EES_SUCCESS)	{
		free(config.log_file);
		config.log_file=NULL;
	    }
	    else
		EEF_log(LOG_DEBUG, "Found and opened logfile path: %s\n",
			config.log_file);
	}
    }

    free(path->string);
    free(path);
}

/*! sets the EES loglevel: both in config and also in logging */
void set_pdl_log_level(record_t* loglevel){
    if (!loglevel)
	return;

    if (config.log_level>=0)  {
	EEF_log(LOG_INFO,
		"Loglevel already initialized to %d, "
		"ignoring setting in config file on line %d\n",
		config.log_level, loglevel->lineno);
    } else {
	if (EEF_set_log_level((int)(loglevel->uval))==EES_SUCCESS)  {
	    config.log_level=(int)(loglevel->uval);
	    EEF_log(LOG_DEBUG, "Found log level: %u\n",loglevel->uval);
	} else
	    EEF_log(LOG_WARNING, "%s: Loglevel %u is invalid, keeping %d\n",
		    __func__, loglevel->uval, EEF_get_log_level());
    }
    free(loglevel);
}

/*! sets the EES logfacility: both in config and also in logging */
void set_pdl_log_facility(record_t* logfacility){
    if (!logfacility || !logfacility->string)
	return;

    if (config.log_facility)   {
	EEF_log(LOG_INFO,
		"Log facility already initialized to %s, "
		"ignoring setting in config file on line %d\n",
		config.log_facility, logfacility->lineno);
    } else  {
	if (EEF_set_log_facility(logfacility->string)==EES_SUCCESS)	{
	    if ( (config.log_facility=strdup(logfacility->string))==NULL)
		EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
	    else
		EEF_log(LOG_DEBUG, "Found logfacility: %s\n",
			logfacility->string);
	} else
	    EEF_log(LOG_ERR, "%s: Cannot parse logfacility: %s\n",
		    __func__, logfacility->string);
    }

    free(logfacility->string);
    logfacility->string = NULL;
    free(logfacility);
}

/*! sets the EES listening port */
void set_pdl_port(record_t* port){
    if (!port)
	return;

    if (config.port>=0)  {
	EEF_log(LOG_INFO,
		"port already initialized to %d, "
		"ignoring setting in config file on line %d\n",
		config.port, port->lineno);
    } else if (port->uval<1 || port->uval>65535)   {
	config.port=-1;
	EEF_log(LOG_ERR, "%s: Not a valid port number: %s\n",
		__func__, port->string);
    } else {
	config.port=(int)port->uval;
	EEF_log(LOG_DEBUG, "Found listening port: %d\n", config.port);
    }
    free(port);
}

/*! sets the EES send timeout */
void set_pdl_send_timeout(record_t* timeout){
    if (!timeout)
	return;

    if (config.send_timeout>=0)  {
	EEF_log(LOG_INFO,
		"send timeout already initialized to %d, "
		"ignoring setting in config file on line %d\n",
		config.send_timeout, timeout->lineno);
    } else {
	config.send_timeout=(int)timeout->uval;
	EEF_log(LOG_DEBUG, "Found listening send timeout: %d\n",
		config.send_timeout);
    }
    free(timeout);
}

/*! sets the EES receive timeout */
void set_pdl_receive_timeout(record_t* timeout){
    if (!timeout)
	return;

    if (config.receive_timeout>=0)  {
	EEF_log(LOG_INFO,
		"receive timeout already initialized to %d, "
		"ignoring setting in config file on line %d\n",
		config.receive_timeout, timeout->lineno);
    } else {
	config.receive_timeout=(int)timeout->uval;
	EEF_log(LOG_DEBUG, "Found listening receive timeout: %d\n",
		config.receive_timeout);
    }
    free(timeout);
}

/*! sets the EES accept timeout */
void set_pdl_accept_timeout(record_t* timeout){
    if (!timeout)
	return;

    if (config.accept_timeout>=0)  {
	EEF_log(LOG_INFO,
		"accept timeout already initialized to %d, "
		"ignoring setting in config file on line %d\n",
		config.accept_timeout, timeout->lineno);
    } else {
	config.accept_timeout=(int)timeout->uval;
	EEF_log(LOG_DEBUG, "Found listening accept timeout: %d\n",
		config.accept_timeout);
    }
    free(timeout);
}

/*! sets the policies to be used */
void set_pdl_policynames(record_t *policy_string)  {
    if (!policy_string || !policy_string->string)
	return;

    if (config.policy_names)    {
	EEF_log(LOG_INFO,
		"policies already initialized to \"%s\", "
		"ignoring setting in config file on line %d\n",
		init_config.policy_string, policy_string->lineno);
    } else {
	config.policy_names=_var_to_argv(policy_string->string, ",");
	if (!config.policy_names)
	    EEF_log(LOG_WARNING,
		    "Found policy names \"%s\" but could not extract\n",
		    policy_string->string);
	else
	    EEF_log(LOG_DEBUG,
		    "Found policy names %s\n",
		    policy_string->string);
    }
    free(policy_string->string);
    free(policy_string);
}

/*! concatenates two strings with a space in between */
record_t* concat_strings_with_space(record_t *r1, record_t* r2){
    record_t *r;
    size_t len;
    int sn_rc;
    
    if (r1==NULL || r1->string==NULL || r2==NULL || r2->string==NULL)
	return NULL;

    len=strlen(r1->string)+strlen(r2->string)+2;

    if ( (r = malloc(sizeof(record_t)))==NULL ||
	 (r->string=calloc(len, sizeof(char)))==NULL )	{
	    EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    free(r);
	    return NULL;
    }

    sn_rc=snprintf(r->string, len, "%s %s", r1->string, r2->string);

    if ( sn_rc<0 || (size_t)sn_rc != (len-1) )	{
	EEF_log(LOG_ERR, "%s: Error concatenating strings (len=%lu, rc=%d)\n",
		__func__, (unsigned long)len, sn_rc);
	free(r->string);
	free(r);
	return NULL;
    }

    free(r1->string);
    free(r1);
    free(r2->string);
    free(r2);
    return r;
}

/*! logs errors encountered during parsing */
int yyerror(char* string){
    data.parse_errors_detected = 1;
    EEF_log(LOG_ERR, "%s: Parse error: %s at line %i in config file %s\n",
	    __func__, string, lineno, init_config.config_file_s);
    return 0;
}


/***********************************************************************
 * Private functions
 ***********************************************************************/

/*! Initializes the parsing of the configuration file. When parsing has
 * completed successfully, the policy_list is available. This can be transformed
 * to a tree using the link_dead_end_rules_in_policies() function. */
static EES_RC parse_config_file(void){
#if ENABLE_DEBUG
    yy_flex_debug = 1;
#endif

    lineno=1;

    if ((config.config_file_fp = fopen(init_config.config_file_s,"r")) == NULL)   {
        EEF_log(LOG_ERR,
		"Failed to open policy config file %s\n",
		init_config.config_file_s);
	return EES_FAILURE;
    }

    yyin = config.config_file_fp;

    /* Call the parser, note this is a wrapper around the actual yyparse() */
    pdl_yyparse();

    /* did the parsing succeed */
    if (data.recursion_was_created ||
	data.unknown_variable_was_referenced ||
	data.starting_state_was_reused ||
	data.parse_errors_detected)
    {
	return EES_FAILURE;
    }
    
    return EES_SUCCESS;
}

/**
 * Converts path and returns absolute path
 */
static char *parse_modules_path(const char *path)	{
    size_t len;
    int sn_rc;
    char *effpath;

    if (path[0]=='/')	{
	/* Absolute path */
	if ( (effpath=strdup(path)) == NULL )
	    EEF_log(LOG_WARNING, "%s: Error: out of memory\n", __func__);
    } else {
	/* Relative path */
	len=strlen(EES_LIB_HOME)+strlen(path)+2;
	if ( (effpath=calloc(len, sizeof(char))) == NULL)
	    EEF_log(LOG_WARNING, "%s: Error: out of memory\n", __func__);
	else {
	    /* Note: length is pre-alloced, so must match */
	    sn_rc=snprintf(effpath,len, "%s/%s\n", EES_LIB_HOME,path);
	    if (sn_rc<0 || (size_t)sn_rc!=len-1) {
		EEF_log(LOG_WARNING, "%s: Error copying path\n", __func__);
		free(effpath);
		return NULL;
	    }
	}
    }

    return effpath;
}

/*! Iterates list of policies and the rules they contain and tries to let the
 * plugin manager prepare plugin structs*/
static EES_RC add_plugin_structs(){
    char**    argv;
    var_t*    temp_var = data.variables_list;

    while (temp_var != NULL)	{
        if ( (argv = _var_to_argv(temp_var->value, " \t"))==NULL)
	    return EES_FAILURE;

	/* this is a callout to the plug-in manager, which adds a struct
	 * describing a single plug-in to its list */
	if (add_plugin_struct(argv, temp_var->name) != EES_SUCCESS)
	    return EES_FAILURE;

        /* Move to next variable */
        temp_var = temp_var->next;
    }

    return EES_SUCCESS;
}

/* Tries to find a recursion in the list of rules by iterating through all the
 * tree branches. When the same rule is encountered in the current list a
 * recursion has been found */
static rule_t* check_for_recursion(rule_t* rule_l, rule_t* rule_r){
    rule_t* temp_rule = rule_l;

    /* right hand rule leads to its own recursive state */
    /* still need to find a way to reuse the code from the while loop to do this
     * check */
    if(rule_r){
        if(rule_r->true_branch &&
           strcmp(rule_r->name, rule_r->true_branch->name) == 0)
            return rule_r->true_branch;

        if(rule_r->false_branch &&
	   strcmp(rule_r->name, rule_r->false_branch->name) == 0)
            return rule_r->false_branch;
    }

    /* iterate list of rules */
    while(temp_rule){
        if((temp_rule != NULL) && (rule_r != NULL)){
            /* left hand and right hand state are equal */
            if(strcmp(temp_rule->name, rule_r->name) == 0)
                return rule_r;

            /* start state and true branch of right hand rule are equal */
            if(rule_r->true_branch &&
               strcmp(temp_rule->name, rule_r->true_branch->name) == 0)
                return rule_r->true_branch;

            /* start state and false branch of right hand side are equal */
            if(rule_r->false_branch &&
               strcmp(temp_rule->name, rule_r->false_branch->name) == 0)
                return rule_r->false_branch;
        }

        /* move to next rule */
        temp_rule = temp_rule->next;
    }
    return NULL;
}

/* This function iterates the passed list of policies and removes unreachable
 * rules from each policy */
static EES_RC remove_unreachable_rules_in_policies(policy_t* policies){
    policy_t *temp_policy = policies;
    EES_RC retval = EES_SUCCESS;

    while(temp_policy){
        temp_policy->rules = remove_unreachable_rules(temp_policy->rules,
						      &retval);
        temp_policy = temp_policy->next;
    }
    return retval;
}

/* This function iterates the passed list of rules and removes those rules that,
 * while they exist on the left-hand side, are never referenced on the
 * right-hand side of a rule in the list */
static rule_t* remove_unreachable_rules(rule_t* rules, EES_RC *retval){
    rule_t *temp_rule = NULL, *next_rule = NULL, *previous_rule = rules;

    if(rules){
        temp_rule = rules;
        while(temp_rule){
            next_rule = temp_rule->next;

            if( !(get_left_hand_rule(rules, temp_rule->name)) &&
		!(get_right_hand_rule(rules, temp_rule->name)) )
	    {
                EEF_log(LOG_WARNING, 
			"Removing unreachable rule %s at line %i\n",
		        temp_rule->name, temp_rule->lineno);
                clean_rules_tree(temp_rule);
                previous_rule->next = next_rule;
                temp_rule = previous_rule;
                *retval = EES_FAILURE;
            }
            previous_rule = temp_rule;
            temp_rule = next_rule;
        }
    }
    return rules;
}

static void link_dead_end_rules_in_policies(policy_t* policy){
    policy_t* temp_policy = policy;

    while(temp_policy){
        link_dead_end_rules(temp_policy, temp_policy->rules);
        temp_policy = temp_policy->next;
    }
}

static void link_dead_end_rules(policy_t* policy, rule_t* rule){
    rule_t *temp_rule = rule, *true_rule, *false_rule;

    while(temp_rule){
        if(temp_rule->true_branch){
            if((true_rule = get_left_hand_rule(policy->rules,
					      temp_rule->true_branch->name)))
	    {
                EEF_log(LOG_DEBUG, "Overwriting rule %s with %s\n",
			temp_rule->true_branch->name, true_rule->name);
                clean_rules_tree(temp_rule->true_branch);
                temp_rule->true_branch = true_rule;
            }
        }
        if(temp_rule->false_branch){
            if((false_rule = get_left_hand_rule(policy->rules,
						temp_rule->false_branch->name)))
	    {
                EEF_log(LOG_DEBUG, "Overwriting rule %s with %s\n",
			temp_rule->false_branch->name,false_rule->name);
                clean_rules_tree(temp_rule->false_branch);
                temp_rule->false_branch = false_rule;
            }
        }
        temp_rule = temp_rule->next; 
    }
    policy->rules_list_transformed_to_tree = 1;
}

/*! Tries to find specified state in the rules_list */
static rule_t* get_left_hand_rule(rule_t* temp_rule, const char* name){
    if(!temp_rule || !name)
        return NULL;

    /* iterate while the rule isn't associated with the given state */
    while(temp_rule){
        if(!strcmp(name, temp_rule->name))
            return temp_rule;
        temp_rule = temp_rule->next;
    }
    return NULL;
}

/*! Tries to find specified state in the rules_list */
static rule_t* get_right_hand_rule(rule_t* temp_rule, const char* name){
    if(!temp_rule || !name)
        return NULL;
    
    /* iterate while the rule isn't associated with the given state */
    if (temp_rule->true_branch && 
        strcmp(name, temp_rule->true_branch->name) == 0)
	    return temp_rule->true_branch;

    if (temp_rule->false_branch &&
	strcmp(name, temp_rule->false_branch->name) == 0)
	    return temp_rule->false_branch;

    return NULL;
}

/* tries to link all policy rules to a plugin in the plugin manager */
static EES_RC link_rules_to_plugins(policy_t* policies){
    policy_t *temp_policy = policies;

    while(temp_policy){
        if (link_rule_to_plugin(temp_policy, temp_policy->rules)==-1)
	    return EES_FAILURE;
        temp_policy = temp_policy->next;
    }
    return EES_SUCCESS;
}

static int link_rule_to_plugin(policy_t *policy, rule_t *rule){
    var_t    *temp_var    = NULL;

    eef_plugindl_t *temp_plugin = NULL;
    if (rule){
        if(!rule->plugin){
            temp_var = get_variable_by_name(rule->name);
            if(temp_var){
                temp_plugin = get_plugin(temp_var->name);
                if(temp_plugin){
                    rule->plugin = temp_plugin;
                } else {
                    EEF_log(LOG_WARNING,
			    "%s: Unknown plugin in variable %s\n",
			    __func__, temp_var->value);
                }
            }
        } else {
            EEF_log(LOG_DEBUG, "%s: Plugin already linked?\n", __func__);
        }
        if (link_rule_to_plugin(policy, rule->true_branch)==-1 ||
	    link_rule_to_plugin(policy, rule->false_branch)==-1)
	    return -1;
    }
    return 0;
}

/* iterates the policies and removes those that are not explicitly named in the
 * passed array of strings */
static policy_t* reduce_policies(policy_t* policies,
				 int number_of_policies,
				 char* names_of_policies[])
{
    int i, policy_should_be_removed;
    policy_t *temp_policy = policies, *next_policy = NULL;

    while(temp_policy){
        policy_should_be_removed = 1;
        EEF_log(LOG_DEBUG, "%s: Checking policy: %s\n",
		__func__, (temp_policy)->name);
        for(i = 0; i < number_of_policies; i++){
            if(strcmp(temp_policy->name, names_of_policies[i]) == 0){
		/* if the policy is in the named list, we can continue to the
		 * next plugin */
                EEF_log(LOG_DEBUG, "%s: Allowed policy: %s\n",
			__func__, names_of_policies[i]);
                policy_should_be_removed = 0;
                break;
            }
        }

        next_policy = temp_policy->next;

        if(policy_should_be_removed){
            EEF_log(LOG_DEBUG, "%s: Removing not-allowed policy: %s\n",
		    __func__, temp_policy->name);
            policies = remove_policy_by_name(policies, temp_policy->name);
        }
        temp_policy = next_policy;
    }

    return policies;
}

/*! Removes a policy from the list of policies */
static policy_t* remove_policy_by_name(policy_t* policies, char* policy){
    policy_t *temp_policy = NULL, *next_policy = NULL, *previous_policy = NULL;

    for(temp_policy = policies; temp_policy != NULL; ) {
        next_policy = temp_policy->next;

        if(strcmp(temp_policy->name, policy) == 0) {
            /* target policy was found */

            /* Clean policy fields */
            if(temp_policy->rules_list_transformed_to_tree)
                clean_rules_tree(temp_policy->rules);
            else
                clean_rules_list(temp_policy->rules);

            EEF_log(LOG_DEBUG, "%s: Deleted policy: %s\n", __func__, policy);
            free(temp_policy->name);
            free(temp_policy);

            /* if the head node is deleted, overwrite it with the next node */
            if(previous_policy)
                previous_policy->next = next_policy;
            else
                policies = next_policy;
            break;
        }

        previous_policy = temp_policy;

        /* Move to next policy */
        temp_policy = next_policy;
    }

    return policies;
}

/*! Iterates the list of var_t structures and tries to free them */
static EES_RC clean_variables_list(){
    data.current_variable = data.variables_list;

    while(data.current_variable){
        free(data.current_variable->name);
        data.current_variable->name = NULL;
        free(data.current_variable->value);
        data.current_variable->value = NULL;

        /* Move to next variable */
        data.variables_list_last = data.current_variable;
        data.current_variable = data.current_variable->next;

        /* Clean last variable struct */
        free(data.variables_list_last);
        data.variables_list_last = NULL;
    } 
    return EES_SUCCESS;
}

/*! Iterates the list of policy_t structures and tries to free them and their
 * rules */
static EES_RC clean_policies_list(policy_t* policies){
    policy_t *temp_policy = policies, *last_policy = NULL;
    while(temp_policy){
        /* Clean policy fields */
        if(temp_policy->rules_list_transformed_to_tree){
            clean_rules_tree(temp_policy->rules);
        } else {
            clean_rules_list(temp_policy->rules);
        }
        free(temp_policy->name);
        temp_policy->name = NULL;

        /* Move to next policy */
        last_policy = temp_policy;
        temp_policy = temp_policy->next;
        free(last_policy);
        last_policy = NULL;
    }
    return EES_SUCCESS;
}

/*! Iterates the list of rule_t structures starting with the passed rule and
 * tries to free them and their true/false branches */
static rule_t* clean_rules_list(rule_t* top_rule){
    rule_t *temp_rule = top_rule, *next_rule = NULL;

    while(temp_rule){
        next_rule = temp_rule->next;
        clean_rules_tree(temp_rule);
        temp_rule = next_rule;
    } 
    return NULL;
}

/*! Iterates the tree of rule_t structures starting with the passed rule and
 * tries to free them and their true/false branches */
static rule_t* clean_rules_tree(rule_t* top_rule){
    rule_t* temp_rule = top_rule;

    if(temp_rule){
        clean_rules_tree(temp_rule->true_branch);

        clean_rules_tree(temp_rule->false_branch);

        free(temp_rule->name);
        free(temp_rule);
        temp_rule = NULL;
    } 
    return NULL;
}

/*! get variable from list */
static var_t* get_variable_by_name(char* name){
    for(data.current_variable = data.variables_list;
	data.current_variable != NULL;
	data.current_variable = data.current_variable->next)
    {
        if(!strncmp(name,
		    data.current_variable->name,
		    strlen(data.current_variable->name)))
            return data.current_variable;
    }
    return NULL;
}

/*! converts a string to a NULL-terminated array of strings by splitting it at
 * each delimiter. */
static char** _var_to_argv(const char* value, const char *delim){
    char *copy_of_value, *next_arg, *str_ptr;
    char **argv;
    int entries, i;

    if (!value)
	return NULL;

    /* Each element is at least one character: so half of the value, rounded up,
     * plus one extra NULL string at the end, should be minimum length.
     * Use calloc for argv to make them NULL-terminated */
    if ( !(copy_of_value = strdup(value)) ||
	 !(argv = calloc(1, (sizeof(char*)*(1+(1+strlen(value))/2)) )) )    {
	EEF_log(LOG_ERR, "%s: out of memory\n", __func__);
	free(copy_of_value);
	return NULL;
    }

    /* First strtok_r must have input, next one NULL */
    entries=0;
    next_arg = strtok_r(copy_of_value, delim, &str_ptr);
    while(next_arg != NULL){
	if (!(argv[entries] = strdup(next_arg)))	{
	    EEF_log(LOG_ERR, "%s: out of memory at entry %d\n",
		    __func__, entries);
	    /* cleanup */
	    for (i=0; i<entries; i++) free(argv[entries]);
	    free(argv); argv=NULL;
	    entries=0;
	    break;
	}
	entries++;
	next_arg = strtok_r(NULL, delim, &str_ptr);
    }
    free(copy_of_value);

    return argv;
}

/**
 * Frees memory in the policy names array
 */
static void clean_policy_name_list(void)   {
    size_t i;

    if (config.policy_names)	{
	for (i=0; config.policy_names[i]; i++)
	    free(config.policy_names[i]);
	free(config.policy_names);
	config.policy_names=NULL;
    }
}

/* prints the list of policies */
static void print_policies(policy_t* policies){
    policy_t* temp_policy = policies;

    while(temp_policy){
        EEF_log(LOG_DEBUG, "Policy: %s\n", temp_policy->name);
        print_rules(temp_policy, temp_policy->rules);
        temp_policy = temp_policy->next;
    }
}


/* prints the list of rules */
static void print_rules(policy_t* policy, rule_t* rule){
    if(rule){
        EEF_log(LOG_DEBUG, "  | Rule %s\n", rule->name);
        EEF_log(LOG_DEBUG, "  -------------------------\n");
        if(rule->true_branch){
	    EEF_log(LOG_DEBUG, "    -> True branch: %s\n",
		    rule->true_branch->name);
            print_rules(policy, get_left_hand_rule(policy->rules,
						   rule->true_branch->name));
        }
        if(rule->false_branch){
	    EEF_log(LOG_DEBUG, "    | False branch: %s\n",
		    rule->true_branch->name);
            print_rules(policy, get_left_hand_rule(policy->rules,
						   rule->false_branch->name));
        }
    } 
}

/**
 * Initializes the config structure from the init_config structure, making sure
 * we can cleanup the config structure.
 */
static EES_RC initialize_config(void) {
    /* first initialize all normal fields from init_config struct */
    config.config_file_fp=NULL;
    config.port=init_config.port;
    config.send_timeout=init_config.send_timeout;
    config.receive_timeout=init_config.receive_timeout;
    config.accept_timeout=init_config.accept_timeout;
    config.log_level=init_config.log_level;

    /* need to strdup the different (char *) separately */

    /* Check config file, may not be NULL */
    if (init_config.config_file_s == NULL)  {
	EEF_log(LOG_ERR, "%s: No config file has been configured\n", __func__);
	return EES_FAILURE;
    }

    /* Policy names */
    if (init_config.policy_string)  {
	if (config.policy_names)
	    clean_policy_name_list();
	config.policy_names=_var_to_argv(init_config.policy_string, ",");
    }

    /* Modules directory */
    if (init_config.modules_path)   {
	if (config.modules_path)
	    free(config.modules_path);
	config.modules_path=parse_modules_path(init_config.modules_path);
    }

    /* log_facility */
    if (init_config.log_facility) {
	if (config.log_facility)
	    free(config.log_facility);
	if ( (config.log_facility=strdup(init_config.log_facility) )==NULL) {
	    EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return EES_FAILURE;
	}
	/* Try setting it now */
	if (EEF_set_log_facility(init_config.log_facility)!=EES_SUCCESS)    {
	    EEF_log(LOG_WARNING,"%s: Cannot set log facility to %s\n",
		    __func__, init_config.log_facility);
	    free(config.log_facility);
	    config.log_facility=NULL;
	}
    }

    /* log_file */
    if (init_config.log_file) {
	if (config.log_file)
	    free(config.log_file);
	if ( (config.log_file=strdup(init_config.log_file) )==NULL) {
	    EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return EES_FAILURE;
	}
	/* Only now open it, note that the log file in any case is not free-d */
	if (EEF_set_log_file(config.log_file)!=EES_SUCCESS) {
	    EEF_log(LOG_WARNING, "%s: Cannot set log file to %s\n",
		    __func__, init_config.log_file);
	    free(config.log_file);
	    config.log_file=NULL;
	}
    }

    /* Also clear the internal data */
    memset(&data,0,sizeof(pdl_data_t));

    return EES_SUCCESS;
}

/**
 * Cleans up memory in the config structs, closes the config file and sets
 * values to uninitialized values
 */
static void clean_config_data(void)	{
    /* Config file */
    if (config.config_file_fp)	{
	fclose(config.config_file_fp);
	config.config_file_fp = NULL;
    }

    /* Modules dir */
    free(config.modules_path);
    config.modules_path = NULL;

    /* HTTP EIC data */
    config.port=-1;
    config.send_timeout=-1;
    config.receive_timeout=-1;
    config.accept_timeout=-1;

    /* Log settings */
    config.log_level=-1;
    free(config.log_facility);
    config.log_facility = NULL;
    free(config.log_file);
    config.log_file = NULL;
}

