/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	600

/* On Solaris need netdb.h with __EXTENSIONS__ for definition of MAXHOSTNAMELEN
 * */
#ifdef __sun
# define __EXTENSIONS__
#endif

#define _MULTI_THREADED

#include "ees_config.h"

#include <xacml_server.h>
#include <pthread.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include <limits.h>

#include "eef/eef_log.h"

#include "eics/eics_common.h"

#include "eics/eics_http.h"


/************************************************************************
 * Defines
 ************************************************************************/

/* Solaris does not define HOST_NAME_MAX but MAXHOSTNAMELEN in netdb.h */
#ifdef __sun
# ifndef HOST_NAME_MAX
#  define HOST_NAME_MAX	MAXHOSTNAMELEN
# endif
#endif

/* BSDs also don't define HOST_NAME_MAX, fall back on _POSIX_HOST_NAME_MAX */
#ifndef HOST_NAME_MAX
# define HOST_NAME_MAX	_POSIX_HOST_NAME_MAX
#endif



#define BACKLOG 100

#define	IP_MAX_CHARS			64
#define	PORT_MAX_DIGITS			8
#define	PREFIX_V4_MAPPED		"::ffff:"

/* Max length for issuer: http://<host>:<port>/ + \0 */
#define ISSUER_MAX			(HOST_NAME_MAX+14+1)

#define MAX_ERR_BUF			256


/************************************************************************
 * Global variables
 ************************************************************************/

static xacml_server_t server=NULL;		/* xacml server object */

static unsigned int running_threads=0;
/*static pthread_mutex_t stopping;*/
static int running=0;
static pthread_mutex_t thread_count=PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t idle;

static int _EES_port=-1;    /*! EES listening port */

static char **_policy_names=NULL;

#if ISSUER_MAX<4
#error ISSUER_MAX is too small to even contain "EES"
#endif
static char issuer[ISSUER_MAX]="EES";


/************************************************************************
 * Private prototypes
 ************************************************************************/

/* callback for libxacml - thread body */
static int ees_xacml_authorize(void *handler_arg,
			       const xacml_request_t request,
			       xacml_response_t response);

/* Creates listening socket, code similar to SCAS */
static int createAndSetUpATCPServerSocket(int port, int max_connections);

/* Determines the issuer for the xacml response */
static int set_xacml_response_issuer(void);

/************************************************************************
 * Public functions
 ************************************************************************/

/**
 * Creates and starts xacml server, port should have been set previously using
 * set_xacml_http_port()
 */
EES_RC start_xacml_http_server(void){
    int fd;
    char errbuf[MAX_ERR_BUF];

    /*shutdown(fd, 0);*/
    if(_EES_port == -1) {
	EEF_log(LOG_ERR, "%s: Port is not set or invalid.\n", __func__);
	return EES_FAILURE;
    }

    /* Determine the hostname, to be used as issuer. Easiest to do now, since
     * we're still single-threaded */
    set_xacml_response_issuer();

    EEF_log(LOG_DEBUG, "%s: Creating listening socket on port %d\n",
	    __func__, _EES_port);
    if ((fd = createAndSetUpATCPServerSocket(_EES_port, BACKLOG)) > -1){
	/* if you look at the code, you'll see this function is empty */
        xacml_init();
	EEF_log(LOG_DEBUG, "Initializing XACML server\n");
        if (xacml_server_init(&server, ees_xacml_authorize, NULL)
		== XACML_RESULT_SUCCESS)
	{
	    EEF_log(LOG_INFO, "Starting XACML server\n");
            xacml_server_set_fd(server, fd);
            xacml_server_start(server);
            return EES_SUCCESS;
        }
	EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	EEF_log(LOG_ERR,"%s: xacml_server_init() failed: %s\n",__func__,errbuf);
    }
    /* Note that failure in createAndSetUpATCPServerSocket() has been handled
     * there */

    return EES_FAILURE;
}

void init_xacml_http_threading(void)	{
    pthread_mutex_lock(&thread_count);
    running_threads = 0;
    running = 1;
    pthread_mutex_unlock(&thread_count);
}


/**
 * Waits for running thread, then stops the xacml HTTP server
 */
EES_RC stop_xacml_http_server(void){
    int rc;
	 
    /* Wait until the service is idle */
    pthread_mutex_lock(&thread_count);
    running = 0;
    if(running_threads > 0){
        EEF_log(LOG_DEBUG, "%s: Waiting for %i running threads\n",
		__func__, running_threads);
        rc = pthread_cond_wait(&idle, &thread_count);
    } else
	rc = 0;
    pthread_mutex_unlock(&thread_count);

    /* Stop xacml server */
    if (server)	{
	xacml_server_destroy(server);
	EEF_log(LOG_INFO, "Stopped XACML server\n");
    }
   
    if (rc!=0)
	return EES_FAILURE;

    return EES_SUCCESS;
}


/**
 * Sets the EES HTTP listening port
 */
EES_RC set_xacml_http_port(int port){
    if (port<0 || port>65535)    {
	EEF_log(LOG_ERR, "%s: Invalid port number %d\n",
		__func__, port);
	return EES_FAILURE;
    }
    _EES_port = port;
    return EES_SUCCESS;
}

/**
 * Sets the EES HTTP server send, receive and accept timeouts for values != -1
 */
EES_RC set_xacml_http_timeouts(int send_timeout,
			       int recv_timeout,
			       int acpt_timeout)	{

    /* Anything to do ? */
    if (send_timeout==-1 && recv_timeout==-1 && acpt_timeout==-1)
	return EES_SUCCESS;

#ifdef HAVE_XACML_SERVER_SET_TIMEOUTS
    /* Valid input ? */
    if ( (send_timeout<0 && send_timeout!=-1) ||
	 (recv_timeout<0 && recv_timeout!=-1) ||
	 (acpt_timeout<0 && acpt_timeout!=-1) ) {
	EEF_log(LOG_ERR, "%s: Invalid input (%d, %d, %d)\n",
		__func__, send_timeout,recv_timeout,acpt_timeout);
	return EES_FAILURE;
    }

    /* Set values */
    EEF_log(LOG_DEBUG, "%s: setting timeouts to (%d, %d, %d)\n",
	    __func__, send_timeout, recv_timeout, acpt_timeout);
    xacml_server_set_timeouts(send_timeout, recv_timeout, acpt_timeout);


    return EES_SUCCESS;
#else
    EEF_log(LOG_ERR,
	    "%s: EES is built without support for xacml_server_set_timeouts()\n",
	    __func__);
    return EES_FAILURE;
#endif
}

/**
 * Sets the EES HTTP policy(ies) to use.
 * When policy_names is NULL all policies are tried, otherwise it should be a
 * NULL-terminated array of strings.
 */
void set_xacml_http_policy_names(char *policy_names[])	{
    _policy_names=policy_names;
}


/************************************************************************
 * Private functions
 ************************************************************************/

/**
 * Callback for the xacml_server_init(), gets calls for a new request and does
 * the main processing part. The to-be-run policies need to be specified using
 * EES_set_http_policy_names().
 */
static int ees_xacml_authorize(void *handler_arg,
			       const xacml_request_t request,
			       xacml_response_t response)
{
    int rc;

    pthread_mutex_lock(&thread_count);
    if(!running)    {
	rc=-1;
        goto end;
    }
    running_threads++;
    pthread_mutex_unlock(&thread_count);

    /* Does all the handling of the request/response */
    rc = eef_xacml_authorize(request, response, _policy_names, issuer);

    pthread_mutex_lock(&thread_count);
    running_threads--;

end:
    if(running == 0 && running_threads == 0)
        pthread_cond_signal(&idle);
    pthread_mutex_unlock(&thread_count);

    return rc;
}

/*
 * Create a server socket using all available addresses of the host until one is
 * successfully created. If the socket is IPv6, the IPV6_V6ONLY socket option is
 * unset. The socket is returned, or -1 in case of failure.
 */
static int createAndSetUpATCPServerSocket(int port, int max_connections)
{
    char errbuf[MAX_ERR_BUF];
    int server_socket=-1;
    int success=0;
    struct addrinfo hints, *address_list = NULL, *paddress;
    int on = 1, off = 0, flags;
    char ip[IP_MAX_CHARS];
    char port_string[PORT_MAX_DIGITS];
    int rc;

    /* getaddrinfo parameters */
    memset(&hints, 0, sizeof(hints));	/* init */
    hints.ai_flags |= AI_PASSIVE;	/* server behaviour: accept connection
					   on any network address */
    hints.ai_family = AF_UNSPEC;	/* find both ipv4 and ipv6 addresses */
    hints.ai_socktype = SOCK_STREAM;	/* type of socket */

    rc=snprintf(port_string, PORT_MAX_DIGITS, "%d", port);
    if (rc<0)   {
	EEF_log(LOG_ERR, "%s: Error converting port %d to string\n",
		__func__, port);
	return -1;
    }
    if (rc>=PORT_MAX_DIGITS)    {
	EEF_log(LOG_ERR, "%s: Error: port %d does not fit in char[%d]\n",
		__func__ ,port, PORT_MAX_DIGITS);
	return -1;
    }

    rc = getaddrinfo(NULL, port_string, &hints, &address_list);
    if (rc != 0)
    {
        EEF_log(LOG_ERR,
		"%s: Error: Failed to getaddrinfo(NULL,%s, *, *): %s\n",
		__func__,port_string,
		(rc==EAI_SYSTEM ? strerror(errno): gai_strerror(rc)) );
	return -1;
    }

    for (paddress = address_list; paddress; paddress = paddress->ai_next)   {
	if (server_socket!=-1)	{
	    close(server_socket);
	    server_socket = -1;
	}

	/* create the socket */
	server_socket = socket(paddress->ai_family,
			       paddress->ai_socktype,
			       paddress->ai_protocol);
	if (server_socket == -1)    {
	    EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	    EEF_log(LOG_WARNING,"%s: Cannot create socket: %s\n",
		    __func__, errbuf);
	    continue; /* try next address */
	}

	/* set SO_REUSEADDR option */
	if (setsockopt(server_socket,
		       SOL_SOCKET, SO_REUSEADDR, &on,
		       (socklen_t)sizeof(int))==-1)
	{
	    EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	    EEF_log(LOG_WARNING,"%s: Cannot set SO_REUSEADDR: %s\n",
		    __func__, errbuf);
	    continue;
	}

	/* we must unset IPV6_V6ONLY in order to allow IPv4 clients to connect
	 * to the IPv6 socket */
	if (paddress->ai_family == AF_INET6)
	    if (setsockopt(server_socket,
			   IPPROTO_IPV6, IPV6_V6ONLY, &off,
			   (socklen_t)sizeof(int))==-1)
	    {
		EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
		EEF_log(LOG_WARNING,"%s: Cannot unset IPV6_V6ONLY: %s\n",
			__func__, errbuf);
		continue;
	    }

	/* Make the master socket non-blocking */
	if ( (flags =fcntl(server_socket, F_GETFL))==-1 ||
	     (fcntl (server_socket, F_SETFL, flags|O_NONBLOCK) == -1) ) {
	    EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	    EEF_log(LOG_WARNING,
		    "%s: fcntl() failed for setting O_NONBLOCK: %s\n",
		    __func__, errbuf);
	    continue;
	}

	/* bind and listen */
	if ( (bind(server_socket,
		   paddress->ai_addr, paddress->ai_addrlen) == -1) ||
	     (listen(server_socket, max_connections) == -1)) {
	    EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
	    EEF_log(LOG_WARNING,"%s: Cannot bind() or listen(): %s\n",
		    __func__, errbuf);
	    continue; /* try next address */
	}

	/* If we get here the bind() and listen() are successful */

	/* get the ip string, display it with the port number  */
	rc=getnameinfo(paddress->ai_addr,
		           (socklen_t)sizeof(struct sockaddr_storage),
			   ip, IP_MAX_CHARS,
			   NULL, 0, NI_NUMERICHOST);
	if (rc==0) {
	    /* if ipv4-mapped address, we remove the mapping notation in order
	     * to get only the IPv4 address */
	    if (strncasecmp(ip, PREFIX_V4_MAPPED, strlen(PREFIX_V4_MAPPED))==0)
		memmove(ip, ip + strlen(PREFIX_V4_MAPPED),
			strlen(ip) - strlen(PREFIX_V4_MAPPED) +1);
	    EEF_log(LOG_DEBUG,
		    "Server socket now listening on %s port %d\n", ip, port);
	} else {
	    if (rc==EAI_SYSTEM) {
		EEF_strerror_r(errno, errbuf, MAX_ERR_BUF);
		EEF_log(LOG_WARNING,"%s: Cannot get nameinfo for socket: %s\n",
			__func__, errbuf);
	    } else
		EEF_log(LOG_WARNING,"%s: Cannot get nameinfo for socket: %s\n",
			__func__, gai_strerror(rc));
	    EEF_log(LOG_DEBUG,
		    "Server socket now listening on port %d\n", port);
	}

	/* Make sure to set success flag */
	success=1;

	/* break if no error */
	break;
    }

    /* Close socket if unsuccesful */
    if (!success && server_socket!=-1)	{
	EEF_log(LOG_ERR, "%s: Failed to set-up listening socket\n", __func__);
	close(server_socket);
	server_socket = -1;
    }

    /* free the list */
    freeaddrinfo(address_list);

    /* return the server socket */
    return server_socket;
}

/**
 * Determines the issuer, of the form http://<full canonical hostname>:<port>/
 * When the (first) full canonical name cannot be determined we use the short
 * name, when that also fails, we set the complete issuer to "EES" and -1 is
 * returned. This function is not thread-safe.
 */
static int set_xacml_response_issuer(void) {
    char tmp_hostname[HOST_NAME_MAX+1], *host;
    struct addrinfo hints, *info;
    int retval;
    int rc=0;	/* return code of this function */

    /* First get (short) hostname */
    if (gethostname(tmp_hostname, HOST_NAME_MAX) != 0)	{
	EEF_log(LOG_WARNING,
		"%s: Cannot determine hostname, "
		"will use \"EES\" as issuer: %s\n",
		__func__, strerror(errno));
	return -1;
    }
    
    /* Set up hints struct for getting host info */
    memset(&hints, 0, sizeof hints);
    hints.ai_family = AF_UNSPEC;	/* either IPV4 or IPV6 */
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_CANONNAME;

    /* Get info and check it's valid */
    if ( (retval = getaddrinfo(tmp_hostname, NULL, &hints, &info)) != 0 ||
	 info==NULL || info->ai_canonname==NULL ||
	 strlen(info->ai_canonname)>HOST_NAME_MAX ) {
	EEF_log(LOG_WARNING,
		"%s: Getting valid hostname failed, will just use shortname: %s\n",
		__func__,
		retval==0 ? "no (valid) info returned" :
		    (retval==EAI_SYSTEM ? strerror(errno) :
					  gai_strerror(retval)));
	host=tmp_hostname;
    } else  {
	/* Found at least one valid entry */
	if (info->ai_next && info->ai_next->ai_canonname)
	    EEF_log(LOG_WARNING, "%s: Multiple hostnames found, using first\n",
		    __func__);
	host=info->ai_canonname;
    }

    /* Now try to write the issuer */
    retval = snprintf(issuer, ISSUER_MAX, "http://%s:%d/", host, _EES_port);
    if (retval<0 || retval>=ISSUER_MAX) {
	EEF_log(LOG_WARNING,
		"%s: Cannot set issuer, using \"EES\" instead: %s\n",
		__func__, retval<0 ? strerror(errno) : "hostname too long");
	/* Revert to EES */
	strcpy(issuer,"EES");
	rc=-1;
    }

    /* Free address info */
    freeaddrinfo(info);

    /* log on debug */
    EEF_log(LOG_DEBUG, "%s: will use \"%s\" as issuer string\n",
	    __func__, issuer);

    return rc;
}
