/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/* Define also _XOPEN_SOURCE for gmtime_r even though we already will define
 * _GNU_SOURCE, this makes Solaris happier */
#define _XOPEN_SOURCE   500

#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif

#include "ees_config.h"

#include <string.h>
#include <unistd.h>
#include <sys/syscall.h>
#include <stdio.h>
#include <time.h>

#include <xacml.h>

#include "eics/eics_common.h"

#include "eef/eef_return_codes.h"
#include "eef/eef_log.h"
#include "eef/eef_polytypes.h"
#include "eef/eef_library.h"
#include "eef/eef_aos.h"


#define	TIMESTRINGLEN	21L

/************************************************************************
 * Private prototypes
 ************************************************************************/

#ifdef HAVE_XACML_REQUEST_GET_ID
/* Converts a time_t to a char* */
static int time_t_to_gmtime_str(const time_t t, char *buf, size_t buf_len);
#endif


/************************************************************************
 * Public functions
 ************************************************************************/

/**
 * Does all the hard work: store the request, run the EEF for given policy names
 * and construct the response. policy_names should be either NULL (all policies)
 * or a NULL-terminated array of policy names
 */
int eef_xacml_authorize(const xacml_request_t request,
			xacml_response_t response,
			char *policy_names[],
			const char *resp_issuer)
{
#if defined(SYS_gettid)
    long thread=(long)syscall(SYS_gettid);
#elif defined(SYS_lwp_self)
    long thread=(long)syscall(SYS_lwp_self);
#else
    long thread=0L;
#endif

#ifdef HAVE_XACML_REQUEST_GET_ID
    const char *req_id=NULL, *resp_id=NULL, *issuer=NULL;
    time_t issue_time;
    char issue_time_str[TIMESTRINGLEN] = "";

    /* Get request ID, issuer and time */
    xacml_request_get_id(request,&req_id);
    xacml_request_get_issuer(request,&issuer);
    xacml_request_get_issue_instant(request,&issue_time);
    if (issue_time>0)
	time_t_to_gmtime_str(issue_time, issue_time_str, TIMESTRINGLEN);

    EEF_log(LOG_NOTICE,
	    "%s: Thread: %ld, incoming request: %s (issuer %s, issue-time %s)\n",
	    __func__, thread,
	    (req_id ? req_id : "unknown"), (issuer ? issuer : "unknown"),
	    (strlen(issue_time_str) ? issue_time_str : "unknown"));
#else
    EEF_log(LOG_NOTICE, "%s: Thread: %ld, incoming request\n",
	    __func__, thread);
#endif /* HAVE_XACML_REQUEST_GET_ID */

    /* Store the request in the AOS */
    extractRequestSubjectToAOS(request);
    extractRequestActionToAOS(request);
    extractRequestResourceToAOS(request);
    extractRequestEnvironmentToAOS(request);

#ifdef HAVE_XACML_REQUEST_GET_ID
    /* Create and set the ID for the response */
    resp_id=xacml_response_set_id(response, NULL);
#endif /* HAVE_XACML_REQUEST_GET_ID */
    /* Set issuer in the response */
    xacml_response_set_issuer(response, resp_issuer);

    /* Do the actual running of the plugins */
    if(EEF_run(policy_names) == EES_FAILURE) { 
	constructFailureResponse(&response);
#ifdef HAVE_XACML_REQUEST_GET_ID
	EEF_log(LOG_ERR,
		"%s: EEF failed in thread %ld, "
		"failure response %s to request %s constructed\n",
		__func__, thread,
		(resp_id ? resp_id : "unknown"), (req_id ? req_id : "unknown"));
#else
	EEF_log(LOG_ERR,
		"%s: EEF failed in thread %ld, failure response constructed\n",
		__func__, thread);
#endif /* HAVE_XACML_REQUEST_GET_ID */

	return 0;
    }

    /* Dump AOS to LOG_DEBUG */
    AOS_dump(LOG_DEBUG);

    /* Construct positive response from the current state */
    constructResponse(&response);
#ifdef HAVE_XACML_REQUEST_GET_ID
    EEF_log(LOG_NOTICE,
	    "%s: EEF succeeded in thread %ld, "
	    "response %s to request %s constructed\n",
	    __func__, thread,
	    (resp_id ? resp_id : "unknown"), (req_id ? req_id : "unknown"));
#else
    EEF_log(LOG_NOTICE,
	    "%s: EEF succeeded in thread %ld, response constructed\n",
	    __func__, thread);
#endif /* HAVE_XACML_REQUEST_GET_ID */

    return 0;
}


/**
 * Function:    extractRequestSubjectToAOS
 * Parameters:  const xacml_request_t           request
 * Description:
 *		This function extracts the Subject information details
 *		from the request and puts that into the EEF AOS.
 * Return:
 *	    0 : good
 *	   !0 : bad 
 */
int extractRequestSubjectToAOS(const xacml_request_t request)
{
    xacml_result_t rc;
    int            new = 0;
    size_t         count = 0;
    size_t         i = 0;

    const char *category;
    const char *attribute_id;
    const char *data_type;
    const char *issuer;
    const char *value;
        
    aos_context_t*   aos_context = NULL;
    aos_attribute_t* aos_attribute = NULL;

    /* Get number of subject attributes in request */
    rc = xacml_request_get_subject_attribute_count(request, &count);

    EEF_log(LOG_DEBUG, "%s: Found %lu subject attribute(s) in request\n",
	    __func__, (long unsigned)count);

    if (count==0)
	return (rc==XACML_RESULT_SUCCESS ? 0 : -1);

    /* Rewind the SUBJECT list */
    if ( rewindContexts(SUBJECT) == EES_FAILURE)
	return 1;

    /* Try reusing an already existing SUBJECT context */
    if ( (aos_context = getNextContext(SUBJECT)) == NULL)	{
	/* Create a new one */
	if ( (aos_context = createContext(SUBJECT)) == NULL )	{
	    EEF_log(LOG_ERR, "%s: Cannot create subject context\n", __func__);
	    return 1;
	}
	new=1;
    }
    /* aos_context should be set now */

    /* Loop over found attributes */
    for (i = 0; i < count; i++) {
        category     = NULL;
        attribute_id = NULL;
        data_type    = NULL;
        issuer       = NULL;
        value        = NULL;

        rc = xacml_request_get_subject_attribute(
                request, i,
		&category,
		&attribute_id,
		&data_type,
		&issuer,
		&value);

	/* Create, set and add attribute into AOS subject context */
	if ( (aos_attribute = createAttribute()) == NULL)    {
	    EEF_log(LOG_ERR,
		    "%s: Cannot create attribute for subject\n", __func__);
	    return 1;
	}
	if (setAttributeId(aos_attribute, attribute_id)==EES_FAILURE ||
	    setAttributeType(aos_attribute, data_type)==EES_FAILURE ||
	    setAttributeIssuer(aos_attribute, issuer)==EES_FAILURE ||
	    setAttributeValue(aos_attribute,
			      value,
			      strlen(value) + 1)==EES_FAILURE) {
	    EEF_log(LOG_ERR,
		    "%s: Cannot set attributes for subject\n", __func__);
	    return 1;
	}
	/* Add attribute to AOS subject context */
	if (addAttribute(aos_context, aos_attribute)==EES_FAILURE) {
	    EEF_log(LOG_ERR,
		    "%s: Cannot add attributes to subject\n", __func__);
	    return 1;
	}
	EEF_log(LOG_DEBUG, "%s: Added subject '%s'='%s'\n",
		__func__, attribute_id, value);
    }

    /* Add AOS subject context itself */
    if (new)	{
	if (addContext(aos_context)==EES_FAILURE) {
	    EEF_log(LOG_ERR,"%s: Cannot add subject context\n", __func__);
	    return 1;
	}
    }

    return (rc==XACML_RESULT_SUCCESS ? 0 : -1);
}


/**
 * Function:    extractRequestActionToAOS
 * Parameters:  const xacml_request_t           request
 * Description:
 *              This function extracts the Action information details
 *              from the request and puts that into the EEF AOS.
 * Return:
 *           0 : good
 *          !0 : bad 
 */
int extractRequestActionToAOS(const xacml_request_t request)
{
    xacml_result_t rc;
    int            new = 0;
    size_t         count = 0;
    size_t         i;

    const char *attribute_id;
    const char *data_type;
    const char *issuer;
    const char *value;

    aos_context_t*   aos_context = NULL;
    aos_attribute_t* aos_attribute = NULL;

    /* Get number of action attributes in request */
    rc = xacml_request_get_action_attribute_count(request, &count);

    EEF_log(LOG_DEBUG, "%s: Found %lu action attribute(s) in request\n",
	    __func__, (long unsigned)count);

    if (count==0)
	return (rc==XACML_RESULT_SUCCESS ? 0 : -1);

    /* Rewind the ACTION list */
    if ( rewindContexts(ACTION) == EES_FAILURE)
	return 1;

    /* Try reusing an already existing ACTION context */
    if ( (aos_context = getNextContext(ACTION)) == NULL)	{
	/* Create a new one */
	if ( (aos_context = createContext(ACTION)) == NULL )	{
	    EEF_log(LOG_ERR, "%s: Cannot create action context\n",__func__);
	    return 1;
	}
	new=1;
    }
    /* aos_context should be set now */

    /* Loop over found attributes */
    for (i = 0; i < count; i++) {
        attribute_id = NULL;
        data_type    = NULL;
        issuer       = NULL;
        value        = NULL;

        rc = xacml_request_get_action_attribute(
                request, i,
                &attribute_id,
                &data_type,
                &issuer,
                &value);
        
	/* Create, set and add attribute into AOS action context */
	if ( (aos_attribute = createAttribute()) == NULL)   {
	    EEF_log(LOG_ERR,"%s: Cannot create attribute for action\n",
		    __func__);
	    return 1;
	}
	if (setAttributeId(aos_attribute, attribute_id)==EES_FAILURE ||
	    setAttributeType(aos_attribute, data_type)==EES_FAILURE ||
	    setAttributeIssuer(aos_attribute, issuer)==EES_FAILURE ||
	    setAttributeValue(aos_attribute,
			      value,
			      strlen(value) + 1)==EES_FAILURE)	{
	    EEF_log(LOG_ERR,"%s: Cannot set attributes for action\n",__func__);
	    return 1;
	}
	/* Add attribute to AOS action context */
	if (addAttribute(aos_context, aos_attribute)==EES_FAILURE)  {
	    EEF_log(LOG_ERR,"%s: Cannot add attributes to action\n",__func__);
	    return 1;
	}
	EEF_log(LOG_DEBUG, "%s: Added action '%s'='%s'\n",
		__func__, attribute_id, value);
    }

    /* Add AOS action context itself */
    if (new)	{
	if (addContext(aos_context)==EES_FAILURE) {
	    EEF_log(LOG_ERR,"%s: Cannot add action context\n",__func__);
	    return 1;
	}
    }

    return (rc==XACML_RESULT_SUCCESS ? 0 : -1);
}


/**
 * Function:    extractRequestResourceToAOS
 * Parameters:  const xacml_request_t           request
 * Description:
 *              This function extracts the Resource information details
 *              from the request and puts that into the EEF AOS
 * Return:
 *           0 : good
 *          !0 : bad 
 */
int extractRequestResourceToAOS(const xacml_request_t request)
{
    xacml_result_t rc;
    size_t         resource_attr_count = 0;
    size_t         count;
    size_t         i;
    size_t         j;

    xacml_resource_attribute_t resource_attribute_set  = NULL;

    const char *attribute_id;
    const char *data_type;
    const char *issuer;
    const char *value;

    aos_context_t*   aos_context = NULL;
    aos_attribute_t* aos_attribute = NULL;

    /* Get number of resource attributes in request */
    rc = xacml_request_get_resource_attribute_count(request,
						    &resource_attr_count);

    EEF_log(LOG_DEBUG, "%s: Found %lu resource attribute set(s) in request\n",
	    __func__, (long unsigned)resource_attr_count);

    /* Loop over found attributes */
    for (i = 0 ; i < resource_attr_count; i++)
    {
	/* Get resource attribute in request */
        rc = xacml_request_get_resource_attribute(
                request,
                i,
                &resource_attribute_set);

        if (resource_attribute_set)
        {
	    count = 0;
	    /* Get number of resource attributes in attribute */
            rc = xacml_resource_attribute_get_count(
                       resource_attribute_set,
                       &count);

	    EEF_log(LOG_DEBUG,
		    "%s: Found %lu resource attribute(s) in attribute set\n",
		    __func__, (long unsigned)count);

	    /* Create a new RESOURCE context */
	    if ( (aos_context = createContext(RESOURCE)) == NULL )  {
		EEF_log(LOG_ERR, "%s: Cannot create resource context\n",
			__func__);
		return 1;
	    }
	    /* aos_context should be set now */

	    /* Loop over found attributes */
            for (j = 0; j < count; j++)
            {
                attribute_id = NULL;
                data_type    = NULL;
                issuer       = NULL;
                value        = NULL;
                rc = xacml_resource_attribute_get_attribute(
                           resource_attribute_set, j,
                           &attribute_id,
                           &data_type,
                           &issuer,
                           &value);

		/* Create, set and add attribute into AOS resource context */
		if ( (aos_attribute = createAttribute()) == NULL)   {
		    EEF_log(LOG_ERR,
			    "%s: Cannot create attribute for resource\n",
			    __func__);
		    return 1;
		}
		if (setAttributeId(aos_attribute, attribute_id)==EES_FAILURE ||
		    setAttributeType(aos_attribute, data_type)==EES_FAILURE ||
		    setAttributeIssuer(aos_attribute, issuer)==EES_FAILURE ||
		    setAttributeValue(aos_attribute,
				      value,
				      strlen(value)+1)==EES_FAILURE)	{
		    EEF_log(LOG_ERR,"%s: Cannot set attributes for resource\n",
			    __func__);
		    return 1;
		}
		/* Add attribute to AOS resource context */
		if (addAttribute(aos_context, aos_attribute)==EES_FAILURE)  {
		    EEF_log(LOG_ERR,"%s: Cannot add attributes to resource\n",
			    __func__);
		    return 1;
		}
		aos_attribute = NULL;
		EEF_log(LOG_DEBUG, "%s: Added resource '%s'='%s'\n",
			__func__, attribute_id, value);
            }
	    /* Add AOS resource context itself */
	    if (addContext(aos_context)==EES_FAILURE) {
		EEF_log(LOG_ERR,"%s: Cannot add resource context\n",
			__func__);
		return 1;
	    }
	    aos_context = NULL;
        }
    }

    return (rc==XACML_RESULT_SUCCESS ? 0 : -1);
}


/**
 * Function:    extractRequestEnvironmentToAOS
 * Parameters:  const xacml_request_t           request
 * Description:
 *              This function extracts the Environment information details
 *              from the request and puts that into the EEF AOS.
 * Return:
 *           0 : good
 *          !0 : bad 
 */
int extractRequestEnvironmentToAOS(const xacml_request_t request)
{
    xacml_result_t rc;
    int            new = 0;
    size_t         count = 0;
    size_t         i;

    const char *attribute_id;
    const char *data_type;
    const char *issuer;
    const char *value;

    aos_context_t*   aos_context   = NULL;
    aos_attribute_t* aos_attribute = NULL;

    /* Get number of environment attributes in request */
    rc = xacml_request_get_environment_attribute_count(request, &count);

    EEF_log(LOG_DEBUG, "%s: Found %lu environment attribute(s) in request\n",
	    __func__, (long unsigned)count);

    if (count==0)
	return (rc==XACML_RESULT_SUCCESS ? 0 : -1);

    /* Rewind the ENVIRONMENT list */
    if ( rewindContexts(ENVIRONMENT) == EES_FAILURE)
	return 1;

    /* Try reusing an already existing ENVIRONMENT context */
    if ( (aos_context = getNextContext(ENVIRONMENT)) == NULL)	{
	/* Create a new one */
	if ( (aos_context = createContext(ENVIRONMENT)) == NULL )   {
	    EEF_log(LOG_ERR, "%s: Cannot create environment context\n",
		    __func__);
	    return 1;
	}
	new=1;
    }
    /* aos_context should be set now */

    /* Loop over found attributes */
    for (i = 0 ; i < count; i++) {
        attribute_id = NULL;
        data_type    = NULL;
        issuer       = NULL;
        value        = NULL;

        rc = xacml_request_get_environment_attribute(
                request, i,
                &attribute_id,
                &data_type,
                &issuer,
                &value);

	/* Create, set and add attribute into AOS action context */
	if ( (aos_attribute = createAttribute()) == NULL)   {
	    EEF_log(LOG_ERR,"%s: Cannot create attribute for environment\n",
		    __func__);
	    return 1;
	}
	if (setAttributeId(aos_attribute, attribute_id)==EES_FAILURE ||
	    setAttributeIssuer(aos_attribute, issuer)==EES_FAILURE ||
	    setAttributeType(aos_attribute, data_type)==EES_FAILURE ||
	    setAttributeValue(aos_attribute,
			      value,
			      strlen(value) + 1)==EES_FAILURE)	{
	    EEF_log(LOG_ERR,"%s: Cannot set attributes for environment\n",
		    __func__);
	    return 1;
	}
	/* Add attribute to AOS environment context */
	if (addAttribute(aos_context, aos_attribute)==EES_FAILURE) {
	    EEF_log(LOG_ERR,"%s: Cannot add attributes to environment\n",
		    __func__);
	    return 1;
	}
	EEF_log(LOG_DEBUG, "%s: Added environment '%s'='%s'\n",
		__func__, attribute_id, value);
    }

    /* Add AOS environment context itself */
    if (new)	{
	if (addContext(aos_context)==EES_FAILURE) {
	    EEF_log(LOG_ERR,"%s: Cannot add environment context\n", __func__);
	    return 1;
	}
    }

    return (rc==XACML_RESULT_SUCCESS ? 0 : -1);
}


/**
 * Function:    constructFailureResponse
 * Parameters:  xacml_response_t * response
 * Description:
 *              The XACML response message to indicate a failure is constructed
 *              here.
 */
int constructFailureResponse(xacml_response_t * response) {
    xacml_response_set_saml_status_code(*response, SAML_STATUS_AuthnFailed);
    xacml_response_set_xacml_status_code(*response, XACML_STATUS_ok);
    xacml_response_set_xacml_decision(*response, XACML_DECISION_Deny);

    return 0;
}


/**
 * Function:    constructResponse
 * Parameters:  xacml_response_t * response
 * Description:
 *              The XACML response message is constructed here.
 */
int constructResponse(xacml_response_t * response){

    /* Mapping Information Translated */
    aos_context_t*       context;
    aos_attribute_t*     attribute;
    xacml_obligation_t   tmp_obligation  = NULL;
    char*                attribute_name;
    char*                attribute_value;
    char*                obligation_id;

    /* Rewind the OBLIGATION list */
    if ( rewindContexts(OBLIGATION) == EES_FAILURE )
	return 1;

    /* Loop over all obligation contexts */
    while((context = getNextContext(OBLIGATION)) != NULL){
	rewindAttributes(context);
	obligation_id = getContextObligationId(context);
	/* Only parse non-empty obligation IDs */
	if (obligation_id)  {
	    xacml_obligation_init(&tmp_obligation,
				  obligation_id,
				  XACML_EFFECT_Permit);
	    /* Loop over attributes in found obligation */
	    while((attribute = getNextAttribute(context)) != NULL){
		attribute_name = getAttributeId(attribute);
		attribute_value = getAttributeValueAsString(attribute);
		if(attribute_name && attribute_value)
		    /* Add attribute name/value to obligation */
		    xacml_obligation_add_attribute(tmp_obligation,
						   attribute_name,
						   XACML_DATATYPE_STRING,
						   attribute_value);
	    }
	    /* Add obligation to response */
	    xacml_response_add_obligation(*response, tmp_obligation);
	    xacml_obligation_destroy(tmp_obligation);
	    tmp_obligation = NULL;
	}
    }

    xacml_response_set_saml_status_code(*response, SAML_STATUS_Success);
    xacml_response_set_xacml_status_code(*response, XACML_STATUS_ok);
    xacml_response_set_xacml_decision(*response, XACML_DECISION_Permit);

    return 0;
}


/************************************************************************
 * Private functions
 ************************************************************************/

#ifdef HAVE_XACML_REQUEST_GET_ID
/* Converts a time_t to a char* */
static int time_t_to_gmtime_str(const time_t t, char *buf, size_t buf_len)  {
    struct tm res_tm;
    int rc;

    if (gmtime_r(&t, &res_tm)==NULL)
	return -1;

    rc=snprintf(buf, buf_len, "%04d-%02d-%02dT%02d:%02d:%02dZ",
		res_tm.tm_year + 1900,
		res_tm.tm_mon + 1,
		res_tm.tm_mday,
		res_tm.tm_hour,
		res_tm.tm_min,
		res_tm.tm_sec);

    if (rc<0 || (size_t)rc>=buf_len)	{
	buf[0]='\0';
	return -1;
    }

    return 0;
}
#endif /* HAVE_XACML_REQUEST_GET_ID */
