/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	600

#include "ees_config.h"

/* Solaris does not define HOST_NAME_MAX but MAXHOSTNAMELEN in netdb.h */
#ifdef __sun
# define __EXTENSIONS__
# include <netdb.h>
#endif

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <pthread.h>

#include "eef/eef_log.h"


/* Solaris does not define HOST_NAME_MAX but MAXHOSTNAMELEN in netdb.h */
#ifdef __sun
# ifndef HOST_NAME_MAX
#  define HOST_NAME_MAX	MAXHOSTNAMELEN
# endif
#endif

/* BSDs also don't define HOST_NAME_MAX, fall back on _POSIX_HOST_NAME_MAX */
#ifndef HOST_NAME_MAX
# define HOST_NAME_MAX	_POSIX_HOST_NAME_MAX
#endif

#define MAX_TIME_STRING_SIZE 256
#define MAX_ERR_SIZE 256


/************************************************************************
 * Global variables
 ************************************************************************/

static int _log_initialized=0;		/* whether log is opened */
static int _debug_mode=0;		/* debugmode: LOG_DEBUG and stderr */
static int _log_level=LOG_INFO;		/* log_level to use */
static int _log_facility=LOG_DAEMON;	/* syslog facility to use */
static char _hostname[HOST_NAME_MAX+1];
static const char* _ident_str=PACKAGE_NAME;
static const char *_log_file_name=NULL;	/* log filename, when NULL, syslog */

static FILE *_log_file_fp=NULL;		/* log stream, when NULL, syslog */

static pthread_mutex_t _log_mutex=PTHREAD_MUTEX_INITIALIZER;

/************************************************************************
 * Public functions
 ************************************************************************/

/**
 * Main log function, thread-safe. Follows syslog() prototype. Depending on
 * which log flags are set through EEF_set_log_options and which log_level is
 * set using EEF_set_log_level, will log to syslog and/or a logfile.
 */
void EEF_log(int priority, const char* format, ...){
    va_list	    args;
    char	    *buffer=NULL;
    int		    len;
    char	    error_buf[MAX_ERR_SIZE];

    time_t	    curtime;
    struct tm	    time_s;
    char	    strf_fmt[MAX_TIME_STRING_SIZE];
    /* Number of bytes times 8*log(2)/log(10) rounded up, plus 1 extra for minus
     * and 1 extra for \0 */
    size_t const    pid_str_len=sizeof(pid_t)*3+2;
    char	    pid_str[sizeof(pid_t)*3+2];


    /* check if we are initialized */
    if (!_log_initialized)
	EEF_openlog();

    /* only log when priority is <= loglevel */
    if (priority>_log_level)
	return;

    /* Dump message in string: to stay thread-safe we want to reduce number of
     * actual file actions. */
    va_start(args, format);
    len=vsnprintf(buffer, 0, format, args);
    va_end(args);
    if (len<0)	{
	EEF_strerror_r(errno, error_buf, MAX_ERR_SIZE);
	/* Log this directly or we might create a loop */
	if (_debug_mode)    {
	    fprintf(stderr, "error in EEF_log(): %s\n", error_buf);
	    fflush(stderr);
	}
	/* lock mutex, print and unlock. Lock before if! */
	pthread_mutex_lock(&_log_mutex);
	if (_log_file_fp)   {
	    fprintf(_log_file_fp, "error in EEF_log(): %s\n", error_buf);
	    fflush(_log_file_fp);
	} else
	    syslog(LOG_ERR, "error in EEF_log(): %s\n", error_buf);
	pthread_mutex_unlock(&_log_mutex);
	/* Error */
	return;
    }

    if ( (buffer=malloc((size_t)len+1))==NULL)	{
	/* Log this directly or we might create a loop */
	if (_debug_mode)    {
	    fprintf(stderr, "Out of memory in EEF_log()\n");
	    fflush(stderr);
	}
	/* lock mutex, print and unlock. Lock before if! */
	pthread_mutex_lock(&_log_mutex);
	if (_log_file_fp)   {
	    fprintf(_log_file_fp, "Out of memory in EEF_log()\n");
	    fflush(_log_file_fp);
	} else
	    syslog(LOG_ERR, "Out of memory in EEF_log()\n");
	pthread_mutex_unlock(&_log_mutex);
	return;
    }
   
    /* Dump message */
    va_start(args, format);
    len=vsnprintf(buffer, (size_t)len+1, format, args);
    va_end(args);

    /* syslog */
    if(_log_file_fp==NULL)
	syslog(priority, "%s", buffer);

    /* file log and stderr */
    if (_log_file_fp || _debug_mode) {
	/* get time */
	if ( time(&curtime)==-1 ||
	     localtime_r(&curtime, &time_s)==NULL ||
	     strftime(strf_fmt,
		      MAX_TIME_STRING_SIZE-1, "%b %d %H:%M:%S", &time_s)==0)
	    strf_fmt[0]='\0';

	/* pid */
	if (snprintf(pid_str, pid_str_len, "%ld", (long)getpid())<=0)
	    pid_str[0]='\0';

	/* Now log */
	if (_debug_mode)    {
	    fprintf(stderr, "%s %s %s[%s]: %s",
		    strf_fmt, _hostname, _ident_str, pid_str, buffer);
	    fflush(stderr);
	}
	/* lock mutex, print and unlock. Lock before if! */
	pthread_mutex_lock(&_log_mutex);
	if (_log_file_fp) {
	    fprintf(_log_file_fp, "%s %s %s[%s]: %s",
		    strf_fmt, _hostname, _ident_str, pid_str, buffer);
	    fflush(_log_file_fp);
	}
	pthread_mutex_unlock(&_log_mutex);
    }

    /* free memory */
    free(buffer);
}

/**
 * Sets the log level
 */
EES_RC EEF_set_log_level(int level)  {
    if (level>5 || level<0)
	return EES_FAILURE;

    /* Change offset to match syslog() levels LOG_CRIT..LOG_DEBUG */
    _log_level=LOG_CRIT+level;

    return EES_SUCCESS;
}

/**
 * Returns the log level
 */
int EEF_get_log_level(void) {
    return _log_level-LOG_CRIT;
}

/**
 * Sets EEF debug mode
 */
void EEF_set_debug_mode(int mode) {
    _debug_mode=mode;
    if (mode)
	_log_level=LOG_DEBUG;
}

/**
 * Returns EEF debug mode
 */
int EEF_get_debug_mode(void) {
    return _debug_mode;
}

/**
 * Sets the logfile used for EEF_openlog(), not strdupped, so do not free
 */
EES_RC EEF_set_log_file(const char *file_name)  {
    /* Set internal value */
    _log_file_name=file_name;

    /* Also open log, will unset _log_file_name if it fails */
    return EEF_openlog();
}

/**
 * Returns the log file
 */
const char *EEF_get_log_file(void) {
    return _log_file_name;
}

/**
 * Sets and updates EEF log ident, not strdupped, so do not free
 */
void EEF_set_log_ident(const char* ident){
    _ident_str = ident;  

    /* Re-open syslog when we aren't logging to file. Note that syslog might
     * have already been opened in which case we're updating the ident */
    if (_log_initialized && !_log_file_fp)
	openlog(_ident_str, LOG_CONS | LOG_PID | LOG_NDELAY, _log_facility);

}

/**
 * Returns log ident, see openlog()
 */
const char *EEF_get_log_ident(void){
  return _ident_str;
}

/**
 * Sets and updates the logfacility (when using syslog) for given name
 * \return EES_FAILURE on error or EES_SUCCESS on success
 */
EES_RC EEF_set_log_facility(const char *fac_name)  {
    if      (strcmp(fac_name,"LOG_AUTH")==0)   _log_facility=LOG_AUTH;
    else if (strcmp(fac_name,"LOG_CRON")==0)   _log_facility=LOG_CRON;
    else if (strcmp(fac_name,"LOG_DAEMON")==0) _log_facility=LOG_DAEMON;
    else if (strcmp(fac_name,"LOG_KERN")==0)   _log_facility=LOG_KERN;
    else if (strcmp(fac_name,"LOG_LOCAL0")==0) _log_facility=LOG_LOCAL0;
    else if (strcmp(fac_name,"LOG_LOCAL1")==0) _log_facility=LOG_LOCAL1;
    else if (strcmp(fac_name,"LOG_LOCAL2")==0) _log_facility=LOG_LOCAL2;
    else if (strcmp(fac_name,"LOG_LOCAL3")==0) _log_facility=LOG_LOCAL3;
    else if (strcmp(fac_name,"LOG_LOCAL4")==0) _log_facility=LOG_LOCAL4;
    else if (strcmp(fac_name,"LOG_LOCAL5")==0) _log_facility=LOG_LOCAL5;
    else if (strcmp(fac_name,"LOG_LOCAL6")==0) _log_facility=LOG_LOCAL6;
    else if (strcmp(fac_name,"LOG_LOCAL7")==0) _log_facility=LOG_LOCAL7;
    else if (strcmp(fac_name,"LOG_LPR")==0)    _log_facility=LOG_LPR;
    else if (strcmp(fac_name,"LOG_MAIL")==0)   _log_facility=LOG_MAIL;
    else if (strcmp(fac_name,"LOG_NEWS")==0)   _log_facility=LOG_NEWS;
    else if (strcmp(fac_name,"LOG_USER")==0)   _log_facility=LOG_USER;
    else if (strcmp(fac_name,"LOG_UUCP")==0)   _log_facility=LOG_UUCP;
   
#ifdef LOG_AUDIT
    else if (strcmp(fac_name,"LOG_AUDIT")==0)  _log_facility=LOG_AUDIT;
#endif
#ifdef LOG_AUTHPRIV
    else if (strcmp(fac_name,"LOG_AUTHPRIV")==0) _log_facility=LOG_AUTHPRIV;
#endif
#ifdef LOG_FTP
    else if (strcmp(fac_name,"LOG_FTP")==0)    _log_facility=LOG_FTP;
#endif
#ifdef LOG_SYSLOG
    else if (strcmp(fac_name,"LOG_SYSLOG")==0) _log_facility=LOG_SYSLOG;
#endif
    else    {
	EEF_log(LOG_ERR,"%s: Unknown log facility %s\n", __func__, fac_name);
	return EES_FAILURE;
    }

    /* Re-open syslog when we aren't logging to file. Note that syslog might
     * have already been opened in which case we're updating the facility */
    if (_log_initialized && !_log_file_fp)
	openlog(_ident_str, LOG_CONS | LOG_PID | LOG_NDELAY, _log_facility);

    return EES_SUCCESS;
}

/**
 * Opens the logfile when previously set (and non-NULL) or syslog. This function
 * is not thread-safe.
 * Returns EES_SUCCESS on success or EES_FAILURE.
 */
EES_RC EEF_openlog(void)	{
    char error_buf[MAX_ERR_SIZE];
    int save_errno=0;
    EES_RC rc=EES_SUCCESS;

    /* We have successfully initialized when we either have a log stream, or
     * when we don't have a log-file-name and have initialized */
    if (_log_file_fp || (_log_initialized && _log_file_name==NULL) ) {
	EEF_log(LOG_DEBUG, "%s: log is already opened\n", __func__);
	return EES_SUCCESS;
    }

    /* store hostname when needed */
    if (_hostname[0]=='\0') {
	if (gethostname(_hostname, HOST_NAME_MAX)<0)
	    _hostname[0]='\0';
    }

    /* open logfile when applicable: when name is set */
    if (_log_file_name)	{
	if ( (_log_file_fp=fopen(_log_file_name, "a")) != NULL)	{
	    _log_initialized=1;
	    return EES_SUCCESS;
	}
	save_errno=errno;
    }

    /* either logfile failed or wasn't specified */
    /* Note: if ident_str==NULL, program name is used */
    openlog(_ident_str, LOG_CONS | LOG_PID | LOG_NDELAY, _log_facility);
    _log_initialized=1;

    /* Check error status */
    if (save_errno) {
	EEF_strerror_r(errno, error_buf, MAX_ERR_SIZE);
	/* Log file opening failed, we have already changed to syslog */
	EEF_log(LOG_ERR,
		"Error opening logfile %s, will use syslog: %s\n",
		_log_file_name, error_buf);
	/* protect for a next time */
	_log_file_name=NULL;
	rc=EES_FAILURE;
    }

    return rc;
}


/**
 * Reopens log file, logs whether it succeeded. Thread-safe.
 */
EES_RC EEF_reopenlog(void) {
    FILE *f_tmp;
    char error_buf[MAX_ERR_SIZE];
    EES_RC rc=EES_SUCCESS;

    if (_log_file_name==NULL && _log_file_fp==NULL) /* syslog, nothing to do */
	return EES_SUCCESS;

    /* lock */
    pthread_mutex_lock(&_log_mutex);

    if (_log_file_fp==NULL) {
	/* No stream yet, might have changed from syslog to file */
	_log_file_fp=fopen(_log_file_name, "a");
	if ( _log_file_fp == NULL) {
	    EEF_strerror_r(errno, error_buf, MAX_ERR_SIZE);
	    /* Cannot use logfile and cannot use EEF_log (we're locked) */
	    syslog(LOG_ERR,
		    "Error reopening logfile %s, will use syslog: %s\n",
		    _log_file_name, error_buf);
	    /* protect for a next time */
	    _log_file_name=NULL;
	    rc=EES_FAILURE;
	} else
	    /* Successfully opened */
	    rc=EES_SUCCESS;
    } else {
	/* Needs reopening */
	if (_log_file_name==NULL)   {
	    /* Name is NULL: change to syslog. Set _log_file_fp to NULL before
	     * closing for threadsafety  */
	    f_tmp=_log_file_fp;
	    _log_file_fp=NULL;
	    fclose(f_tmp);
	    /* Cannot use logfile and cannot use EEF_log (we're locked) */
	    syslog(LOG_ERR,
		    "Error reopening logfile, name is NULL, will use syslog\n");
	    rc=EES_FAILURE;
	} else {
	    /* we have a _log_file_fp and a _log_file_name */
	    _log_file_fp=freopen(_log_file_name, "a", _log_file_fp);
	    if ( _log_file_fp == NULL) {
		EEF_strerror_r(errno, error_buf, MAX_ERR_SIZE);
		/* Cannot use logfile and cannot use EEF_log (we're locked) */
		syslog(LOG_ERR,
			"Error reopening logfile %s, will use syslog: %s\n",
			_log_file_name, error_buf);
		/* protect for a next time */
		_log_file_name=NULL;
		rc=EES_FAILURE;
	    } else
		/* Successfully reopened */
		rc=EES_SUCCESS;
	}
    }

    /* unlock */
    pthread_mutex_unlock(&_log_mutex);

    /* Log success */
    if (rc==EES_SUCCESS)
	EEF_log(LOG_INFO, "Reopened log file\n");

    return rc;
}

/**
 * Closes log file
 */
EES_RC EEF_closelog(void) {
    int rc=0;

    /* lock, close and unlock */
    pthread_mutex_lock(&_log_mutex);
    /* only do something when we have an open logstream */
    if (_log_file_fp)	{
	rc=fclose(_log_file_fp);
	_log_file_fp=NULL;
    }
    pthread_mutex_unlock(&_log_mutex);

    return (rc ? EES_FAILURE : EES_SUCCESS);
}

/**
 * Wrapper around strerror_r()
 */
void EEF_strerror_r(int errnum, char *buf, size_t len)	{
    int rc;
    int save_errno=errno;
#ifndef _GNU_SOURCE
    rc=strerror_r(errnum, buf, len);
#else
    char *tmpbuf=strerror_r(errnum, buf, len);
    int sn_rc;

    /* if strerror_r did not use our buf, copy it into it */
    if (tmpbuf!=buf)	{
	/* copy and check return value */
	sn_rc=snprintf(buf, len, "%s", tmpbuf);
	if (sn_rc>=len)
	    rc=ERANGE;
	else if (sn_rc<0)
	    rc=errno; /* snprintf errno */
	else
	    rc=0;
    }
#endif

    switch(rc)	{
	case 0:
	    break;
	case EINVAL:
	    snprintf(buf, len, "strerror_r: invalid errno %d", errnum);
	    break;
	case ERANGE:
	    snprintf(buf, len, "strerror_r: buffer too small %lu, errnum=%d",
		    (unsigned long)len, errnum);
	    break;
	default:
	    snprintf(buf, len, "strerror_r returned %d, errno=%d\n",
		    rc, errnum);
	    break;
    }
    
    errno=save_errno;
}
