/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#include "ees_config.h"

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "eef/eef_library.h"
#include "eef/eef_log.h"
#include "eval_manager.h"
#include "plugin_manager.h"
#include "aos_service.h"



/************************************************************************
 * Public functions
 ************************************************************************/

unsigned int EEF_getMajorVersion(void){
    unsigned int major = 0, minor = 0, patch = 0;

    if (sscanf(VERSION, "%u.%u.%u", &major, &minor, &patch) != 3) {
        EEF_log(LOG_ERR,
		"%s error: could not parse compile-time version information for EEF.\n",
		__func__);
        return 0;
    }
    return major;
}

unsigned int EEF_getMinorVersion(void){
  unsigned int major = 0, minor = 0, patch = 0;

    if (sscanf(VERSION, "%u.%u.%u", &major, &minor, &patch) != 3) {
        EEF_log(LOG_ERR,
		"%s error: could not parse compile-time version information for EEF.\n",
		__func__);
        return 0;
    }
    return minor;
}

unsigned int EEF_getPatchVersion(void){
    unsigned int major = 0, minor = 0, patch = 0;

    if (sscanf(VERSION, "%u.%u.%u", &major, &minor, &patch) != 3) {
        EEF_log(LOG_ERR,
		"%s error: could not parse compile-time version information for EEF.\n",
		__func__);
        return 0;
    }
    return patch;
}

/**
 * Tries to daemonize the process
 */
EES_RC EEF_daemonize(void){
    pid_t pid;

    EEF_log(LOG_DEBUG, "Daemonizing...\n");

    /* If launched by init there's no need to detach
      * Note: this test is unreliable due to an unavoidable race
      * condition if the process is orphaned.
      */
    if(getppid() == 1)
        goto out;

#ifdef SIGTTOU
    signal(SIGTTOU, SIG_IGN);
#endif

#ifdef SIGTTIN
    signal(SIGTTIN, SIG_IGN);
#endif

#ifdef SIGTSTP
    signal(SIGTSTP, SIG_IGN);
#endif

#ifdef BSD
    pid = getpid();
    if(setpgid(0, pid) != -1){
        if((fd = open("/dev/tty", O_RDWR)) >=0){
            ioctl(fd, TIOCNOTTY, 0);
            close(fd);
        }
    }
#else
    if (setpgrp()<0) {
	EEF_log(LOG_ERR,"%s: setpgrp() failed: %s\n",
		__func__, strerror(errno));
        return EES_FAILURE;
    }
#endif

out:
    /* Actual fork, parent will exit */
    pid=fork();

    switch(pid){
        case 0:
            break;
        case -1:
            EEF_log(LOG_ERR, "Error forking: %s\n", strerror(errno));
            exit(1);
        default:
	    EEF_log(LOG_DEBUG, "child process is %ld\n",(long)pid);
	    /* Need to clean up configuration memory */
	    stop_eval_manager();
	    /* Need to close log */
	    EEF_closelog();
            exit(0);
    }

    /* change dir to '/' */
    if(chdir("/") != 0) {
	EEF_log(LOG_ERR, "%s: Cannot chdir to /: %s\n",
		__func__, strerror(errno));
        return EES_FAILURE; 
    }

    /* redirect std{in,out,err} */
    if (freopen("/dev/null","a",stdout)==NULL ||
        freopen("/dev/null","a",stderr)==NULL ||
        freopen("/dev/null","r",stdin)==NULL)
    {
        EEF_log(LOG_ERR,"%s: Cannot reattach stdin/stderr/stdout: %s\n",
		__func__, strerror(errno));
        return EES_FAILURE;
    }
    
    /* set file creation mask to '750' */
    umask(027);

    EEF_log(LOG_DEBUG,"...successfully daemonized\n");

    return EES_SUCCESS;

}

/**
 * Try to initialize the AOS, enable logging, initialize the plug-in manager and
 * try to parse the config file and ascertain the module path
 */
EES_RC EEF_init(void){
    /* Parse the config file */
    if ( start_eval_manager(0, NULL) != EES_SUCCESS )    {
	EEF_log(LOG_ERR, "%s: Failed to parse config file\n", __func__);
	stop_eval_manager();
	return EES_FAILURE;
    }
    EEF_log(LOG_DEBUG, "%s: Config file successfully parsed\n", __func__);

    /* Start the AOS */
    if ( AOS_Init() != EES_SUCCESS )	{
	EEF_log(LOG_ERR, "%s: AOS initialization failed\n", __func__);
	EEF_term();
	return EES_FAILURE;
    }
    EEF_log(LOG_DEBUG, "%s: AOS successfully initialized\n", __func__);

    /* this is a callout to the plugin manager, which dlopen()'s the plugin,
     * dlsym()'s the plugin's functions and calls the plugin's initialize
     * function */
    if ( initialize_plugins() != EES_SUCCESS) {
	EEF_log(LOG_ERR, "%s: Failed to initialize plug-ins\n", __func__);
	EEF_term();
	return EES_FAILURE;
    }
    EEF_log(LOG_DEBUG,"%s: Plugins successfully initialized\n", __func__);
    
    return EES_SUCCESS;
}

/**
 * Runs all the plugins for all (NULL) or given policies. When non-NULL, the
 * list should be NULL-terminated.
 */
EES_RC EEF_run(char *policy_names[]){
    /* run plugins */
    return run_plugins(policy_names);
}

/**
 * Terminates the parser, stops the plug-in manager, terminates the AOS and
 * closes the log.
 */
EES_RC EEF_term(){
    EES_RC retval=EES_SUCCESS;

    EEF_log(LOG_DEBUG, "%s: Cleaning up EEF\n", __func__);

    /* Clean thread-local AOS storage */
    AOS_Clean();
    EEF_log(LOG_INFO, "%s: Cleaned AOS\n", __func__);

    /* Terminate the configuration parser */
    stop_eval_manager();
    
    /* Stop all the plugins in the plugin manager */
    if (stop_plugin_manager() == EES_SUCCESS )
	EEF_log(LOG_DEBUG, "%s: Plugin manager successfully stopped\n", __func__);
    else    {
	EEF_log(LOG_ERR, "%s: Stopping plugin manager failed\n", __func__);
	retval=EES_FAILURE;
    }

    /* Terminate the AOS */
    if ( AOS_Term() == EES_SUCCESS)
	EEF_log(LOG_DEBUG, "%s: AOS successfully terminated\n", __func__);
    else    {
	EEF_log(LOG_ERR, "%s: Stopping AOS failed\n", __func__);
	retval=EES_FAILURE;
    }

    return retval;
}

/**
 * instructs the AOS to go into threaded mode, trying to store new data in
 * thread-local storage
 */
EES_RC EEF_startThreading(void){
    AOS_start_threading();
    plugin_manager_start_threading();
    return EES_SUCCESS;
}

/**
 * Returns the name of the currently running plugin
 */
const char* EEF_getRunningPluginName(void){
    return get_running_plugin_name();
}
