/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif

#include <pthread.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/syscall.h>

#include "plugin_manager.h"
#include "aos_service.h"
#include "eef/eef_aos.h"
#include "eef/eef_log.h"
#include "eef/eef_polytypes.h"
#include "eef_plugindl_s.h"


/************************************************************************
 * Macros
 ************************************************************************/

#if defined(SYS_gettid)
# define GETTID()	    ((long)syscall(SYS_gettid))
#elif defined(SYS_lwp_self)
# define GETTID()	    ((long)syscall(SYS_lwp_self))
#else
# define GETTID()	    (0L)
#endif

/************************************************************************
 * Global variables
 ************************************************************************/

/* Thread-local storage space for AOS data */
static pthread_key_t           _aos_key;

/* Normally not okay to do initialisation outside of a function, but this way it (hopefully) will only be set one - during compile time */
static const pthread_once_t    _aos_key_once_init = PTHREAD_ONCE_INIT;
static pthread_once_t  	       _aos_key_once = PTHREAD_ONCE_INIT;

/* The backup storage storage, used to store references to aos_context_t
 * instances globally, used when no thread-local attributes were found */
static aos_storage_t *         _global_storage;

/* Flags for runtime checks */
static int                     _is_threading;


/************************************************************************
 * Private structure definitions
 ************************************************************************/
/**
 * \struct aos_attribute_s
 * A struct representing an attribute containing arbitrary data.
 */
struct aos_attribute_s {
  char *                   id;             /*! label to identify the data */
  char *                   issuer;         /*! string to identify the issuer */
  char *                   type;           /*! type identifier */
  int                      needs_free;     /*! whether the data should be free'd */
  void*                    data;           /*! pointer to store the data */
  struct aos_attribute_s * parent;         /*! pointer to the 'parent' object */
  struct aos_attribute_s * child;          /*! pointer to the 'child' object */
  struct aos_attribute_s * next;           /*! pointer to the next node */
  eef_plugindl_t *         setting_plugin; /*! pointer to the plugin that initialized this data */
};

/**
 * \struct aos_context_s
 * A container for attributes of a specific class.
 */
struct aos_context_s {
    /** an aos_context_class_e denoting the type of the context */
    enum aos_context_class_e class;
    /** used to store the obligation_id if context is of class OBLIGATION */
    char*                    obligation_name;
    /** pointer to the next context in the list */
    struct aos_context_s *   next;
    /** pointer to first of the list of attributes */
    struct aos_attribute_s*  first_attribute;
    /** pointer to the current attribute in the list */
    struct aos_attribute_s*  current_attribute;
};

/**
 * \struct aos_storage_s
 * A container for a list of contexts.
 */
struct aos_storage_s {
    /** pointer to first of the list of contexts */
    struct aos_context_s*    first_context;
    /** array of pointers to the thread-local current context of given context
     * class */
    struct aos_context_s*    current_context[CONTEXT_LAST];
    /** array of pointers to the global current context of given context class
     * */
    struct aos_context_s*    global_current_context[CONTEXT_LAST];
};


/************************************************************************
 * Private prototypes
 ************************************************************************/

static void	      aos_make_key(void);
static void	      aos_free_storage_callback(void* storage);

static aos_storage_t *aos_get_storage(void);
static aos_storage_t *aos_create_storage(void);
static EES_RC         aos_free_storage(aos_storage_t* storage);


/************************************************************************
 * Public functions
 ************************************************************************/

/* Main AOS control functions */

/**
 * Initializes the AOS and creates thread-local storage key
 */
EES_RC AOS_Init (void){
    _is_threading = 0;

    /* Need to ensure _aos_key_once == PTHREAD_ONCE_INIT, or we will not create
     * new storage upon a restart (SIGHUP), however, PTHREAD_ONCE_INIT can be a
     * something we cannot use for the assignment, use second static instead */
    _aos_key_once=_aos_key_once_init;
    pthread_once(&_aos_key_once, aos_make_key);

    /* Create new global storage */
    _global_storage=aos_create_storage();

    return  (_global_storage ? EES_SUCCESS : EES_FAILURE);
}

/**
 * Frees thread-local storage
 */
EES_RC AOS_Clean(void){
    aos_storage_t* storage;
    
    if(_is_threading){
        EEF_log(LOG_DEBUG, "%s: Killing thread %ld\n",
		__func__, GETTID());
	/* Try to get thread-local storage handle if it already exists */
	if ( (storage = pthread_getspecific(_aos_key)) != NULL )    {
	    aos_free_storage((aos_storage_t*) storage);
	    /* set it to NULL to prevent double free()s */
	    pthread_setspecific(_aos_key, NULL);
	} else
	    EEF_log(LOG_DEBUG, "%s: no thread-local storage, nothing to free\n",
		    __func__);
    } else
	EEF_log(LOG_DEBUG,
		"%s: Not threading hence not cleaning thread-local AOS\n",
		__func__);

    /* _global_storage will be freed by AOS_Term */
    return EES_SUCCESS;
}

/**
 * Terminates AOS, frees global storage and removes key
 */
EES_RC AOS_Term (void){
    _is_threading = 0;
    /* Free the global aos_storage_t */
    if (_global_storage)
	aos_free_storage(_global_storage);
    else
	EEF_log(LOG_DEBUG, "%s: no global storage, nothing to free\n",
		__func__);

    /* Remove thread-local storage key */
    pthread_setspecific(_aos_key, NULL);
    pthread_key_delete(_aos_key);

    /* AOS terminated */
    return EES_SUCCESS;
}

/**
 * sets threading flag
 */
void AOS_start_threading(void){
    _is_threading = 1;
}

/**
 * Dumps out all the data present in the AOS, as seen from the current thread's
 * perspective
 */
void AOS_dump(int loglevel) {
    /* Temporary aos_context_t and aos_attribute_t pointers */
    aos_context_t   *context;
    const char      *context_name;
    aos_attribute_t *attribute;
    /* Temporary char*s to hold names and values */
    char            *attribute_name;
    char            *attribute_value;
    long            thread_id;

    /* When specified loglevel is bigger than current loglevel, return */
    if (loglevel > LOG_CRIT+EEF_get_log_level())
	return;

    /* Get thread ID to print */
    thread_id=GETTID();

    rewindContexts(ANY);

    while((context = getNextContext(ANY)) != NULL){
	switch (context->class)	{
	    case SUBJECT:	context_name="subject";	    break;
	    case ACTION:	context_name="action";	    break;
	    case RESOURCE:	context_name="resource";    break;
	    case ENVIRONMENT:	context_name="environment"; break;
	    case NONE:		context_name="none";	    break;
	    case ANY:		context_name="any";	    break;
	    case OBLIGATION:	context_name="obligation";  break;
	    case CONTEXT_FIRST:
	    case CONTEXT_LAST:	context_name="invalid";     break;
	    default:		context_name="unknown";	    break;
	}
        EEF_log(loglevel, "AOS[%ld]: Context class: %s (address %p)\n",
		thread_id, context_name, (void*)context);
	if (context->class==OBLIGATION)
	    EEF_log(loglevel, "AOS[%ld]:  ObligationId: %s\n",
		    thread_id, context->obligation_name
			? context->obligation_name
			: "NULL");
        
	rewindAttributes(context);
        while((attribute = getNextAttribute(context)) != NULL){
            attribute_name = getAttributeId(attribute);
            if (attribute_name)	{
		attribute_value = getAttributeValueAsString(attribute);
                EEF_log(loglevel, "AOS[%ld]:    %s=%s\n",
			thread_id, 
			attribute_name,
			attribute_value ? attribute_value : "NULL");
	    }
        }
    }
}

/*
 * Context handling functions
 */

/**
 * Returns a pointer to a freshly allocated aos_context_t struct
 */
aos_context_t * createContext(aos_context_class_t context_class){
    aos_context_t * context = calloc(1, sizeof(aos_context_t));

    if (context==NULL)
	EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
    else
	/* Only class should be non-zero */
        context->class = context_class;

    return context;
}

/**
 * Adds supplied context to correct storage:
 * - threading: thread-local, since global is read-only
 * - non-threading: global
 */
EES_RC addContext(aos_context_t* new_context)  {
    aos_storage_t* storage;
    aos_context_t* context; /* Temporary pointer, to loop over the list */
    const char *storage_string;

    /* Check input */
    if (new_context==NULL)	{
	EEF_log(LOG_ERR, "%s: cannot add NULL context\n", __func__);
	return EES_FAILURE;
    }

    /* Get the correct storage - thread-local or global */
    if ( (storage=aos_get_storage()) == NULL)  {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return EES_FAILURE;
    }

    storage_string=(storage==_global_storage ? "global" : "thread-local");
    EEF_log(LOG_DEBUG, "%s: Adding context at %p to %s storage at %p\n",
	    __func__, (void*)new_context, storage_string, (void*)storage);

    /* Do we have a list? */
    if( (context=storage->first_context) == NULL){
        /* We are creating the head node */
        storage->first_context = new_context;
        EEF_log(LOG_DEBUG, "%s: Created new start of list\n", __func__);
    } else {
	/* Loop over all the contexts till the last non-NULL context or till we
	 * find a duplicate */
	while (context != new_context && context->next != NULL)
	    context = context->next;

	/* Did we end due to a duplicate? */
	if (context == new_context){
	    EEF_log(LOG_NOTICE, "%s: Skipping duplicate context\n",
		    __func__);
	    return EES_FAILURE;
	}

        /* Add the context to the end of the list */
	context->next = new_context;
	EEF_log(LOG_DEBUG, "%s: added context after %p\n", __func__,
		(void*)context);
    }
   
    /* Set the pointer used by getNextContext to the just added context */
    storage->current_context[new_context->class]=new_context;

    return EES_SUCCESS;
}

/**
 * Removes supplied context from correct storage:
 * - threading: thread-local, since global is read-only
 * - non-threading: global
 */
EES_RC destroyContext(aos_context_t* context)	{
    aos_storage_t *storage;
    aos_context_t *previous_context, *current_context;
    aos_attribute_t *attribute, *tmp_attribute;
    aos_context_class_t class;
    const char *storage_string;

    /* Check context itself */
    if (context==NULL)	{
	EEF_log(LOG_ERR, "%s: cannot destroy NULL context\n", __func__);
	return EES_FAILURE;
    }

    /* Get the correct storage - thread-local or global */
    if ( (storage=aos_get_storage()) == NULL)  {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return EES_FAILURE;
    }

    storage_string=(storage==_global_storage ? "global" : "thread-local");
    EEF_log(LOG_DEBUG, "%s: Removing context at %p from %s storage at %p\n",
	    __func__, (void*)context, storage_string, (void*)storage);

    /* Set to first context */
    current_context=storage->first_context;
    /* Keep track of previous, so that we reconnect after deleting current */
    previous_context=NULL;

    /* Find the context: can just loop over all contexts */
    while(current_context && current_context != context) {
	/* Move the pointers */
	previous_context=current_context;
	current_context =current_context->next;
    }

    /* Check we found it */
    if (current_context==NULL)  {
	EEF_log(LOG_ERR,
		"%s: Cannot find context at %p in %s storage at %p\n",
		__func__, (void*)context, storage_string, (void*)storage);
	return EES_FAILURE;
    }

    /* context found, now first clean the attributes from the found context,
     * before removing the context itself */

    /* Get the list of attributes from the aos_context_t */
    attribute = context->first_attribute;
    while(attribute)	{
	tmp_attribute=attribute; /* store for getting the next one */
	free(attribute->type);
	free(attribute->id);
	free(attribute->issuer);
	if(attribute->needs_free)
	    free(attribute->data);
	free(attribute);
	attribute=tmp_attribute->next;
    }

    /* Re-set pointers in the list and free the current one */
    if (previous_context)
	previous_context->next=current_context->next;
    else
	/* New start point: first element is removed */
	storage->first_context=current_context->next;

    /* If one of the pointers pointed to current_context, move it to the next */
    for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)	{
	if (storage->current_context[class] == current_context)
	    storage->current_context[class] = current_context->next;
    }

    /* Obligations also have a name */
    if (context->class==OBLIGATION)
	free(context->obligation_name);

    /* Now we can finally free the context itself */
    free(context);

    return EES_SUCCESS;
}

/**
 * Sets the current-context pointers in the thread-local or global storage to
 * the first in the list.
 * For the thread local storage, we also keep a thread local pointer to a member
 * of the global storage list, these are also rewound.
 * When the context class is ANY all pointers are rewound, otherwise only the
 * specified pointer is rewound.
 */
EES_RC rewindContexts(aos_context_class_t context_class)	{
    aos_storage_t * storage=aos_get_storage();
    aos_context_class_t class;

    if (storage==NULL)  {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return EES_FAILURE;
    }

    /* For class any, rewind all pointers */
    if (context_class==ANY)
	for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)
	    storage->current_context[class]=storage->first_context;
    else
	/* otherwise only rewind the correct one */
	storage->current_context[context_class]=storage->first_context;

    /* Do the same for the thread-local pointers to the _global_storage */
    if (storage!=_global_storage)   {
	if (context_class==ANY)
	    for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)
		storage->global_current_context[class] =
		    _global_storage->first_context;
	else
	    /* otherwise only rewind the correct one */
	    storage->global_current_context[context_class] =
		_global_storage->first_context;
    }

    return EES_SUCCESS;
}

/**
 * Sets only the 'global' current-context pointer of specified class in the
 * thread-local storage to the first in the list.
 * When we are not threading, this is the same as rewindContext().
 * When the context class is ANY all pointers are rewound, otherwise only the
 * specified pointer is rewound.
 */
EES_RC rewindGlobalContexts(aos_context_class_t context_class)	{
    aos_storage_t * storage=aos_get_storage();
    aos_context_class_t class;

    if (storage==NULL)  {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return EES_FAILURE;
    }

    /* For thread-local storage, rewind the global pointers, otherwise rewind
     * the 'local' pointers (which then are already the global ones) */
    if (storage==_global_storage)   {
	/* For class any, rewind all pointers */
	if (context_class==ANY)
	    for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)
		storage->current_context[class]=storage->first_context;
	else
	    /* otherwise only rewind the correct one */
	    storage->current_context[context_class]=storage->first_context;
    } else {
	if (context_class==ANY)
	    for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)
		storage->global_current_context[class] =
		    _global_storage->first_context;
	else
	    /* otherwise only rewind the correct one */
	    storage->global_current_context[context_class] =
		_global_storage->first_context;
    }

    return EES_SUCCESS;
}

/**
 * returns the next context of supplied context_class from either the
 * thread-local or global list of contexts.
 * The global list is used after we reach the end of the thread local list or
 * when we are not threading.
 */
aos_context_t * getNextContext(aos_context_class_t context_class) {
    aos_storage_t * storage=aos_get_storage();
    aos_context_t * context;

    if ( storage == NULL ) {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return NULL;
    }

    /* Start from current_context, this is true for thread-local and global */
    context=storage->current_context[context_class];

    /* Only need to loop in case of specific class, otherwise it's just the
     * first */
    if (context_class!=ANY) {
	while (context && context->class != context_class)
	    context=context->next;
    }

    /* If we found one, we're done now, update the pointers */
    if (context)    {
	/* Update thread-local pointers to local storage */
	storage->current_context[context_class]= context->next;
	if (context_class!=ANY)
	    storage->current_context[ANY]=context->next;
	return context;
    }

    /* Not present in the local storage, set the pointers to the end/NULL to
     * indicate */
    storage->current_context[context_class]=NULL;
    if (context_class!=ANY)
	storage->current_context[ANY]=NULL;

    /* If we we're using global, we're done now */
    if (storage==_global_storage)
	return NULL;

    /* Now continue with the global storage */
	
    /* Start in the global list */
    context=storage->global_current_context[context_class];

    /* Only need to loop in case of specific class, otherwise it's just the
     * first */
    if (context_class!=ANY) {
	while (context && context->class != context_class)
	    context=context->next;
    }

    /* If we found it, set the correct thread-local pointer to next element in
     * the global list */
    if (context)    {
	/* Update thread-local pointers to global storage */
	storage->global_current_context[context_class]=context->next;
	if (context_class!=ANY)
	    storage->global_current_context[ANY]=context->next;
	return context;
    }
    
    /* Not present in the local storage, set the pointers to the end/NULL to
     * indicate */
    storage->global_current_context[context_class]=NULL;
    if (context_class!=ANY)
	storage->global_current_context[ANY]=NULL;

    return NULL;
}

/**
 * returns the next context of supplied context_class from the list of contexts
 * in the global storage.
 */
aos_context_t * getNextGlobalContext(aos_context_class_t context_class) {
    aos_storage_t * storage=aos_get_storage();
    aos_context_t * context;

    if ( storage == NULL ) {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return NULL;
    }

    /* Start from 'global' current_context */
    if (storage==_global_storage)
	context=storage->current_context[context_class];
    else
	context=storage->global_current_context[context_class];
    
    /* Only need to loop in case of specific class, otherwise it's just the
     * first */
    if (context_class!=ANY) {
	while (context && context->class != context_class)
	    context=context->next;
    }

    /* If we found it, set the correct thread-local pointer to next element in
     * the thread-local or global list */
    if (context)    {
	/* Found: update pointers to the next element */
	if (storage==_global_storage)  {
	    /* Update thread-local pointers to local storage */
	    storage->current_context[context_class]= context->next;
	    if (context_class!=ANY)
		storage->current_context[ANY]=context->next;
	} else {
	    /* Update thread-local pointers to global storage */
	    storage->global_current_context[context_class]=context->next;
	    if (context_class!=ANY)
		storage->global_current_context[ANY]=context->next;
	}
    } else {
	/* Not found: update the pointers to indicate we reached the end */
	if (storage==_global_storage)  {
	    /* Update thread-local pointers to local storage */
	    storage->current_context[context_class]= NULL;
	    if (context_class!=ANY)
		storage->current_context[ANY]=NULL;
	} else {
	    /* Update thread-local pointers to global storage */
	    storage->global_current_context[context_class]=NULL;
	    if (context_class!=ANY)
		storage->global_current_context[ANY]=NULL;
	}
    }


    /* Return found context or NULL */
    return context;
}

/**
 * Convenience function to obtain an Obligation directly from obligation ID
 */
aos_context_t * getObligation(const char *obligation_id)    {
    aos_storage_t *storage;
    aos_context_t *context;

    /* Check input */
    if (obligation_id==NULL)	{
	EEF_log(LOG_ERR, "%s: invalid obligation_id\n", __func__);
	return NULL;
    }

    /* Get the correct storage - thread-local or global */
    if ( (storage=aos_get_storage()) == NULL)  {
        EEF_log(LOG_ERR, "%s: no valid storage\n", __func__);
        return NULL;
    }

    /* Get the head node of the local list of contexts*/
    context = storage->first_context;

    /* Walk the list as long as we have contexts and they are either not
     * obligation or not the right one */
    while (context && (context->class!=OBLIGATION ||
		       context->obligation_name==NULL ||
		       strcmp(context->obligation_name, obligation_id)!=0) )
	context=context->next;

    /* Continue in the global storage */
    if (!context && storage != _global_storage )   {
	/* Get the head node of the global list of contexts*/
	context = _global_storage->first_context;
    
	/* Walk the list as long as we have contexts and they are either not
	 * obligation or not the right one */
	while (context && (context->class!=OBLIGATION ||
			   context->obligation_name==NULL ||
			   strcmp(context->obligation_name, obligation_id)!=0) )
	    context=context->next;
    }

    /* return found obligation or NULL */
    return context;
}

/*
 * Getters/setters context fields: class, obligation ID
 */

/**
 * Returns the context class from the aos_context_t pointed to by context
 */
aos_context_class_t getContextClass(aos_context_t* context){
    return context->class;
}

/**
 * Sets the obligation name in the aos_context_t pointed to by context to
 * obligation
 */
EES_RC setContextObligationId(aos_context_t* context, const char* obligation){
    char *tmp_obligation=NULL;

    if (context==NULL)    {
	EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
	return EES_FAILURE;
    }
    if (context->class!=OBLIGATION)    {
	EEF_log(LOG_ERR, "%s: context is not an obligation\n", __func__);
	return EES_FAILURE;
    }
    if (obligation && (tmp_obligation=strdup(obligation))==NULL )	{
	EEF_log(LOG_ERR, "%s: out of memory\n", __func__);
	return EES_FAILURE;
    }

    free(context->obligation_name);
    context->obligation_name = tmp_obligation;

    return EES_SUCCESS;
}

/**
 * Returns the obligation name from the aos_context_t pointed to by context
 */
char* getContextObligationId(aos_context_t* context){
    return context->obligation_name;
}

/*
 * Attribute handling functions
 */

/**
 * Returns a pointer to a freshly allocated aos_attribute_t struct
 */
aos_attribute_t* createAttribute(void){
    aos_attribute_t* attribute = calloc(1, sizeof(aos_attribute_t));

    if (attribute==NULL)
	EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
    else
	/* Note: all other elements are initialized to 0 */
        attribute->setting_plugin = get_running_plugin();

    return attribute;
}

/**
 * Adds aos_attribute_t pointed to by attribute to the list of attributes in
 * aos_context_t pointed to by context
 */
EES_RC addAttribute(aos_context_t * context, aos_attribute_t* attribute){
    /* Temporary aos_attribute_t pointer */
    aos_attribute_t* tmp_attribute;

    if (attribute==NULL)	{
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }
    if (context==NULL)  {
        EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
        return EES_FAILURE;
    }

    /* Do we have a list? */
    if ( (tmp_attribute=context->first_attribute) == NULL) {
	/* We are creating the head node */
        context->first_attribute = attribute;
    } else {
	/* Loop over all the attributes till the end of the list or till we find
	 * a duplicate */
	while (tmp_attribute != attribute && tmp_attribute->next != NULL)
	    tmp_attribute = tmp_attribute->next;

	/* Did we end due to a duplicate? */
	if (tmp_attribute == attribute) {
	    EEF_log(LOG_NOTICE, "%s: Skipping duplicate attribute\n",
		    __func__);
	    return EES_FAILURE;
	}

	/* Add the attribute at the end of the list */
	tmp_attribute->next=attribute;
    }

    /* Set the pointer used by getNextAttribute to the just added attribute */
    context->current_attribute=attribute;

    return EES_SUCCESS;
}

/**
 * Removes a supplied attribute from the supplied context
 */
EES_RC destroyAttribute(aos_context_t* context, aos_attribute_t* attribute){
    aos_attribute_t *previous_attribute, *current_attribute;
    
    /* Check attribute itself */
    if (context==NULL)	{
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }

    if (context==NULL)	{
	EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
	return EES_FAILURE;
    }

    /* Set to first attribute */
    current_attribute=context->first_attribute;
    /* Keep track of previous, so that we reconnect after deleting */
    previous_attribute=NULL;

    /* Find the attribute: can just loop over all the attributes */
    while(current_attribute && current_attribute != attribute)	{
	/* Move the pointers */
	previous_attribute=current_attribute;
	current_attribute =current_attribute->next;
    }

    /* Check we found it */
    if (current_attribute==NULL)    {
	EEF_log(LOG_ERR,
		"%s: Cannot find attribute at %p in context at %p\n",
		__func__, (void*)attribute, (void*)context);
	return EES_FAILURE;
    }

    /* TODO: is it really a problem? */
    if(current_attribute->setting_plugin != get_running_plugin()){
	EEF_log(LOG_ERR,
		"%s: Argument %s is not owned by running plugin %s\n",
		__func__, current_attribute->id,
		get_running_plugin()->name);
	return EES_FAILURE;
    }

    /* Re-set pointers in the list and free the current one */
    if (previous_attribute)
	previous_attribute->next = current_attribute->next;
    else
	/* New start point: first element is removed */
	context->first_attribute = current_attribute->next;

    /* Free data */
    free(attribute->type);
    free(attribute->id);
    free(attribute->issuer);
    if(attribute->needs_free)
	free(attribute->data);
    free(attribute);

    return EES_SUCCESS;
}

EES_RC rewindAttributes(aos_context_t* context){
    if (context==NULL)  {
        EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
        return EES_FAILURE;
    }

    context->current_attribute = context->first_attribute;

    return EES_SUCCESS;
}

/**
 * Returns the next aos_attribute_t pointer from list of attributes in the
 * supplied context
 */
aos_attribute_t* getNextAttribute(aos_context_t* context){
    aos_attribute_t* attribute;
    
    if (context==NULL)  {
        EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
        return NULL;
    }

    /* Find current attribute */
    attribute=context->current_attribute;

    /* If there is a current attribute, move the pointer to the next one,
     * otherwise set to NULL */
    context->current_attribute = (attribute ? attribute->next : NULL);

    return attribute;
}

/**
 * Convenience function to obtain an attribute directly from attribute ID
 */
aos_attribute_t *getAttribute(aos_context_t* context, const char *attribute_id){
    aos_attribute_t* attribute;
    
    /* Check input */
    if (context==NULL)  {
        EEF_log(LOG_ERR, "%s: context is NULL\n", __func__);
        return NULL;
    }
    if (attribute_id==NULL)	{
	EEF_log(LOG_ERR, "%s: invalid attribute_id\n", __func__);
	return NULL;
    }

    /* Get the head node of the list of attributes in context */
    attribute = context->first_attribute;

    /* Walk the list as long as we have attributes and they are not the right
     * one */
    while (attribute && strcmp(attribute->id, attribute_id)!=0)
	attribute=attribute->next;

    /* return found attribute or NULL */
    return attribute;
}

/*
 * Getters/setters attribute fields: type, ID, issuer, value
 */

/**
 * Sets attribute id in supplied attribute
 */
EES_RC setAttributeId(aos_attribute_t* attribute, const char* id){
    char *tmp_id=NULL;

    if (attribute==NULL)    {
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }

    if (id && (tmp_id=strdup(id))==NULL )	{
	EEF_log(LOG_ERR, "%s: out of memory\n", __func__);
	return EES_FAILURE;
    }

    free(attribute->id);
    attribute->id = tmp_id;

    return EES_SUCCESS;
}

/**
 * Sets attribute type in supplied attribute
 */
EES_RC setAttributeType(aos_attribute_t * attribute, const char* type){
    char *tmp_type=NULL;

    if (attribute==NULL)    {
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }

    if (type && (tmp_type=strdup(type))==NULL )	{
	EEF_log(LOG_ERR, "%s: out of memory\n", __func__);
	return EES_FAILURE;
    }

    free(attribute->type);
    attribute->type=tmp_type;

    return EES_SUCCESS;
}

/**
 * Sets attribute issuer in supplied attribute
 */
EES_RC setAttributeIssuer(aos_attribute_t* attribute, const char* issuer){
    char *tmp_issuer=NULL;

    if (attribute==NULL)    {
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }

    if (issuer && (tmp_issuer=strdup(issuer))==NULL )	{
	EEF_log(LOG_ERR, "%s: out of memory\n", __func__);
	return EES_FAILURE;
    }

    free(attribute->issuer);
    attribute->issuer = tmp_issuer;

    return EES_SUCCESS;
}

/**
 * Sets attribute value in supplied attribute
 */
EES_RC setAttributeValue(aos_attribute_t* attribute,
			 const void* value, size_t size){
    if (attribute==NULL)    {
	EEF_log(LOG_ERR, "%s: attribute is NULL\n", __func__);
	return EES_FAILURE;
    }
    if(value==NULL) {
	EEF_log(LOG_ERR, "%s: value is NULL\n", __func__);
        return EES_FAILURE;
    }

    /* Free previous value if needed */
    if(attribute->needs_free == 1)
	free(attribute->data);

    /* size==0 is special: we just copy the pointer because we don't know
     * the corresponding size. We also will not touch it. */
    if(size == 0) {
	/* the needs_free==0 indicates elsewhere in the code that we may not
	 * change it. */
	attribute->data = (void*)value;
	attribute->needs_free = 0;
    } else {
	if ( (attribute->data = calloc(1, size+1)) == NULL )	{
	    EEF_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return EES_FAILURE;
	}
	memcpy(attribute->data, value, size);
	attribute->needs_free = 1;
    }

    return EES_SUCCESS;
}

/**
 * Returns id from supplied attribute
 */
char* getAttributeId(aos_attribute_t* attribute){
    return attribute->id;
}

/**
 * Returns type from supplied attribute
 */
char* getAttributeType(aos_attribute_t * attribute){
    return attribute->type;
}

/**
 * Returns issuer from supplied attribute
 */
char* getAttributeIssuer(aos_attribute_t* attribute){
    return attribute->issuer;
}

/**
 * Returns data from supplied attribute casted to a char*
 */
char* getAttributeValueAsString(aos_attribute_t* attribute){
    return (char*) attribute->data;
}

/**
 * Returns data from supplied attribute casted to a int
 */
int getAttributeValueAsInt(aos_attribute_t* attribute){
    return *((int*) attribute->data);
}

/**
 * Returns data from supplied attribute casted to a void*
 */
void* getAttributeValueAsVoidPointer(aos_attribute_t* attribute){
    return (void*) attribute->data;
}



/************************************************************************
 * Private functions
 ************************************************************************/

/**
 * Creates thread-local storage key
 * This is a callback for pthread_once
 */
static void aos_make_key(void){
    pthread_key_create(&_aos_key, aos_free_storage_callback);
    EEF_log(LOG_DEBUG, "%s: Created storage key\n", __func__);
}

/**
 * Frees thread-local storage - callback for pthread_key_create
 */
static void aos_free_storage_callback(void* storage){
    EEF_log(LOG_DEBUG, "%s: Freeing thread-local storage for thread %ld\n",
	    __func__, GETTID());
    aos_free_storage((aos_storage_t*) storage);
}

/**
 * Returns a handle for the storage of the current thread context, a new one is
 * created if there isn't one yet. If the current context is not threaded, a
 * pointer to a global storage struct is returned.
 */
static aos_storage_t * aos_get_storage(void){
    aos_storage_t * storage;
    aos_context_class_t class;

    /* If we are not in threading mode, no thread-local storage should be
     * accessed */
    if (!_is_threading)
        return _global_storage;

    /* Try to get thread-local storage handle if it already exists */
    if ( (storage = pthread_getspecific(_aos_key)) != NULL )
	return storage;

    /* No storage yet, create a new one */
    storage=aos_create_storage();

    /* Set the handle in the pthread thread-local storage */
    if (storage)    {
	/* Initialize the global_current_context pointers */
	for (class=CONTEXT_FIRST+1; class<CONTEXT_LAST; class++)
	    storage->global_current_context[class]=
		_global_storage->first_context;
	pthread_setspecific(_aos_key, storage);
    }

    return storage;
}

/**
 * Allocates and initializes a new storage and returns it
 */
static aos_storage_t * aos_create_storage(void) {
    aos_storage_t * storage;

    /* Log */
    if (!_is_threading)
	EEF_log(LOG_DEBUG, "%s: creating global storage\n", __func__);
    else
	EEF_log(LOG_DEBUG, "%s: creating storage for thread %ld\n",
		__func__, GETTID());

    /* Allocate and initialize a new aos_storage_t struct */
    if( (storage = calloc(1, sizeof(aos_storage_t)))==NULL ) {
	EEF_log(LOG_ERR,"%s: Out of memory\n", __func__);
	return NULL;
    }

    return storage;
}

/**
 * Frees all members and lists of an aos_storage_t structs
 */
static EES_RC aos_free_storage(aos_storage_t* storage)	{
    /* Temporary pointers to aos_context_t and aos_attribute_t */
    aos_context_t   *context, *next_context;
    aos_attribute_t *attribute, *next_attribute;
    /* The number of attributes we've found */
    int             attribute_count = 0;

    /* We need a valid pointer to a storage object to iterate the linked list of
     * aos_context_t structs */
    if(!storage)	{
	EEF_log(LOG_ERR,"%s: invalid storage specified\n", __func__);
        return EES_FAILURE;
    }

    EEF_log(LOG_DEBUG, "%s: Cleaning %s aos storage at %p\n",
	    __func__,
	    (storage==_global_storage ? "global" : "thread-local"),
	    (void*)storage);

    /* Get the head node of the list of contexts*/
    context = storage->first_context;

    /* Walk the list */
    while(context){
        /* Get the list of attributes from the aos_context_t */
        attribute = context->first_attribute;
        /* Walk this list too */
        while(attribute){
            /* Free heap data fields */
            free(attribute->id);
            free(attribute->issuer);
            free(attribute->type);

            /* Some fields may point to automatically allocated variables */
            if(attribute->needs_free){
                EEF_log(LOG_DEBUG, "%s: Freeing arg data %i at %p\n",
			__func__, attribute_count, (void*)(attribute->data));
                free(attribute->data);
            }

            /* Go to the next attribute in the list, and free the current one */
            next_attribute = attribute->next;
            free(attribute);
            attribute = next_attribute;
            attribute_count++;
        }

        EEF_log(LOG_DEBUG, "%s: Freeing context at %p\n", __func__, (void*)context);
        /* Go to the next context in the list, and free the current one */
        next_context = context->next;
	if(context->class == OBLIGATION)
	    free(context->obligation_name);
        free(context);
        context = next_context;
    }

    /* Free the storage node that contained the contexts */
    free(storage);
    return EES_SUCCESS;
}
