/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#include "eef/eef_plugin.h"
#include "eef/eef_aos.h"
#include "eef/eef_log.h"
#include "eef/eef_polytypes.h"


/************************************************************************
 * Defines and typedefs
 ************************************************************************/

/* Defines plugin name to be used in logs */
#define PLUGIN_NAME	"transformer"

/* Defines special attribute ID used by the EES-PEPD-OH in the PEPd to wrap the
 * obligations and their attributes. The attribute value of this attribute ID
 * will be the actual obligation. */
#define SPECIAL_EES_OBLIGATION_LIST_ATTR_ID   "http://authz-interop.org/xacml/attribute/trans-obligation"

/* The wrapped attributes and their values are wrapped as attributes with ID 
 * <obligationID>SPECIAL_EES_OBLIGATION_LIST_DELIMITER<attributeID> with as
 * values the attribute values. */
#define SPECIAL_EES_OBLIGATION_LIST_DELIMITER "?attrId="


/************************************************************************
 * Global variables
 ************************************************************************/

/* maximum number of obligations, will be increased when needed */
static size_t max_obligations=100;


/************************************************************************
 * Private prototypes
 ************************************************************************/

/*
 * getAttributeIDfromAttributeID() will strip the attribute ID at the
 * SPECIAL_EES_OBLIGATION_LIST_DELIMITER followed by urldecoding the string on
 * the right hand side of the delimiter and return it as allocated space. The
 * result must be freed.
 */
static char * getAttributeIDfromAttributeID(const char *);

/* 
 * Returns the URL-decoding of encoded. Result is locally allocated and needs to
 * be freed.
 */ 
static char * url_decode(const char * encoded);

/************************************************************************
 * Public functions
 ************************************************************************/

/* Initialize function */
EES_PL_RC plugin_initialize(int argc, char* argv[]){
    /* All done */
    EEF_log(LOG_INFO, "%s: plugin initialized\n", PLUGIN_NAME);

    return EES_PL_SUCCESS;
}

/* Run function */
EES_PL_RC plugin_run(){
    aos_context_t    *_context            = NULL;
    aos_attribute_t  *_tmp_attr           = NULL;
    aos_context_t    *_context_oblig      = NULL;
    aos_attribute_t  *_tmp_attr_oblig     = NULL;

    char             *_tmp_attr_id        = NULL;
    size_t            i                   = 0;
    char ** list_of_obligations =  NULL;
    size_t  list_of_obligations_cnt = 0;
    char * new_attrib_id     = NULL;
    char * new_attrib_issuer = NULL;
    char * new_attrib_type   = NULL;
    char * new_attrib_value  = NULL;

    
    EEF_log(LOG_INFO, "%s: Running %s\n",
	    PLUGIN_NAME, EEF_getRunningPluginName());

    if ( (list_of_obligations=calloc(max_obligations,sizeof(char *))) == NULL )
    {
	EEF_log(LOG_ERR, "%s: Out of memory\n", PLUGIN_NAME);
	return EES_PL_FAILURE;
    }


    /* Search for a special Environment attributeId
     * It's used as a list of Obligation Ids to search for and transform */
    rewindContexts(ENVIRONMENT);
    while ((_context = getNextContext(ENVIRONMENT))) {
        rewindAttributes(_context);
        while ((_tmp_attr = getNextAttribute(_context))) {
            if ( (_tmp_attr_id = getAttributeId(_tmp_attr)) == NULL)
		continue;

	    /* String match beginning of the attributeID with the
	     * SPECIAL_EES_OBLIGATION_LIST_ATTR_ID to list its values */
	    if (strncmp(_tmp_attr_id,
			SPECIAL_EES_OBLIGATION_LIST_ATTR_ID,
			strlen(SPECIAL_EES_OBLIGATION_LIST_ATTR_ID)) != 0)
		continue;

	    /* List Obligation ID to fetch, make a copy, as we will destroy the
	     * attribute */
	    list_of_obligations[list_of_obligations_cnt] =
		    strdup(getAttributeValueAsString (_tmp_attr));	
	    if (list_of_obligations[list_of_obligations_cnt] == NULL)	{
		EEF_log(LOG_ERR, "%s: out of memory\n", PLUGIN_NAME);
		return EES_PL_FAILURE; 
	    }
	    EEF_log(LOG_DEBUG,
		    "%s: Found transformer obligation attribute ID: %s\n",
		    PLUGIN_NAME, list_of_obligations[list_of_obligations_cnt]);
	    list_of_obligations_cnt++;

	    /* Remove the attribute */
	    destroyAttribute(_context, _tmp_attr);

	    /* Check if we hit the maximum */
	    if(list_of_obligations_cnt == max_obligations){
		EEF_log(LOG_DEBUG,
			"%s: Increasing max number of obligations %lu with 10\n",
			PLUGIN_NAME, (long unsigned)max_obligations);
		max_obligations+=10;
		list_of_obligations = realloc(list_of_obligations,
					      max_obligations);
		if (list_of_obligations==NULL)	{
		    EEF_log(LOG_ERR, "%s: Out of memory\n", PLUGIN_NAME);
		    return EES_PL_FAILURE;
		}
	    }
        }
    }

    /* Loop through explicitly selected Obligation IDs to look for attributes */
    for (i = 0; i < list_of_obligations_cnt; i++) {
	/* Search the Environment attributes for encoded obligations to
	 * transform into real obligations and attributes. When found, register
	 * them. */

	/* Create context for the obligation */
	if ((_context_oblig = createContext(OBLIGATION)) == NULL) {
	    EEF_log(LOG_ERR,
		    "%s: Cannot create obligation context for '%s'\n",
		    PLUGIN_NAME, list_of_obligations[i]);
	    goto error_transformer_bailout;
	}

	/* Set the obligation_ID */
	if (setContextObligationId(_context_oblig, list_of_obligations[i]) !=
		EES_SUCCESS)
	    goto error_transformer_bailout;

	/*
	 * Compare currently selected obligationID with all listed
	 * attribute IDs.
	 * The left-hand side of the attribute ID must match, as it ALSO contains
	 * the obligation's attribute ID
	 * Example:  "http://namespace.tld/xacml/obligation?attribid=http%3A%2F%2Fnamespace.tld%2Fxacml%2Fattribute%2Fexample"
	 * Oblig id: "http://namespace.tld/xacml/obligation"
	 * Attr  id: "http://namespace.tld/xacml/attribute/example"
	 */

	EEF_log(LOG_DEBUG,
		"%s: Looking for attributes for obligationID %s\n",
		PLUGIN_NAME, list_of_obligations[i]);

	/* Loop over all ENVIRONMENT contexts.
	 * Note that for each obligation we need to rewind the contexts */
	rewindContexts(ENVIRONMENT);
	while((_context = getNextContext(ENVIRONMENT))){
	    rewindAttributes(_context);

	    /* Loop over attributes */
	    while((_tmp_attr = getNextAttribute(_context))){
		/* Skip empty or non-matching attributes.
		 * Note: we need to match the first bit of the attributeID as it
		 * will have a ?attribid=... suffix */
		if ( (_tmp_attr_id = getAttributeId(_tmp_attr))==NULL ||
		     strncmp(_tmp_attr_id,
			     list_of_obligations[i],
			     strlen(list_of_obligations[i]))!=0 )
		    continue;

		/* Next corresponding attribute */
		if( (_tmp_attr_oblig = createAttribute()) == NULL) {
		    EEF_log(LOG_ERR,
			    "%s: Cannot create attribute for obligation '%s'\n",
			    PLUGIN_NAME, list_of_obligations[i]);
		    goto error_transformer_bailout;
		}

		/* Let getAttributeIDfromAttributeID() strip the _tmp_attr_id
		 * and return the (URL encoded) right hand side, which contains
		 * the attribute ID */
		new_attrib_id = getAttributeIDfromAttributeID(_tmp_attr_id);
		if (!new_attrib_id){
		    EEF_log(LOG_ERR,
			    "%s: Could not get AttributeID for obligation '%s'\n",
			    PLUGIN_NAME, list_of_obligations[i]);
		    goto error_transformer_bailout;
		}
		EEF_log(LOG_DEBUG, "%s:    found attribute_id '%s'\n",
				    PLUGIN_NAME, new_attrib_id);
		if (setAttributeId(_tmp_attr_oblig, new_attrib_id)==EES_FAILURE)
		    goto error_transformer_bailout;
		/* new_attrib_id is calloc-ed by url-decoding */
		free(new_attrib_id); new_attrib_id=NULL;

		/* Set issuer in the new attribute */
		new_attrib_issuer = getAttributeIssuer(_tmp_attr);
		EEF_log(LOG_DEBUG, "%s:    found attribute_issuer '%s'\n",
				    PLUGIN_NAME,
				    new_attrib_issuer ? new_attrib_issuer : "unset");
		if (setAttributeIssuer(_tmp_attr_oblig, new_attrib_issuer)==EES_FAILURE)
		    goto error_transformer_bailout;

		/* Set type in the new attribute */
		new_attrib_type   = getAttributeType(_tmp_attr);
		EEF_log(LOG_DEBUG, "%s:    found attribute_type '%s'\n",
				    PLUGIN_NAME,
				    new_attrib_type ? new_attrib_type : "unset");
		if (setAttributeType(_tmp_attr_oblig, new_attrib_type)==EES_FAILURE)
		    goto error_transformer_bailout;

		/* Data by pointer */
		new_attrib_value = getAttributeValueAsVoidPointer(_tmp_attr);
		EEF_log(LOG_DEBUG, "%s:    found attribute_value '%s'\n",
				    PLUGIN_NAME, new_attrib_value);
		if (setAttributeValue(_tmp_attr_oblig,
				      new_attrib_value,
				      strlen(new_attrib_value))==EES_FAILURE)
		    goto error_transformer_bailout;

		/* Add attribute to Obligation */
		if (addAttribute(_context_oblig, _tmp_attr_oblig)==EES_FAILURE)
		    goto error_transformer_bailout;

		/* Now remove Environment attribute that got converted. */
		destroyAttribute(_context, _tmp_attr);
            }
        }

	/* We're done with the current list_of_obligations[i] */
	free(list_of_obligations[i]); list_of_obligations[i]=NULL;

	/* Now add the obligation context */
	if (addContext(_context_oblig)==EES_FAILURE)
	    goto error_transformer_bailout;
    }

    /* NOTE: even if we haven't found any matching attribute, it still is valid
     * as obligation. */

    /* Debug */
    AOS_dump(LOG_DEBUG);

    /* Free memory */
    free(list_of_obligations);

    EEF_log(LOG_INFO, "%s: plugin succeeded\n", PLUGIN_NAME);

    return EES_PL_SUCCESS;

error_transformer_bailout:
    if (_context_oblig)	{
	/* free memory of the obligation ID */
	setContextObligationId(_context_oblig, NULL);
	free(_context_oblig);
    }
    for (i=0; i<list_of_obligations_cnt; i++)
	free(list_of_obligations[i]);
    free(list_of_obligations);
    return EES_PL_FAILURE;
}

/* Terminate function */
EES_PL_RC plugin_terminate(){
    EEF_log(LOG_INFO, "%s: plugin terminated\n", PLUGIN_NAME);

    return EES_PL_SUCCESS;
}


/************************************************************************
 * Private functions
 ************************************************************************/

/*
 * getAttributeIDfromAttributeID() will strip the attribute ID at the
 * SPECIAL_EES_OBLIGATION_LIST_DELIMITER followed by urldecoding the string on
 * the right hand side of the delimiter and return it as allocated space. The
 * result must be freed.
 */
static char * getAttributeIDfromAttributeID(const char * compoundID){
    char * attr_id = NULL;

    if (!compoundID){
        EEF_log(LOG_INFO, "%s: %s: nothing to do, empty input.\n",
		PLUGIN_NAME, __func__);
        return NULL;
    }

    /* Strip */
    if (!(attr_id = strstr(compoundID, SPECIAL_EES_OBLIGATION_LIST_DELIMITER))){
        EEF_log(LOG_INFO,
		"%s: No need to convert anything. No occurance found of %s\n",
		PLUGIN_NAME, SPECIAL_EES_OBLIGATION_LIST_DELIMITER);
        return NULL;
    }

    /* Move the pointer beyond SPECIAL_EES_OBLIGATION_LIST_DELIMITER */
    attr_id = &attr_id[strlen(SPECIAL_EES_OBLIGATION_LIST_DELIMITER)];

    /* URL decode */
    if (!attr_id)
        return NULL;
   
    /* Note: we need to free this memory */
    return url_decode(attr_id);
}

/*
 * Returns the URL-decoding of encoded. Result is locally allocated and needs to
 * be freed.
 */
static char * url_decode(const char * encoded){
    size_t i = 0, e = 0, d = 0;
    size_t encoded_counter = 0;
    char * decoded = NULL;
    char   encoded_char[3]={0,0,0};
    char * illval=NULL;
    long   longval;

    if (!encoded)
        return NULL;

    /* Count encoded character occurances */
    for (i = 0; i < strlen(encoded); i++){
        if (encoded[i] == '%' &&
	    isxdigit(encoded[i + 1]) &&
	    isxdigit(encoded[i + 2]))
	{
            encoded_counter++;
        }
    }

    /* Allocate space for the result to fit the decoded form */
    decoded = calloc(strlen(encoded) - 2 * encoded_counter + 1, sizeof(char));
    if (!decoded)
	return NULL;

    /* Decode string */
    for (d = 0, e = 0; e < strlen(encoded); d++, e++){
        /* Search for encoded characters */
        if (encoded[e] == '%' &&
	    isxdigit(encoded[e + 1]) &&
	    isxdigit(encoded[e + 2]))
	{
            encoded_char[0] = encoded[e + 1];
            encoded_char[1] = encoded[e + 2];
            e += 2;

	    errno=0;
	    longval=strtol(encoded_char, &illval, 16);
	    /* Check for all kinds of errors: errno, non-parsed characters or
	     * value that's not a single byte (latter one is not really
	     * possible) */
	    if (errno || *illval!='\0' || longval<0 || longval>255)  {
		/* Something is wrong here */
		free(decoded);
		return NULL;
	    }
	    decoded[d] = (char)longval;
        } else {
            decoded[d] = encoded[e];
        }
    }
    return decoded;
}
