/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _XOPEN_SOURCE	500

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "eef/eef_plugin.h"
#include "eef/eef_aos.h"
#include "eef/eef_polytypes.h"
#include "eef/eef_log.h"


/************************************************************************
 * Defines and typedefs
 ************************************************************************/

/* Defines plugin name to be used in logs */
#define PLUGIN_NAME	"ees_dummy_good"

/* Internal obligation structure: id + N attribute ID/value pairs */
typedef struct oblig_s	{
    char *id;		/* obligation ID */
    size_t n_attr;	/* number of attributes */
    char **attr_id;	/* list of attribute IDs */
    char **attr_val;	/* list of corresponding attribute values */
} obligation_t;


/************************************************************************
 * Global variables
 ************************************************************************/

static obligation_t *obligation=NULL;	/* list of obligations */
static size_t n_obligation=0;		/* number of obligations */


/************************************************************************
 * Private prototypes
 ************************************************************************/

/* Cleans all the memory allocated so far */
static void clean_memory(void);


/************************************************************************
 * Public functions
 ************************************************************************/

/* Initialize function */
EES_PL_RC plugin_initialize(int argc, char* argv[]){
    int i;
    char *pos;
    obligation_t *curr_oblig=NULL;
    size_t len;
    size_t max_oblig=0;
    size_t max_attr=0;
    
    /* Parse its command line options */
    for (i=1; i<argc; i++)  {
	/* Find = sign */
	if ( (pos=strchr(argv[i],'=')) == NULL )    {
	    EEF_log(LOG_ERR, "%s: Not a valid option: %s\n",
		    PLUGIN_NAME, argv[i]);
	    goto init_failed;
	}
	/* find len of ID */
	len=strlen(argv[i])-strlen(pos);
	/* start of value */
	pos++;
	/* What type of ID do we have */
	if (strncmp(argv[i],"--obligation_id",len)==0 ||
	    strncmp(argv[i],"-obligation_id",len)==0)
	{
	    /* Check wether we need to allocate (more) memory */
	    if (n_obligation==max_oblig) {
		max_oblig+=10;
		obligation=realloc(obligation,
				   max_oblig*sizeof(obligation_t));
		if (obligation==NULL)    {
		    EEF_log(LOG_ERR,"%s: out of memory\n", PLUGIN_NAME);
		    goto init_failed;
		}
	    }
	    /* Set shortcut to new obligation */
	    curr_oblig=&(obligation[n_obligation++]);
	    /* Initialize attribute fields */
	    curr_oblig->n_attr=max_attr=0;
	    curr_oblig->attr_id=curr_oblig->attr_val=NULL;
	    /* set obligation_id */
	    if ( (curr_oblig->id=strdup(pos))==NULL)	{
		EEF_log(LOG_ERR,"%s: out of memory\n", PLUGIN_NAME);
		goto init_failed;
	    }
	} else if (strncmp(argv[i],"--attribute_id",len)==0 ||
		 strncmp(argv[i],"-attribute_id",len)==0)
	{
	    /* Check we have an obligation */
	    if (curr_oblig==NULL) {
		EEF_log(LOG_ERR,
			"%s: Found attribute_id without obligation\n",
			PLUGIN_NAME);
		goto init_failed;
	    }
	    /* Check wether we need to allocate (more) memory */
	    if (curr_oblig->n_attr==max_attr)	{
		max_attr+=10;
		curr_oblig->attr_id=realloc(curr_oblig->attr_id,
					    max_attr*(sizeof(char *)));
		curr_oblig->attr_val=realloc(curr_oblig->attr_val,
					     max_attr*(sizeof(char *)));
		if (curr_oblig->attr_id==NULL || curr_oblig->attr_val==NULL) {
		    EEF_log(LOG_ERR,"%s: out of memory\n", PLUGIN_NAME);
		    goto init_failed;
		}
	    }
	    /* Initialize value, set attribute_id and increase number of
	     * attributes */
	    curr_oblig->attr_val[curr_oblig->n_attr]=NULL;
	    curr_oblig->attr_id[curr_oblig->n_attr]=strdup(pos);
	    if ( curr_oblig->attr_id[curr_oblig->n_attr] == NULL ) {
		EEF_log(LOG_ERR,"%s: out of memory\n", PLUGIN_NAME);
		goto init_failed;
	    }
	    curr_oblig->n_attr++;
	}
	else if (strncmp(argv[i],"--attribute_value",len)==0 ||
		 strncmp(argv[i],"-attribute_value",len)==0)
	{
	    /* Check we have an obligation AND have already found and ID and
	     * hence allocated memory for the values */
	    if (curr_oblig==NULL || curr_oblig->attr_val==NULL) {
		EEF_log(LOG_ERR,
			"%s: Found attribute_value without obligation "
			"or attribute_id\n", PLUGIN_NAME);
		goto init_failed;
	    }
	    /* Set attribute_value */
	    curr_oblig->attr_val[(curr_oblig->n_attr)-1]=strdup(pos);
	    if ( curr_oblig->attr_val[(curr_oblig->n_attr)-1] == NULL ) {
		EEF_log(LOG_ERR,"%s: out of memory\n", PLUGIN_NAME);
		goto init_failed;
	    }
	} else {
	    /* Unknown command line option */
	    EEF_log(LOG_ERR, "%s: Unknown option: %s\n", PLUGIN_NAME, argv[i]);
	    goto init_failed;
	}
    }
    
    /* All done */
    EEF_log(LOG_INFO, "%s: plugin initialized\n", PLUGIN_NAME);

    return EES_PL_SUCCESS;

init_failed:
    clean_memory();
    
    return EES_PL_FAILURE;
}

/* Run function */
EES_PL_RC plugin_run(void)
{
    aos_context_t    *_context        = NULL;
    aos_attribute_t  *_tmp_attr       = NULL;
    size_t i,j,size;
    const char *attr_val;

    /* Loop over obligations */
    for (i=0; i<n_obligation; i++)	{
	if ((_context = createContext(OBLIGATION)) == NULL) {
	    EEF_log(LOG_ERR,
		    "%s: Error, cannot create obligation context\n",
		    PLUGIN_NAME);
	    return EES_PL_FAILURE;
	}
	if (setContextObligationId(_context, obligation[i].id)==EES_FAILURE) {
	    EEF_log(LOG_ERR, "%s: setting obligationID failed\n", PLUGIN_NAME);
	    return EES_PL_FAILURE;
	}
	/* Loop over attributes */
	for (j=0; j<obligation[i].n_attr; j++)   {
	    if ( (_tmp_attr = createAttribute()) == NULL) {
		EEF_log(LOG_ERR,
			"%s: Error, cannot create attribute\n",
			PLUGIN_NAME);
		return EES_PL_FAILURE;
	    }

	    /* Determine correct attribute value */
	    if (obligation[i].attr_val[j] == NULL)  {
		attr_val="";
		size=0;
	    } else {
		attr_val=obligation[i].attr_val[j];
		size=strlen(attr_val)+1;
	    }

	    /* Set attribute ID and value */
	    if (setAttributeId(_tmp_attr, obligation[i].attr_id[j])==EES_FAILURE ||
		setAttributeValue(_tmp_attr, attr_val, size)==EES_FAILURE)
	    {
		EEF_log(LOG_ERR, "%s: setting attributeID or attributeValue failed\n",
			PLUGIN_NAME);
		return EES_PL_FAILURE;
	    }

	    /* Log  */
	    EEF_log(LOG_INFO, "%s: Adding obligation %s with attribute %s=%s\n",
		    PLUGIN_NAME,
		    obligation[i].id, obligation[i].attr_id[j], attr_val);

	    /* Add attribute to the obligation */
	    if (addAttribute(_context, _tmp_attr)==EES_FAILURE)    {
		EEF_log(LOG_ERR, "%s: adding attribute to context failed\n",
			PLUGIN_NAME);
		return EES_PL_FAILURE;
	    }
	}
	/*  Log and add the obligation itself */
	EEF_log(LOG_INFO, "%s: Adding obligation %s\n",
		PLUGIN_NAME, obligation[i].id);

	if (addContext(_context)==EES_FAILURE)    {
	    EEF_log(LOG_ERR, "%s: adding obligation context failed\n",
		    PLUGIN_NAME);
	    return EES_PL_FAILURE;
	}
    }

    EEF_log(LOG_INFO, "%s: plugin succeeded\n", PLUGIN_NAME);

    return EES_PL_SUCCESS;
}

/* Terminate function */
EES_PL_RC plugin_terminate(void){
    clean_memory();

    EEF_log(LOG_INFO, "%s: plugin terminated\n", PLUGIN_NAME);
    
    return EES_PL_SUCCESS;
}


/************************************************************************
 * Private functions
 ************************************************************************/

/*
 * Cleans all the memory allocated so far
 */
static void clean_memory(void)   {
    size_t i,j;

    /* Free the contents for each obligation */
    for (i=0; i<n_obligation; i++)	{
	free(obligation[i].id);
	for (j=0; j<obligation[i].n_attr; j++)	{
	    free(obligation[i].attr_id[j]);
	    free(obligation[i].attr_val[j]);
	}
	free(obligation[i].attr_id);
	free(obligation[i].attr_val);
    }
    /* Free the obligation array itself */
    free(obligation);
    obligation=NULL;
    n_obligation=0;
}
