/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*!

        \file           eval_manager.h
        \brief  Functions for the evaluation manager
        \author Aram Verstegen \& Mischa Sall\'e

        This header contains the functions used by the PDL parser:
        It contains the following functions:

  -# start_eval_manager(int, char**): starts the evaluation manager
  -# stop_eval_manager(void): stops the evaluation manager

  -# get_policies(void): returns all policies

  -# pdl_lex_cleanup(void): cleans up lex memory, only for flex
  -# int pdl_yyparse(void): wrapper around yyparse
  -# int yylex(void): prototype for lex function
  -# int yyerror(char*): prototype for lex function
  -# void set_yylval(record_t* r): prototype for lex function
  -# void delete_lex_buffer(void): prototype for lex function

  -# add_variable(record_t*, record_t* ): adds a variable (plug-in) to the list
  -# add_policy(record_t*, rule_t*): adds a new policy
  -# add_rule(record_t*, record_t*, record_t*): adds a new rule
  -# remove_policy(record_t* ): removes a policy
  -# concat_strings_with_space(record_t*, record_t*): concatenates two strings separated by a space 

  -# set_pdl_modules_path(record_t*): sets plugin dir
  -# set_pdl_port(record_t*): sets HTTP port
  -# set_pdl_send_timeout(record_t*): sets HTTP send timeout
  -# set_pdl_receive_timeout(record_t*): sets HTTP receive timeout
  -# set_pdl_accept_timeout(record_t*): sets HTTP accept timeout
  -# set_pdl_log_file(record_t*): sets log file
  -# set_pdl_log_level(record_t*): sets log level
  -# set_pdl_log_facility(record_t*): sets syslog facility
  -# set_pdl_policynames(record_t*): sets policy names to run
 */

#ifndef PDL_H
#define PDL_H

#include "ees_config.h"

#include <stdio.h>
#include "eef_plugindl_s.h"
#include "eef/eef_return_codes.h"


/************************************************************************
 * Type definitions
 ************************************************************************/

/**
 * \struct em_record_s
 * \brief Structure for symbols used by the parser internally
 */
typedef struct em_record_s {
  char*               string;       /*!<  Hold the value that lex has found. */
  unsigned int        uval;	    /*!<  Hold the value that lex has found. */
  int                 lineno;       /*!<  Hold the line number the symbol has been found. */
} record_t;


/**
 * \struct em_var_s
 * \brief Structure keeps track of the variables, their value and the line number they are defined on.
 */
typedef struct em_var_s {
  char*               name;         /*!<  Name of the variable. */
  char*               value;        /*!<  Value of the variable. */
  int		      lineno;       /*!<  Line number the variable appears on. */
  struct em_var_s*    next;         /*!<  Next variable, or 0 if none. */
} var_t;

/**
 * \struct em_rule_s
 * \brief Structure to make a tree of 
 */
typedef struct em_rule_s {
  int		      lineno;       /*!<  Line number where rule appeared. */
  char*               name;         /*!<  Name of the state.*/
  struct em_rule_s*   true_branch;  /*!<  True branch or NULL if none */
  struct em_rule_s*   false_branch; /*!<  False branch or NULL if none */
  struct em_rule_s*   next;         /*!<  Next rule, or NULL if none */
  eef_plugindl_t*     plugin;
} rule_t;

/**
 * \struct em_policy_s
 * \brief Structure to build a tree of plugins as defined in a policy in the config file
 */
typedef struct em_policy_s {
  char*               name;
  int		      lineno;
  rule_t*             rules;
  int                 rules_list_transformed_to_tree;
  struct em_policy_s* next;
} policy_t;


/************************************************************************
 * Variable declarations
 ************************************************************************/

/* since pdl_lex.c already declares these, skip 'em here: this has only effect
 * for when we are currently being included in pdl_lex.c which is created by
 * (f)lex */
#ifndef FLEX_SCANNER
extern FILE* yyin;
#endif
extern int lineno;


/************************************************************************
 * Function prototypes
 ************************************************************************/

/*
 * lifecycle functions
 */

/*! Starts the parser */
EES_RC	  start_eval_manager(int number_of_policies, char* names_of_policies[]);

/*! Cleans up lex/yacc memory including config options */
void	  stop_eval_manager(void);

/*! Returns all policies */
policy_t* get_policies(void);


/*
 * Lex/Yacc stuff
 */ 

/* Wrapper function to call proper cleanup functions for lex memory, dependent
 * on flex version. Only called for flex. */
void pdl_lex_cleanup(void);

/* Wrapper around yyparse() since RH5 does not declare yyparse() in the created
 * header file, while newer yacc/bison does */
int pdl_yyparse(void);

/* Also declare yylex() for pdl_yacc.{y,c} */
int yylex(void);
int yyerror(char*);
void set_yylval(record_t* r);
#if HAVE_FLEX
void delete_lex_buffer(void);
#endif


/*
 * callback functions for lex/yacc
 */
void         add_variable(record_t* name, record_t* value);
void         add_policy(record_t*, rule_t*);
rule_t *     add_rule(record_t*, record_t*, record_t*);
void         remove_policy(record_t* policy);
/* convenience methods used by the lexer */
record_t *   concat_strings_with_space(record_t*, record_t*);

/*
 * functions to set the different options specified in the config file
 */
void         set_pdl_modules_path(record_t*);	/* plugin dir */
void         set_pdl_port(record_t*);		/* HTTP port */
void         set_pdl_send_timeout(record_t*);	/* HTTP send timeout */
void         set_pdl_receive_timeout(record_t*);/* HTTP receive timeout */
void         set_pdl_accept_timeout(record_t*);	/* HTTP accept timeout */
void         set_pdl_log_file(record_t*);	/* log file */
void         set_pdl_log_level(record_t*);	/* log level */
void         set_pdl_log_facility(record_t*);	/* syslog facility */
void         set_pdl_policynames(record_t*);    /* policy names to run */

#endif
