/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*!

	\file eef_log.h
	\brief API of the EEF logging functions
	\author Aram Verstegen \& Mischa Sall\'e

	This header contains the declarations of the EEF library logfunctions:
	-# EEF_log(int, const char*, ...): EEF_log function
	-# EEF_openlog(void): Open log, for logfile, set using EEF_(init_)set_log_file()
	-# EEF_reopenlog(void): Re-open log
	-# EEF_closelog(void): Close log

	-# EEF_set_log_level(int loglevel): Sets the log level [0...5]
	-# EEF_get_log_level(void): Returns the log level [0...5]
	-# EEF_set_debug_mode(int mode): Set the debugmode on/off. Debug mode: also to stderr, level LOG_DEBUG
	-# EEF_get_debug_mode(void): Returns the debugmode
	-# EEF_set_log_file(const char *): Sets the log file and opens it
	-# EEF_get_log_file(void): Returns the log file, NULL indicates syslog
	-# EEF_set_log_ident(const char*): Set the log `ident', PACKAGE_NAME by default
	-# EEF_get_log_ident(void): Returns the log `ident'
	-# EEF_set_log_facility(const char *): Sets the log facility e.g. LOG_LOCAL7

	-# EEF_strerror_r(int , char *, size_t ): Wrapper around strerror_r to always mimick the XSI behaviour
	-# EEF_set_conf_log_level(int ): Sets the log level [0...5], overriding the value in the config file
	-# EEF_set_conf_log_facility(const char *): Sets the log facility, e.g. LOG_LOCAL7, overriding the value in the config file
	-# EEF_set_conf_log_file(const char *): Sets the log file and opens it, overriding the value in the config file

*/

#ifndef EEF_LOG_H
#define EEF_LOG_H

/* Include syslog.h for LOG_* levels, otherwise always need to include both */
#include <syslog.h>
#include <eef/eef_return_codes.h>

/* EEF_log function */
#ifdef __GNUC__
void EEF_log(int, const char*, ...) __attribute__ ((format(printf, 2, 3)));
#else
void EEF_log(int, const char*, ...);
#endif


/*! Open log, for logfile, set using EEF_(init_)set_log_file() */
EES_RC EEF_openlog(void);
/*! Re-open log */
EES_RC EEF_reopenlog(void);
/*! Close log */
EES_RC EEF_closelog(void);


/*! Sets the log level [0...5] */
EES_RC EEF_set_log_level(int loglevel);

/*! Returns the log level [0...5] */
int EEF_get_log_level(void);


/*! Set the debugmode on/off. Debug mode: also to stderr, level LOG_DEBUG */
void EEF_set_debug_mode(int mode);

/*! Returns the debugmode */
int  EEF_get_debug_mode(void);


/*! Sets the log file and opens it */
EES_RC EEF_set_log_file(const char *logfile);

/*! Returns the log file, NULL indicates syslog */
const char *EEF_get_log_file(void);


/*! Set the log `ident', PACKAGE_NAME by default */ 
void EEF_set_log_ident(const char*);

/*! Returns the log `ident', PACKAGE_NAME by default */ 
const char *EEF_get_log_ident(void);


/*! Sets the log facility e.g. LOG_LOCAL7 */
EES_RC EEF_set_log_facility(const char *facility_name);


/*! Wrapper around strerror_r to always mimick the XSI behaviour */
void EEF_strerror_r(int errnum, char *buf, size_t len);

/**
 * Configuration setting functions (into eval_manager.c)
 * The values set using the following functions will be used instead of the
 * corresponding values in the config file and will also survive a restart of
 * the service (when the config file is reparsed).
 * The input will *not* be freed by any function and should only be freed (when
 * needed) upon final shutdown and after the EEF has been already been stopped.
 * They are typically used for implementing command line options.
 */

/*! Sets the log level [0...5], overriding the value in the config file */
EES_RC EEF_set_conf_log_level(int level);

/*! Sets the log facility, e.g. LOG_LOCAL7, overriding the value in the config
 * file */
EES_RC EEF_set_conf_log_facility(const char *facility);

/*! Sets the log file and opens it, overriding the value in the config file */
EES_RC EEF_set_conf_log_file(const char *file);

#endif
