/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*!

	\file eef_library.h
	\brief API of the EEF library
	\author Aram Verstegen \& Mischa Sall\'e

	This header contains the declaration of the EEF library functions:
	-# EEF_daemonize(void): Daemonizes the EEF process
	-# EEF_init(void): Initializes EEF sub processes: evaluation manager, plugin manager, AOS and plugings
	-# EEF_run(char *policy_names[]): Runs plugins for given policies
	-# EEF_term(void): Stops the EEF sub processes
	-# EEF_startThreading(void): Enables threading for the EEF: the AOS and plugin manager

	-# EEF_getMajorVersion(void): Return EEF major version
	-# EEF_getMinorVersion(void): Return EEF minor version
	-# EEF_getPatchVersion(void): Return EEF patch version

	-# EEF_getRunningPluginName(void): Returns the name of the currently running plugin

	-# EEF_set_config_file(const char *file): Sets the config file
	-# EEF_set_conf_modules_path(const char *path): Sets the module path, overriding the value in the config file
	-# EEF_set_conf_http_port(int port): Sets the HTTP EIC port, overriding the value in the config file
	-# EEF_set_conf_http_timeouts(int, int, int): Sets the HTTP EIC timeouts, overriding the values in config file
	-# EEF_set_conf_policy_names(const char *): Sets the list of policies, overriding the values in config file

	-# EEF_get_config_file(void): Return the config file
	-# EEF_get_modules_path(void): Return the current module path
	-# EEF_get_http_port(void): Returns the port for the HTTP EIC
	-# EEF_get_http_timeouts(int *, int *, int *): Returns the timeouts for the HTTP EIC
	-# EEF_get_policy_names(void): Returns the list of policy names
*/
#ifndef EEF_H
# define EEF_H

#include <eef/eef_return_codes.h>

/* lifecycle functions */

/* Daemonizes the EEF process */
EES_RC  /*@alt void@*/ EEF_daemonize(void);

/* Starts the evaluation manager (which also starts the plugin manager),
 * initializes AOS and initializes the plugins */
EES_RC  /*@alt void@*/ EEF_init(void);

/* Runs all the plugins for given or all policies. When NULL run all, otherwise
 * the list should be NULL-terminated. */
EES_RC  /*@alt void@*/ EEF_run(char *policy_names[]);

/* Stops the evaluation and plug-in manager, the AOS and the logging. */
EES_RC  /*@alt void@*/ EEF_term(void);

/* Enables threading for the EEF: the AOS and plugin manager */
EES_RC  /*@alt void@*/ EEF_startThreading(void);


/* Note: The next functions are also declared in eef_plugin.h */

/* Return the EEF version */
unsigned int           EEF_getMajorVersion(void);
unsigned int           EEF_getMinorVersion(void);
unsigned int           EEF_getPatchVersion(void);

/* Returns the name of the currently running plugin */
const char* /*@alt void@*/ EEF_getRunningPluginName(void);


/**
 * Configuration setting functions (into eval_manager.c)
 * The values set using the following functions will be used instead of the
 * corresponding values in the config file and will also survive a restart of
 * the service (when the config file is reparsed).
 * The input will *not* be freed by any function and should only be freed (when
 * needed) upon final shutdown and after the EEF has been already been stopped.
 * They are typically used for implementing command line options.
 */

/*! Sets the config file */
void EEF_set_config_file(const char *file);

/*! Sets the module path, overriding the value in the config file */
void EEF_set_conf_modules_path(const char *path);

/*! Sets the HTTP EIC port, overriding the value in the config file */
EES_RC EEF_set_conf_http_port(int port);

/*! Sets the HTTP EIC timeouts, overriding the values in config file */
EES_RC EEF_set_conf_http_timeouts(int send_timeout,
				  int recv_timeout,
				  int acpt_timeout);

/*! Sets the list of policies, overriding the values in config file */
EES_RC EEF_set_conf_policy_names(const char *policy_string);


/**
 * Configuration querying functions (from eval_manager.c)
 * The returned values are the combination of those set by `hand' using the
 * above functions, and those set in the config file.
 */

/*! Return the config file */
const char *EEF_get_config_file(void);

/*! Return the current module path */
const char* EEF_get_modules_path(void);

/*! Returns the port for the HTTP EIC */
int EEF_get_http_port(void);

/*! Returns the timeouts for the HTTP EIC */
void EEF_get_http_timeouts(int *send_timeout,
			   int *recv_timeout,
			   int *acpt_timeout);

/*! Returns the list of policy names */
char **EEF_get_policy_names(void);

#endif

