/**
 * Copyright 2010-2013  Members of the EMI Collaboration.
 * Copyright 2010-  Stichting Fundamenteel Onderzoek der Materie (FOM-Nikhef)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

/*!

	\file		eef_aos.h
	\brief	API of the AOS (Attribute Object Store)
	\author	Aram Verstegen \& Mischa Sall\'e

	This header contains the declarations of the Attribute Object Store.
	It contains the following core functions:
	-# AOS_dump(int loglevel): 

	-# createContext(aos_context_class_t): Create new empty context of given class
	-# addContext(aos_context_t*): Add context to thread-local or global storage
	-# destroyContext(aos_context_t*): Removes and cleans context from thread-local or global storage

	-# rewindContexts(aos_context_class_t): Rewinds context iterator for given class for thread-local AND read-only global storage
	-# rewindGlobalContexts(aos_context_class_t): Rewinds context iterator for given class for global storage
	-# getNextContext(aos_context_class_t): Returns next context in thread-local, then global storage
	-# getNextGlobalContext(aos_context_class_t): Returns next context in global storage
	-# getObligation(const char *): Convenience function: returns first obligation context with given obligation_id

	-# getContextClass(aos_context_t*): Returns context class for given context
	-# setContextObligationId(aos_context_t*, const char *): For obligation context only, set ObligationId
	-# getContextObligationId(aos_context_t*): For obligation context only, set ObligationId

	-# createAttribute(void): Create new empty attribute
	-# addAttribute(aos_context_t*, aos_attribute_t*): Add attribute to context
	-# destroyAttribute(aos_context_t*, aos_attribute_t*): Removes attribute from context and destroys it

	-# rewindAttributes(aos_context_t*): Rewind attribute iterator for given context
	-# getNextAttribute(aos_context_t*): Return next attribute in context
	-# getAttribute(aos_context_t*, const char *): Convenience function: returns first attribute with given attribute_id from context

	-# setAttributeId(aos_attribute_t*, const char* id): Set attribute ID field of attribute
	-# setAttributeType(aos_attribute_t*, const char* type): Set attribute Type field of attribute
	-# setAttributeIssuer(aos_attribute_t*, const char* issuer): Set attribute Issuer field of attribute
	-# setAttributeValue(aos_attribute_t*, const void* value, size_t size): Set attribute Value field of attribute

	-# getAttributeId(aos_attribute_t*): Returns attribute ID field of attribute
	-# getAttributeType(aos_attribute_t *): Returns attribute Type field of attribute
	-# getAttributeIssuer(aos_attribute_t*): Returns attribute Issuer field of attribute
	-# getAttributeValueAsString(aos_attribute_t*): Returns attribute Value field of attribute as char*
	-# getAttributeValueAsInt(aos_attribute_t*): Returns attribute Value field of attribute as int
	-# getAttributeValueAsVoidPointer(aos_attribute_t*): Returns attribute Value field of attribute as void pointer

*/

#ifndef AOS_API_H
#define AOS_API_H

/* Either stdlib.h or stddef.h for size_t */
#include <stdlib.h>

#include <eef/eef_polytypes.h>
#include <eef/eef_return_codes.h>


/************************************************************************
 * Type definitions
 ************************************************************************/

typedef struct aos_context_s       aos_context_t;
typedef struct aos_attribute_s     aos_attribute_t;
typedef struct aos_storage_s       aos_storage_t;


/************************************************************************
 * Function prototypes
 ************************************************************************/

/* General functions */

/*! Dump contents of AOS on giving loglevel */
void AOS_dump(int loglevel);


/*
 * context handling functions
 */

/*! Create new empty context of given class. After it has been added (\see
 * addContext), it can be removed and cleaned using destroyContext() */
aos_context_t *     createContext(aos_context_class_t);

/*! Add context to thread-local or global storage */
EES_RC              addContext(aos_context_t*);

/*! Removes and cleans context from thread-local or global storage */
EES_RC		    destroyContext(aos_context_t*);

/*! Rewinds context iterator for given class (use ANY for all) for thread-local
 * AND read-only global storage */
EES_RC              rewindContexts(aos_context_class_t);

/*! As rewindContexts(), but only for read-only global storage */
EES_RC              rewindGlobalContexts(aos_context_class_t);

/*! Returns next context of given class: first iterate through thread-local,
 * then continue in read-only global storage */
aos_context_t *     getNextContext(aos_context_class_t);

/*! Returns next context in global (read-only) storage */
aos_context_t *     getNextGlobalContext(aos_context_class_t);

/*! Convenience function: returns first obligation context with given
 * obligation_id */
aos_context_t *     getObligation(const char *);


/*
 * handling of context fields
 */

/*! Returns context class for given context */
aos_context_class_t getContextClass(aos_context_t*);

/*! For obligation context only, set ObligationId. The obligation ID argument is
 * strdup-ped so the argument can be freed directly by the caller */
EES_RC              setContextObligationId(aos_context_t*, const char *);

/*! For obligation context only, set ObligationId */
char*               getContextObligationId(aos_context_t*);


/*
 * attribute handling functions
 */

/*! Create new empty attribute, after it has been added to context (\see
 * addAttribute) it can be removed and cleaned with destroyAttribute() */
aos_attribute_t*    createAttribute(void);

/*! Add attribute to context */
EES_RC              addAttribute(aos_context_t*, aos_attribute_t*);

/*! Removes attribute from context and destroys it */
EES_RC              destroyAttribute(aos_context_t*, aos_attribute_t*);

/*! Rewind attribute iterator for given context */
EES_RC              rewindAttributes(aos_context_t*);

/*! Return next attribute in context */
aos_attribute_t*    getNextAttribute(aos_context_t*);

/*! Convenience function: returns first attribute with given attribute_id from
 * context */
aos_attribute_t*    getAttribute(aos_context_t*, const char *);

/* set the actual fields of the attribute */

/*! Set attribute ID field of attribute. id is strdup-ped, so can be freed by
 * the caller directly */
EES_RC              setAttributeId(aos_attribute_t*, const char* id);
/*! Set attribute Type field of attribute. type is strdup-ped, so can be freed
 * by the caller directly */
EES_RC              setAttributeType(aos_attribute_t*, const char* type);
/*! Set attribute Issuer field of attribute. issuer is strdup-ped, so can be
 * freed by the caller directly */
EES_RC              setAttributeIssuer(aos_attribute_t*, const char* issuer);
/*! Set attribute Value field of attribute. If size!=0, value is duplicated, in
 * that case it can be freed by the caller directly */
EES_RC              setAttributeValue(aos_attribute_t*, const void* value, size_t size);

/* get the actual fields of the attribute */

/*! Returns attribute ID field of attribute */
char*               getAttributeId(aos_attribute_t*);
/*! Returns attribute Type field of attribute */
char*               getAttributeType(aos_attribute_t *);
/*! Returns attribute Issuer field of attribute */
char*               getAttributeIssuer(aos_attribute_t*);

/*! Returns attribute Value field of attribute as char* */
char*               getAttributeValueAsString(aos_attribute_t*);
/*! Returns attribute Value field of attribute as int */
int                 getAttributeValueAsInt(aos_attribute_t*);
/*! Returns attribute Value field of attribute as void pointer */
void*               getAttributeValueAsVoidPointer(aos_attribute_t*);

#endif

