/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.pool.repository;

import diskCacheV111.util.CacheException;
import diskCacheV111.util.ChecksumFactory;
import diskCacheV111.util.DiskErrorCacheException;
import diskCacheV111.util.FileInCacheException;
import diskCacheV111.util.PnfsHandler;
import diskCacheV111.util.PnfsId;
import diskCacheV111.vehicles.StorageInfo;
import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import org.dcache.pool.classic.ChecksumModuleV1;
import org.dcache.pool.classic.ReplicaStatePolicy;
import org.dcache.pool.repository.DuplicateEntryException;
import org.dcache.pool.repository.EntryState;
import org.dcache.pool.repository.FileStore;
import org.dcache.pool.repository.MetaDataRecord;
import org.dcache.pool.repository.MetaDataStore;
import org.dcache.pool.repository.StickyRecord;
import org.dcache.vehicles.FileAttributes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsistentStore
implements MetaDataStore {
    private static final Logger _log = LoggerFactory.getLogger(ConsistentStore.class);
    private static final String RECOVERING_MSG = "Recovering %1$s...";
    private static final String MISSING_MSG = "Recovering: Reconstructing meta data for %1$s";
    private static final String PARTIAL_FROM_TAPE_MSG = "Recovering: Removed %1$s because it was not fully staged";
    private static final String RECOVERING_FETCHED_STORAGE_INFO_FOR_1$S_FROM_PNFS = "Recovering: Fetched storage info for %1$s from PNFS";
    private static final String FILE_NOT_FOUND_MSG = "Recovering: Removed %1$s because name space entry was deleted";
    private static final String UPDATE_SIZE_MSG = "Recovering: Setting size of %1$s in PNFS to %2$d";
    private static final String MARKED_MSG = "Recovering: Marked %1$s as %2$s";
    private static final String REMOVING_REDUNDANT_META_DATA = "Removing redundant meta data for %s";
    private static final String BAD_MSG = "Marked %1$s bad: %2$s";
    private static final String BAD_SIZE_MSG = "File size mismatch for %1$s. Expected %2$d bytes, but found %3$d bytes.";
    private final PnfsHandler _pnfsHandler;
    private final MetaDataStore _metaDataStore;
    private final FileStore _fileStore;
    private final ChecksumModuleV1 _checksumModule;
    private final ReplicaStatePolicy _replicaStatePolicy;
    private String _poolName;

    public ConsistentStore(PnfsHandler pnfsHandler, ChecksumModuleV1 checksumModule, FileStore fileStore, MetaDataStore metaDataStore, ReplicaStatePolicy replicaStatePolicy) {
        this._pnfsHandler = pnfsHandler;
        this._checksumModule = checksumModule;
        this._fileStore = fileStore;
        this._metaDataStore = metaDataStore;
        this._replicaStatePolicy = replicaStatePolicy;
    }

    public void setPoolName(String poolName) {
        if (poolName == null || poolName.isEmpty()) {
            throw new IllegalArgumentException("Invalid pool name");
        }
        this._poolName = poolName;
    }

    public String getPoolName() {
        return this._poolName;
    }

    @Override
    public synchronized Collection<PnfsId> list() {
        List<PnfsId> files = this._fileStore.list();
        Collection<PnfsId> records = this._metaDataStore.list();
        records.removeAll(new HashSet<PnfsId>(files));
        for (PnfsId id : records) {
            _log.warn(String.format(REMOVING_REDUNDANT_META_DATA, id));
            this._metaDataStore.remove(id);
        }
        return files;
    }

    @Override
    public MetaDataRecord get(PnfsId id) throws IllegalArgumentException, CacheException, InterruptedException {
        File file = this._fileStore.get(id);
        if (!file.isFile()) {
            return null;
        }
        MetaDataRecord entry = this._metaDataStore.get(id);
        if (this.isBroken(entry)) {
            _log.warn(String.format(RECOVERING_MSG, id));
            if (entry == null) {
                entry = this._metaDataStore.create(id);
                _log.warn(String.format(MISSING_MSG, id));
            }
            try {
                switch (entry.getState()) {
                    case FROM_STORE: 
                    case REMOVED: 
                    case DESTROYED: {
                        this.delete(id, file);
                        _log.info(String.format(PARTIAL_FROM_TAPE_MSG, id));
                        return null;
                    }
                }
                entry = this.rebuildEntry(entry);
            }
            catch (IOException e) {
                throw new DiskErrorCacheException("I/O error in healer: " + e.getMessage());
            }
            catch (CacheException e) {
                switch (e.getRc()) {
                    case 10001: {
                        this.delete(id, file);
                        _log.warn(String.format(FILE_NOT_FOUND_MSG, id));
                        return null;
                    }
                    case 10006: {
                        throw e;
                    }
                }
                entry.setState(EntryState.BROKEN);
                _log.error(String.format(BAD_MSG, id, e.getMessage()));
            }
        }
        return entry;
    }

    private boolean isBroken(MetaDataRecord entry) throws CacheException {
        return entry == null || entry.getStorageInfo() == null || entry.getStorageInfo().getFileSize() != entry.getSize() || entry.getState() != EntryState.CACHED && entry.getState() != EntryState.PRECIOUS;
    }

    private MetaDataRecord rebuildEntry(MetaDataRecord entry) throws CacheException, InterruptedException, IOException {
        PnfsId id = entry.getPnfsId();
        EntryState state = entry.getState();
        if (state == EntryState.BROKEN) {
            state = EntryState.FROM_CLIENT;
        }
        _log.warn(String.format(RECOVERING_FETCHED_STORAGE_INFO_FOR_1$S_FROM_PNFS, id));
        StorageInfo info = this._pnfsHandler.getStorageInfoByPnfsId(id).getStorageInfo();
        entry.setStorageInfo(info);
        long length = entry.getDataFile().length();
        if ((state != EntryState.FROM_CLIENT || info.getFileSize() != 0L) && info.getFileSize() != length) {
            throw new CacheException(String.format(BAD_SIZE_MSG, id, info.getFileSize(), length));
        }
        if (this._checksumModule != null) {
            ChecksumFactory factory = this._checksumModule.getDefaultChecksumFactory();
            this._checksumModule.setMoverChecksums(id, entry.getDataFile(), factory, null, null);
        }
        FileAttributes fileAttributes = new FileAttributes();
        fileAttributes.setLocations(Collections.singleton(this._poolName));
        if (state == EntryState.FROM_CLIENT && info.getFileSize() == 0L) {
            fileAttributes.setSize(length);
            fileAttributes.setAccessLatency(info.getAccessLatency());
            fileAttributes.setRetentionPolicy(info.getRetentionPolicy());
            info.setFileSize(length);
            entry.setStorageInfo(info);
            _log.warn(String.format(UPDATE_SIZE_MSG, id, length));
        }
        this._pnfsHandler.setFileAttributes(id, fileAttributes);
        if (state != EntryState.CACHED && state != EntryState.PRECIOUS) {
            EntryState targetState = this._replicaStatePolicy.getTargetState(info);
            List<StickyRecord> stickyRecords = this._replicaStatePolicy.getStickyRecords(info);
            for (StickyRecord record : stickyRecords) {
                entry.setSticky(record.owner(), record.expire(), false);
            }
            entry.setState(targetState);
            _log.warn(String.format(MARKED_MSG, new Object[]{id, targetState}));
        }
        return entry;
    }

    @Override
    public MetaDataRecord create(PnfsId id) throws DuplicateEntryException, CacheException {
        MetaDataRecord entry;
        File dataFile;
        if (_log.isInfoEnabled()) {
            _log.info("Creating new entry for " + id);
        }
        if ((dataFile = this._fileStore.get(id)).exists()) {
            _log.warn("Entry already exists: " + id);
            throw new FileInCacheException("Entry already exists: " + id);
        }
        try {
            entry = this._metaDataStore.create(id);
        }
        catch (DuplicateEntryException e) {
            _log.warn("Deleting orphaned meta data entry for " + id);
            this._metaDataStore.remove(id);
            try {
                entry = this._metaDataStore.create(id);
            }
            catch (DuplicateEntryException f) {
                throw new RuntimeException("Unexpected repository error", e);
            }
        }
        return entry;
    }

    @Override
    public MetaDataRecord create(MetaDataRecord entry) throws DuplicateEntryException, CacheException {
        return this._metaDataStore.create(entry);
    }

    @Override
    public void remove(PnfsId id) {
        File f = this._fileStore.get(id);
        if (!f.delete() && f.exists()) {
            _log.error("Failed to delete {}", (Object)f);
            throw new RuntimeException("Failed to delete " + id + " on " + this._poolName);
        }
        this._metaDataStore.remove(id);
    }

    @Override
    public boolean isOk() {
        return this._fileStore.isOk() && this._metaDataStore.isOk();
    }

    @Override
    public void close() {
        this._metaDataStore.close();
    }

    public String toString() {
        return String.format("[data=%s;meta=%s]", this._fileStore, this._metaDataStore);
    }

    @Override
    public long getFreeSpace() {
        return this._metaDataStore.getFreeSpace();
    }

    @Override
    public long getTotalSpace() {
        return this._metaDataStore.getTotalSpace();
    }

    private void delete(PnfsId id, File file) {
        this._metaDataStore.remove(id);
        if (!file.delete() && file.exists()) {
            _log.error("Failed to delete {}", (Object)file);
        }
        this._pnfsHandler.clearCacheLocation(id);
    }
}

