/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.boot;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import dmg.util.Args;
import dmg.util.CommandException;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.Objects;
import java.util.Set;
import java.util.logging.LogManager;
import org.dcache.boot.DebugLayoutPrinter;
import org.dcache.boot.Domain;
import org.dcache.boot.Layout;
import org.dcache.boot.LayoutPrinter;
import org.dcache.boot.PythonOracleLayoutPrinter;
import org.dcache.boot.ShellOracleLayoutPrinter;
import org.dcache.boot.XmlEntityLayoutPrinter;
import org.dcache.util.ConfigurationProperties;
import org.slf4j.LoggerFactory;
import org.slf4j.bridge.SLF4JBridgeHandler;

public class BootLoader {
    private static final String CMD_START = "start";
    private static final String CMD_COMPILE = "compile";
    private static final String CMD_COMPILE_OP_SHELL = "shell";
    private static final String CMD_COMPILE_OP_PYTHON = "python";
    private static final String CMD_COMPILE_OP_XML = "xml";
    private static final String CMD_COMPILE_OP_DEBUG = "debug";
    private static final String CMD_CHECK = "check-config";
    private static final char OPT_SILENT = 'q';
    private static Set<File> _sourceFiles = Sets.newHashSet();
    private static Set<File> _sourceDirectories = Sets.newHashSet();
    private static final Function<File, String> QUOTE_FILE = new Function<File, String>(){

        public String apply(File input) {
            return '\"' + input.getPath() + '\"';
        }
    };

    private BootLoader() {
    }

    private static void help() {
        System.err.println("SYNOPSIS:");
        System.err.println("  java org.dcache.util.BootLoader [-q] COMMAND [ARGS]");
        System.err.println();
        System.err.println("OPTIONS:");
        System.err.println("    -q    Suppress warnings and errors.");
        System.err.println();
        System.err.println("COMMANDS:");
        System.err.println("   start DOMAIN");
        System.err.println("          Start a domain.");
        System.err.println();
        System.err.println("   check-config");
        System.err.println("          Check configuration for any problems.");
        System.err.println();
        System.err.println("   compile <format>");
        System.err.println("          Compiles the layout to some particular format, determined by <format>.");
        System.err.println("          Valid values of <format> are:");
        System.err.println("                  -shell POSIX shell declaration of an oracle function");
        System.err.println("                  -python Python declaration of an oracle function");
        System.err.println("                  -xml an set of XML entity definitions");
        System.err.println("                  -debug a format providing human-readable format");
        System.exit(1);
    }

    private static ConfigurationProperties loadSystemProperties() throws UnknownHostException {
        ConfigurationProperties config = new ConfigurationProperties(System.getProperties());
        InetAddress localhost = InetAddress.getLocalHost();
        config.setProperty("host.name", localhost.getHostName().split("\\.")[0]);
        config.setProperty("host.fqdn", localhost.getCanonicalHostName());
        return config;
    }

    private static ConfigurationProperties loadConfiguration(ConfigurationProperties config, File path) throws IOException {
        config = new ConfigurationProperties(config);
        if (path.isFile()) {
            _sourceFiles.add(path);
            config.loadFile(path);
        } else if (path.isDirectory()) {
            _sourceDirectories.add(path);
            Object[] files = path.listFiles();
            if (files != null) {
                Arrays.sort(files);
                for (Object file : files) {
                    if (!((File)file).isFile() || !((File)file).getName().endsWith(".properties")) continue;
                    _sourceFiles.add((File)file);
                    config.loadFile((File)file);
                }
            }
        }
        return config;
    }

    private static ConfigurationProperties loadConfiguration(ConfigurationProperties config, String property) throws IOException {
        String paths = config.getValue(property);
        if (paths != null) {
            for (String path : paths.split(":")) {
                config = BootLoader.loadConfiguration(config, new File(path));
            }
        }
        return config;
    }

    private static ConfigurationProperties loadPlugins(ConfigurationProperties config, File directory) throws IOException {
        _sourceDirectories.add(directory);
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (!file.isDirectory()) continue;
                config = BootLoader.loadConfiguration(config, file);
            }
        }
        return config;
    }

    private static ConfigurationProperties loadConfiguration(ConfigurationProperties.ProblemConsumer consumer) throws UnknownHostException, IOException, URISyntaxException {
        ConfigurationProperties config = BootLoader.loadSystemProperties();
        config.setProblemConsumer(consumer);
        config = BootLoader.loadConfiguration(config, "dcache.paths.defaults");
        for (String dir : BootLoader.getPluginDirs()) {
            config = BootLoader.loadPlugins(config, new File(dir));
        }
        config = BootLoader.loadConfiguration(config, "dcache.paths.setup");
        return config;
    }

    private static Layout loadLayout(ConfigurationProperties config) throws IOException, URISyntaxException {
        String path = config.getValue("dcache.layout.uri");
        if (path == null) {
            throw new IOException("Undefined property: dcache.layout.uri");
        }
        URI uri = new URI(path);
        Layout layout = new Layout(config);
        layout.load(uri);
        if (Objects.equals(uri.getScheme(), "file")) {
            _sourceFiles.add(new File(uri.getPath()));
        } else {
            layout.properties().setProperty("dcache.config.cache", "false");
        }
        layout.properties().setProperty("dcache.config.files", Joiner.on((String)" ").join(Iterables.transform(_sourceFiles, QUOTE_FILE)));
        layout.properties().setProperty("dcache.config.dirs", Joiner.on((String)" ").join(Iterables.transform(_sourceDirectories, QUOTE_FILE)));
        return layout;
    }

    private static String[] getPluginDirs() throws IOException, URISyntaxException {
        ConfigurationProperties config = BootLoader.loadSystemProperties();
        OutputStreamProblemConsumer silentConsumer = new OutputStreamProblemConsumer(new ByteArrayOutputStream());
        config.setProblemConsumer(silentConsumer);
        config = BootLoader.loadConfiguration(config, "dcache.paths.defaults");
        config = BootLoader.loadConfiguration(config, "dcache.paths.setup");
        config = BootLoader.loadLayout(config).properties();
        String dir = config.getValue("dcache.paths.plugins");
        return dir == null ? new String[]{} : dir.split(":");
    }

    public static void main(String[] arguments) {
        String command = null;
        try {
            Args args = new Args(arguments);
            if (args.argc() < 1) {
                BootLoader.help();
            }
            command = args.argv(0);
            LogManager.getLogManager().reset();
            SLF4JBridgeHandler.install();
            OutputStreamProblemConsumer checkConsumer = new OutputStreamProblemConsumer(System.out);
            ConfigurationProperties.DefaultProblemConsumer problemConsumer = command.equals(CMD_CHECK) ? checkConsumer : new ConfigurationProperties.DefaultProblemConsumer();
            problemConsumer = args.isOneCharOption('q') ? new ErrorsAsWarningsProblemConsumer(problemConsumer) : problemConsumer;
            Layout layout = BootLoader.loadLayout(BootLoader.loadConfiguration(problemConsumer));
            switch (command) {
                case "start": {
                    if (args.argc() != 2) {
                        throw new IllegalArgumentException("Missing argument: Domain name");
                    }
                    Domain domain = layout.getDomain(args.argv(1));
                    if (domain == null) {
                        throw new IllegalArgumentException("No such domain: " + args.argv(1));
                    }
                    domain.start();
                    break;
                }
                case "check-config": {
                    checkConsumer.printSummary();
                    System.exit(checkConsumer.getReturnCode());
                }
                case "compile": {
                    LayoutPrinter printer = BootLoader.printerForArgs(args, layout);
                    printer.print(System.out);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid command: " + command);
                }
            }
        }
        catch (CommandException | FileNotFoundException | IllegalArgumentException | URISyntaxException e) {
            BootLoader.handleFatalError(e.getMessage(), command);
        }
        catch (IOException e) {
            BootLoader.handleFatalError(e.toString(), command);
        }
        catch (RuntimeException e) {
            BootLoader.handleFatalError(e, command);
        }
    }

    private static void handleFatalError(Object message, String command) {
        String logMessage;
        if (message instanceof RuntimeException) {
            logMessage = "Unexpected failure at startup; this is probably a bug";
            ((RuntimeException)message).printStackTrace();
        } else {
            logMessage = "Failure at startup: {}";
            System.err.println(message);
        }
        if (CMD_START.equals(command)) {
            LoggerFactory.getLogger((String)"root").error(logMessage, message);
        }
        System.exit(1);
    }

    private static LayoutPrinter printerForArgs(Args args, Layout layout) {
        boolean compileForPython;
        boolean compileForShell = args.hasOption(CMD_COMPILE_OP_SHELL);
        boolean compileForXml = args.hasOption(CMD_COMPILE_OP_XML);
        boolean compileForDebug = args.hasOption(CMD_COMPILE_OP_DEBUG);
        if ((compileForShell ? 1 : 0) + ((compileForPython = args.hasOption(CMD_COMPILE_OP_PYTHON)) ? 1 : 0) + (compileForXml ? 1 : 0) + (compileForDebug ? 1 : 0) != 1) {
            throw new IllegalArgumentException("Must specify exactly one of -shell, -python, -xml and -debug");
        }
        if (compileForShell) {
            return new ShellOracleLayoutPrinter(layout);
        }
        if (compileForPython) {
            return new PythonOracleLayoutPrinter(layout);
        }
        if (compileForXml) {
            return new XmlEntityLayoutPrinter(layout, new String[0]);
        }
        return new DebugLayoutPrinter(layout);
    }

    private static class OutputStreamProblemConsumer
    extends ConfigurationProperties.DefaultProblemConsumer {
        private int _errors;
        private int _warnings;
        private final PrintStream _out;

        public OutputStreamProblemConsumer(OutputStream out) {
            this._out = new PrintStream(out);
        }

        @Override
        public void error(String message) {
            this._out.println("[ERROR] " + this.addContextTo(message));
            ++this._errors;
        }

        @Override
        public void warning(String message) {
            this._out.println("[WARNING] " + this.addContextTo(message));
            ++this._warnings;
        }

        public int getReturnCode() {
            if (this._errors > 0) {
                return 2;
            }
            if (this._warnings > 0) {
                return 1;
            }
            return 0;
        }

        public void printSummary() {
            if (this._warnings == 0 && this._errors == 0) {
                System.out.println("No problems found.");
            } else {
                System.out.println(this.buildProblemsMessage());
            }
        }

        private String buildProblemsMessage() {
            StringBuilder sb = new StringBuilder();
            sb.append("Found ");
            this.cardinalMessage(sb, this._errors, "error");
            if (this._warnings > 0 && this._errors > 0) {
                sb.append(" and ");
            }
            this.cardinalMessage(sb, this._warnings, "warning");
            sb.append(".");
            return sb.toString();
        }

        private void cardinalMessage(StringBuilder sb, int count, String label) {
            switch (count) {
                case 0: {
                    break;
                }
                case 1: {
                    sb.append("1 ").append(label);
                    break;
                }
                default: {
                    sb.append(count).append(" ").append(label).append("s");
                }
            }
        }
    }

    private static class ErrorsAsWarningsProblemConsumer
    implements ConfigurationProperties.ProblemConsumer {
        ConfigurationProperties.ProblemConsumer _inner;

        ErrorsAsWarningsProblemConsumer(ConfigurationProperties.ProblemConsumer inner) {
            this._inner = inner;
        }

        @Override
        public void setFilename(String name) {
            this._inner.setFilename(name);
        }

        @Override
        public void setLineNumberReader(LineNumberReader reader) {
            this._inner.setLineNumberReader(reader);
        }

        @Override
        public void error(String message) {
            this._inner.warning(message);
        }

        @Override
        public void warning(String message) {
            this._inner.warning(message);
        }
    }
}

