/*
 * Decompiled with CFR 0.152.
 */
package org.dcache.gplazma.monitor;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.Principal;
import java.security.PublicKey;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.Stack;
import java.util.concurrent.TimeUnit;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Sequence;
import org.dcache.auth.FQAN;
import org.dcache.gplazma.configuration.ConfigurationItemControl;
import org.dcache.gplazma.monitor.LoginMonitor;
import org.dcache.gplazma.monitor.LoginResult;
import org.dcache.utils.Bytes;
import org.glite.voms.ac.AttributeCertificate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.security.rsa.RSAPublicKeyImpl;

public class LoginResultPrinter {
    private static final Logger _log = LoggerFactory.getLogger(LoginResultPrinter.class);
    private static final String ATTRIBUTE_CERTIFICATE_OID = "1.3.6.1.4.1.8005.100.100.5";
    private static final ImmutableMap<String, String> OID_TO_NAME = new ImmutableMap.Builder().put((Object)"1.2.840.113549.1.1.4", (Object)"MD5 with RSA").put((Object)"1.2.840.113549.1.1.5", (Object)"SHA-1 with RSA").put((Object)"1.2.840.113549.1.1.11", (Object)"SHA-256 with RSA").put((Object)"2.16.840.1.101.3.4.2.1", (Object)"SHA-256").put((Object)"2.16.840.1.101.3.4.2.2", (Object)"SHA-384").put((Object)"2.16.840.1.101.3.4.2.3", (Object)"SHA-512").put((Object)"1.3.6.1.5.5.7.3.1", (Object)"SSL server").put((Object)"1.3.6.1.5.5.7.3.2", (Object)"SSL client").put((Object)"1.3.6.1.5.5.7.3.3", (Object)"code signing").put((Object)"1.3.6.1.5.5.7.3.4", (Object)"email protection").put((Object)"1.3.6.1.5.5.7.3.5", (Object)"IPSec end system").put((Object)"1.3.6.1.5.5.7.3.6", (Object)"IPSec tunnel").put((Object)"1.3.6.1.5.5.7.3.7", (Object)"IPSec user").put((Object)"1.3.6.1.5.5.7.3.8", (Object)"time stamp").put((Object)"1.3.6.1.5.5.7.3.9", (Object)"OCSP signing").put((Object)"1.3.6.1.4.1.311.10.3.4", (Object)"Microsoft EPS").build();
    private static final ImmutableList<String> BASIC_KEY_USAGE_LABELS = new ImmutableList.Builder().add((Object)"digital signature").add((Object)"non-repudiation").add((Object)"key encipherment").add((Object)"data encipherment").add((Object)"key agreement").add((Object)"key certificate signing").add((Object)"CRL signing").add((Object)"encipher only").add((Object)"decipher only").build();
    private static final Collection<ConfigurationItemControl> ALWAYS_OK = EnumSet.of(ConfigurationItemControl.OPTIONAL, ConfigurationItemControl.SUFFICIENT);
    private final LoginResult _result;
    private StringBuilder _sb;

    public LoginResultPrinter(LoginResult result) {
        this._result = result;
    }

    public String print() {
        this._sb = new StringBuilder();
        this.printInitialPart();
        this.printPhase(this._result.getAuthPhase());
        this.printPhase(this._result.getMapPhase());
        this.printPhase(this._result.getAccountPhase());
        this.printPhase(this._result.getSessionPhase());
        this.printValidation();
        return this._sb.toString();
    }

    private void printInitialPart() {
        LoginMonitor.Result result = this.getOverallResult();
        this._sb.append("LOGIN ").append(this.stringFor(result)).append("\n");
        this.printLines(" in", this.buildInLines());
        this.printLines("out", this.buildOutItems());
        this._sb.append(" |\n");
    }

    private void printLines(String label, List<String> lines) {
        boolean isFirst = true;
        for (String line : lines) {
            this._sb.append(" |   ");
            if (isFirst) {
                this._sb.append(label).append(": ");
            } else {
                this._sb.append("     ");
            }
            this._sb.append(line).append("\n");
            isFirst = false;
        }
    }

    private List<String> buildInLines() {
        String display;
        LinkedList<String> lines = new LinkedList<String>();
        for (Principal principal : this.initialPrincipals()) {
            lines.add(principal.toString());
        }
        for (Object credential : this.publicCredentials()) {
            display = this.print(credential);
            for (String line : Splitter.on((char)'\n').split((CharSequence)display)) {
                lines.add(line);
            }
        }
        for (Object credential : this.privateCredentials()) {
            display = this.print(credential);
            for (String line : Splitter.on((char)'\n').split((CharSequence)display)) {
                lines.add(line);
            }
        }
        return lines;
    }

    private String print(Object credential) {
        if (credential instanceof X509Certificate[]) {
            return this.print((X509Certificate[])credential);
        }
        return credential.toString();
    }

    private String print(X509Certificate[] certificates) {
        StringBuilder sb = new StringBuilder();
        sb.append("X509 Certificate chain:\n");
        int i = 1;
        for (X509Certificate certificate : certificates) {
            boolean isLastCertificate = i == certificates.length;
            sb.append("  |\n");
            String certDetails = this.print(certificate);
            boolean isFirstLine = true;
            for (String line : Splitter.on((char)'\n').omitEmptyStrings().split((CharSequence)certDetails)) {
                if (isFirstLine) {
                    sb.append("  +--");
                } else if (!isLastCertificate) {
                    sb.append("  |  ");
                } else {
                    sb.append("     ");
                }
                sb.append(line).append('\n');
                isFirstLine = false;
            }
            ++i;
        }
        return sb.toString();
    }

    private String print(X509Certificate certificate) {
        StringBuilder sb = new StringBuilder();
        String subject = certificate.getSubjectX500Principal().getName("RFC2253");
        String issuer = certificate.getIssuerX500Principal().getName("RFC2253");
        boolean isSelfSigned = subject.equals(issuer);
        sb.append(String.format("%s [%d]", subject, certificate.getSerialNumber()));
        if (isSelfSigned) {
            sb.append(" (self-signed)");
        }
        sb.append('\n');
        sb.append("  |\n");
        if (!isSelfSigned) {
            sb.append("  +--Issuer: ").append(issuer).append('\n');
        }
        sb.append("  +--Validity: ").append(this.validityStatementFor(certificate)).append('\n');
        sb.append("  +--Algorithm: ").append(LoginResultPrinter.nameForOid(certificate.getSigAlgOID())).append('\n');
        sb.append("  +--Public key: ").append(LoginResultPrinter.describePublicKey(certificate.getPublicKey())).append('\n');
        String sanInfo = LoginResultPrinter.subjectAlternateNameInfoFor(certificate);
        if (!sanInfo.isEmpty()) {
            sb.append("  +--Subject alternative names:");
            if (LoginResultPrinter.isSingleLine(sanInfo)) {
                sb.append(" ").append(sanInfo).append('\n');
            } else {
                sb.append('\n');
                for (String line : Splitter.on((char)'\n').omitEmptyStrings().split((CharSequence)sanInfo)) {
                    sb.append("  |      ").append(line).append('\n');
                }
            }
        }
        String vomsInfo = this.vomsInfoFor(certificate);
        String extendedKeyUsage = this.extendedKeyUsageFor(certificate);
        if (!vomsInfo.isEmpty()) {
            sb.append("  +--Attribute certificates:");
            if (LoginResultPrinter.isSingleLine(vomsInfo)) {
                sb.append(" ").append(vomsInfo).append('\n');
            } else {
                sb.append('\n');
                for (String line : Splitter.on((char)'\n').omitEmptyStrings().split((CharSequence)vomsInfo)) {
                    if (extendedKeyUsage.isEmpty()) {
                        sb.append("     ");
                    } else {
                        sb.append("  |  ");
                    }
                    sb.append(line).append('\n');
                }
            }
        }
        if (!extendedKeyUsage.isEmpty()) {
            sb.append("  +--Key usage: ").append(extendedKeyUsage).append('\n');
        }
        return sb.toString();
    }

    private static String describePublicKey(PublicKey key) {
        StringBuilder sb = new StringBuilder();
        sb.append(key.getAlgorithm());
        if (key instanceof RSAPublicKeyImpl) {
            int bits = ((RSAPublicKeyImpl)key).getModulus().bitLength() + 7 & 0xFFFFFFF8;
            sb.append(' ').append(bits).append(" bits");
        } else {
            sb.append(" (unknown ").append(key.getClass().getCanonicalName()).append(")");
        }
        return sb.toString();
    }

    private static String subjectAlternateNameInfoFor(X509Certificate certificate) {
        Collection<List<?>> nameLists;
        StringBuilder sb = new StringBuilder();
        try {
            nameLists = certificate.getSubjectAlternativeNames();
        }
        catch (CertificateParsingException e) {
            return "problem (" + e.getMessage() + ")";
        }
        if (nameLists == null) {
            return "";
        }
        boolean isFirst = true;
        for (List<?> nameList : nameLists) {
            int tag = (Integer)nameList.get(0);
            Object object = nameList.get(1);
            if (!isFirst) {
                sb.append('\n');
            }
            switch (tag) {
                case 0: {
                    byte[] bytes = (byte[])object;
                    sb.append("otherName: ").append(Bytes.toHexString((byte[])bytes));
                    break;
                }
                case 1: {
                    sb.append("email: ").append(object);
                    break;
                }
                case 2: {
                    sb.append("DNS: ").append(object);
                    break;
                }
                case 3: {
                    byte[] bytes = (byte[])object;
                    sb.append("x400: ").append(Bytes.toHexString((byte[])bytes));
                    break;
                }
                case 4: {
                    sb.append("dirName: ").append(object);
                    break;
                }
                case 5: {
                    byte[] bytes = (byte[])object;
                    sb.append("EDI party name: ").append(Bytes.toHexString((byte[])bytes));
                    break;
                }
                case 6: {
                    sb.append("URI: ").append(String.valueOf(object));
                    break;
                }
                case 7: {
                    sb.append("IP: ").append(String.valueOf(object));
                    break;
                }
                case 8: {
                    byte[] bytes = (byte[])object;
                    sb.append("oid: ").append(Bytes.toHexString((byte[])bytes));
                    break;
                }
                default: {
                    byte[] bytes = (byte[])object;
                    sb.append(object.getClass().getSimpleName());
                    sb.append(" [").append(tag).append("]");
                    sb.append(" ").append(Bytes.toHexString((byte[])bytes));
                }
            }
            isFirst = false;
        }
        return sb.toString();
    }

    private String vomsInfoFor(X509Certificate x509Certificate) {
        List<AttributeCertificate> certificates;
        try {
            certificates = LoginResultPrinter.extractAttributeCertificates(x509Certificate);
        }
        catch (IOException e) {
            return "problem (" + e.getMessage() + ")";
        }
        StringBuilder sb = new StringBuilder();
        int i = 1;
        for (AttributeCertificate certificate : certificates) {
            boolean isLastCertificate = i == certificates.size();
            String info = this.attributeCertificateInfoFor(certificate);
            boolean isFirstLine = true;
            for (String line : Splitter.on((char)'\n').omitEmptyStrings().split((CharSequence)info)) {
                if (isFirstLine) {
                    sb.append("  |\n");
                    sb.append("  +--");
                    isFirstLine = false;
                } else if (isLastCertificate) {
                    sb.append("     ");
                } else {
                    sb.append("  |  ");
                }
                sb.append(line).append('\n');
            }
            ++i;
        }
        return sb.toString();
    }

    private static List<AttributeCertificate> extractAttributeCertificates(X509Certificate certificate) throws IOException {
        ArrayList<AttributeCertificate> certificates = new ArrayList<AttributeCertificate>();
        byte[] payload = certificate.getExtensionValue(ATTRIBUTE_CERTIFICATE_OID);
        if (payload == null) {
            return Collections.emptyList();
        }
        payload = LoginResultPrinter.decodeEncapsulation(payload);
        ByteArrayInputStream in = new ByteArrayInputStream(payload);
        ASN1Sequence acSequence = (ASN1Sequence)new ASN1InputStream((InputStream)in).readObject();
        Enumeration e1 = acSequence.getObjects();
        while (e1.hasMoreElements()) {
            ASN1Sequence acSequence2 = (ASN1Sequence)e1.nextElement();
            Enumeration e2 = acSequence2.getObjects();
            while (e2.hasMoreElements()) {
                ASN1Sequence acSequence3 = (ASN1Sequence)e2.nextElement();
                try {
                    certificates.add(new AttributeCertificate(acSequence3));
                }
                catch (IOException e) {
                    _log.debug("Problem decoding AttributeCertificate: {}", (Object)e.getMessage());
                }
            }
        }
        return certificates;
    }

    private static byte[] decodeEncapsulation(byte[] payload) throws IOException {
        ASN1InputStream payloadStream = new ASN1InputStream((InputStream)new ByteArrayInputStream(payload));
        return ((ASN1OctetString)payloadStream.readObject()).getOctets();
    }

    private String attributeCertificateInfoFor(AttributeCertificate certificate) {
        StringBuilder sb = new StringBuilder();
        sb.append(certificate.getIssuerX509().getName()).append('\n');
        sb.append("  +--Validity: ").append(this.validityStatementFor(certificate)).append('\n');
        String oid = certificate.getSignatureAlgorithm().getObjectId().toString();
        sb.append("  +--Algorithm: ").append(LoginResultPrinter.nameForOid(oid)).append('\n');
        String fqanInfo = LoginResultPrinter.fqanInfoFor(certificate);
        if (!fqanInfo.isEmpty()) {
            sb.append("  +--FQANs: ").append(fqanInfo).append('\n');
        }
        return sb.toString();
    }

    private static String fqanInfoFor(AttributeCertificate certificate) {
        List fqans = certificate.getListOfFQAN();
        if (fqans.size() > 0) {
            StringBuilder sb = new StringBuilder();
            FQAN fqan = new FQAN(String.valueOf(fqans.get(0)));
            sb.append(fqan);
            if (fqans.size() > 1) {
                FQAN fqan2 = new FQAN(String.valueOf(fqans.get(1)));
                sb.append(", ").append(fqan2);
                if (fqans.size() > 2) {
                    sb.append(", ...");
                }
            }
            return sb.toString();
        }
        return "";
    }

    private static String nameForOid(String oid) {
        String name = (String)OID_TO_NAME.get((Object)oid);
        if (name == null) {
            name = oid;
        }
        return name;
    }

    private String extendedKeyUsageFor(X509Certificate certificate) {
        LinkedList<String> labels = new LinkedList<String>();
        try {
            List<String> extendedUses;
            boolean[] usageAllowed = certificate.getKeyUsage();
            if (usageAllowed != null) {
                int i = 0;
                for (String usageLabel : BASIC_KEY_USAGE_LABELS) {
                    if (usageAllowed[i]) {
                        labels.add(usageLabel);
                    }
                    ++i;
                }
            }
            if ((extendedUses = certificate.getExtendedKeyUsage()) != null) {
                for (String use : extendedUses) {
                    labels.add(LoginResultPrinter.nameForOid(use));
                }
            }
            return Joiner.on((String)", ").join(labels);
        }
        catch (CertificateParsingException e) {
            return "parsing problem (" + e.getMessage() + ")";
        }
    }

    private String validityStatementFor(X509Certificate certificate) {
        Date notBefore = certificate.getNotBefore();
        Date notAfter = certificate.getNotAfter();
        return this.validityStatementFor(notBefore, notAfter);
    }

    private String validityStatementFor(AttributeCertificate certificate) {
        try {
            Date notBefore = certificate.getNotBefore();
            Date notAfter = certificate.getNotAfter();
            return this.validityStatementFor(notBefore, notAfter);
        }
        catch (ParseException e) {
            return "problem parsing validity info (" + e.getMessage() + ")";
        }
    }

    private String validityStatementFor(Date notBefore, Date notAfter) {
        Date now = new Date();
        if (now.after(notAfter)) {
            return this.validityStatementFor("expired ", now.getTime() - notAfter.getTime(), " ago");
        }
        if (now.before(notBefore)) {
            return this.validityStatementFor("not yet, in ", notBefore.getTime() - now.getTime());
        }
        return this.validityStatementFor("OK for ", notAfter.getTime() - now.getTime());
    }

    private String validityStatementFor(String prefix, long milliseconds) {
        return this.validityStatementFor(prefix, milliseconds, "");
    }

    private String validityStatementFor(String prefix, long milliseconds, String suffix) {
        long days = TimeUnit.MILLISECONDS.toDays(milliseconds);
        long hours = TimeUnit.MILLISECONDS.toHours(milliseconds -= TimeUnit.DAYS.toMillis(days));
        long minutes = TimeUnit.MILLISECONDS.toMinutes(milliseconds -= TimeUnit.HOURS.toMillis(hours));
        long seconds = TimeUnit.MILLISECONDS.toSeconds(milliseconds -= TimeUnit.MINUTES.toMillis(minutes));
        StringBuilder sb = new StringBuilder();
        sb.append(prefix);
        Stack<String> phrases = new Stack<String>();
        if (seconds > 0L || (milliseconds -= TimeUnit.SECONDS.toMillis(seconds)) > 0L) {
            double secondsAndMillis = (double)seconds + (double)milliseconds / 1000.0;
            phrases.push(String.format("%.1f seconds", secondsAndMillis));
        }
        if (minutes > 0L) {
            phrases.push(this.buildPhrase(minutes, "minute"));
        }
        if (hours > 0L) {
            phrases.push(this.buildPhrase(hours, "hour"));
        }
        if (days > 0L) {
            phrases.push(this.buildPhrase(days, "day"));
        }
        while (!phrases.isEmpty()) {
            sb.append((String)phrases.pop());
            if (phrases.size() > 1) {
                sb.append(", ");
                continue;
            }
            if (phrases.size() != 1) continue;
            sb.append(" and ");
        }
        sb.append(suffix);
        return sb.toString();
    }

    private String buildPhrase(long count, String scalar) {
        if (count == 1L) {
            return "1 " + scalar;
        }
        return count + " " + scalar + "s";
    }

    private Set<Principal> initialPrincipals() {
        LoginResult.AuthPhaseResult auth = this._result.getAuthPhase();
        Set<Principal> principal = auth.hasHappened() ? auth.getPrincipals().getBefore() : Collections.emptySet();
        return principal;
    }

    private Set<Object> publicCredentials() {
        LoginResult.AuthPhaseResult auth = this._result.getAuthPhase();
        return auth.getPublicCredentials();
    }

    private Set<Object> privateCredentials() {
        LoginResult.AuthPhaseResult auth = this._result.getAuthPhase();
        return auth.getPrivateCredentials();
    }

    private List<String> buildOutItems() {
        LinkedList<String> labels = new LinkedList<String>();
        for (Principal principal : this.finalPrincipals()) {
            labels.add(principal.toString());
        }
        return labels;
    }

    private Set<Principal> finalPrincipals() {
        LoginResult.SessionPhaseResult session = this._result.getSessionPhase();
        Set<Principal> principals = session.hasHappened() ? session.getPrincipals().getAfter() : Collections.emptySet();
        return principals;
    }

    private <T extends LoginResult.PAMPluginResult> void printPhase(LoginResult.PhaseResult<T> result) {
        if (result.hasHappened()) {
            this._sb.append(String.format(" +--%s %s\n", result.getName(), this.stringFor(result.getResult())));
            this.printPrincipalsDiff(" |   |  ", result.getPrincipals());
            int count = result.getPluginResults().size();
            if (count > 0) {
                this._sb.append(" |   |\n");
                int index = 1;
                for (LoginResult.PAMPluginResult plugin : result.getPluginResults()) {
                    boolean isLast = index == count;
                    this.printPlugin(plugin, isLast);
                    ++index;
                }
            }
            this._sb.append(" |\n");
        } else {
            this._sb.append(" +--(").append(result.getName()).append(") skipped\n");
            this._sb.append(" |\n");
        }
    }

    private void printPlugin(LoginResult.PAMPluginResult result, boolean isLast) {
        this.printPluginHeader(result);
        if (result instanceof LoginResult.AuthPluginResult) {
            this.printPluginBehaviour((LoginResult.AuthPluginResult)result, isLast);
        } else if (result instanceof LoginResult.MapPluginResult) {
            this.printPluginBehaviour((LoginResult.MapPluginResult)result, isLast);
        } else if (result instanceof LoginResult.AccountPluginResult) {
            this.printPluginBehaviour((LoginResult.AccountPluginResult)result, isLast);
        } else if (result instanceof LoginResult.SessionPluginResult) {
            this.printPluginBehaviour((LoginResult.SessionPluginResult)result, isLast);
        } else {
            throw new IllegalArgumentException("unknown type of plugin result: " + result.getClass().getCanonicalName());
        }
        if (!isLast) {
            this._sb.append(" |   |\n");
        }
    }

    private void printPluginBehaviour(LoginResult.AuthPluginResult plugin, boolean isLast) {
        String prefix = isLast ? " |        " : " |   |    ";
        this.printPrincipalsDiff(prefix, plugin.getIdentified());
    }

    private void printPluginBehaviour(LoginResult.MapPluginResult plugin, boolean isLast) {
        String prefix = isLast ? " |        " : " |   |    ";
        this.printPrincipalsDiff(prefix, plugin.getPrincipals());
    }

    private void printPluginBehaviour(LoginResult.AccountPluginResult plugin, boolean isLast) {
        String prefix = isLast ? " |        " : " |   |    ";
        this.printPrincipalsDiff(prefix, plugin.getAuthorized());
    }

    private void printPluginBehaviour(LoginResult.SessionPluginResult plugin, boolean isLast) {
        String prefix = isLast ? " |        " : " |   |    ";
        this.printPrincipalsDiff(prefix, plugin.getAuthorized());
    }

    private void printValidation() {
        if (this._result.hasValidationHappened()) {
            LoginMonitor.Result result = this._result.getValidationResult();
            String label = this.stringFor(this._result.getValidationResult());
            this._sb.append(" +--VALIDATION ").append(label);
            if (result == LoginMonitor.Result.FAIL) {
                this._sb.append(" (").append(this._result.getValidationError()).append(")");
            }
            this._sb.append('\n');
        } else {
            this._sb.append(" +--(VALIDATION) skipped\n");
        }
    }

    private void printPluginHeader(LoginResult.PAMPluginResult plugin) {
        ConfigurationItemControl control = plugin.getControl();
        LoginMonitor.Result result = plugin.getResult();
        String resultLabel = this.stringFor(result);
        String name = plugin.getName();
        String error = result == LoginMonitor.Result.SUCCESS ? "" : " (" + plugin.getError() + ")";
        this._sb.append(String.format(" |   +--%s %s:%s%s => %s", name, plugin.getControl().name(), resultLabel, error, ALWAYS_OK.contains((Object)control) ? "OK" : resultLabel));
        if (result == LoginMonitor.Result.SUCCESS && control == ConfigurationItemControl.SUFFICIENT || result == LoginMonitor.Result.FAIL && control == ConfigurationItemControl.REQUISITE) {
            this._sb.append(" (ends the phase)");
        }
        this._sb.append('\n');
    }

    private LoginMonitor.Result getOverallResult() {
        LoginResult.AuthPhaseResult auth = this._result.getAuthPhase();
        LoginResult.MapPhaseResult map = this._result.getMapPhase();
        LoginResult.AccountPhaseResult account = this._result.getAccountPhase();
        LoginResult.SessionPhaseResult session = this._result.getSessionPhase();
        boolean success = auth.hasHappened() && auth.getResult() == LoginMonitor.Result.SUCCESS && map.hasHappened() && map.getResult() == LoginMonitor.Result.SUCCESS && account.hasHappened() && account.getResult() == LoginMonitor.Result.SUCCESS && session.hasHappened() && session.getResult() == LoginMonitor.Result.SUCCESS && this._result.getValidationResult() == LoginMonitor.Result.SUCCESS;
        return success ? LoginMonitor.Result.SUCCESS : LoginMonitor.Result.FAIL;
    }

    private String stringFor(LoginMonitor.Result result) {
        return result == LoginMonitor.Result.SUCCESS ? "OK" : "FAIL";
    }

    private void printPrincipalsDiff(String prefix, LoginResult.SetDiff<Principal> diff) {
        if (diff == null) {
            return;
        }
        Set<Principal> added = diff.getAdded();
        boolean isFirst = true;
        for (Principal principal : added) {
            if (isFirst) {
                this._sb.append(prefix).append("  added: ");
                isFirst = false;
            } else {
                this._sb.append(prefix).append("         ");
            }
            this._sb.append(principal).append('\n');
        }
        Set<Principal> removed = diff.getRemoved();
        isFirst = true;
        for (Principal principal : removed) {
            if (isFirst) {
                this._sb.append(prefix).append("removed: ");
                isFirst = false;
            } else {
                this._sb.append(prefix).append("         ");
            }
            this._sb.append(principal).append('\n');
        }
    }

    private static boolean isSingleLine(String data) {
        return !data.contains("\n");
    }
}

