import os, pipes, re, time
from datetime import datetime
from subprocess import Popen, PIPE
try:
    from subprocess import CalledProcessError
except ImportError:
    class CalledProcessError(OSError):
	def __init__(self, exitcode, cmd):
	    OSError.__init__(self, '%s exited with %d'%(cmd, exitcode))

terminal_job_states = \
    ['FINISHED', 'KILLED', 'FAILED', 'DELETED']
nonterminal_job_states = \
    ['ACCEPTING', 'ACCEPTED', 'EXECUTED', 'PREPARING',
     'SUBMITTING', 'SUBMITTED',
     'HOLD', 'QUEUING',
     'RUNNING', 'FINISHING']
all_job_states = terminal_job_states + nonterminal_job_states

class ParseError(Exception):
    pass

def _simple_cmd(argv, log = None):
    p = Popen(argv, stderr = PIPE)
    output, errors = p.communicate()
    if p.returncode != 0:
	if log:
	    log.error('Errors from %s:\n%s'%(argv[0], errors))
	raise CalledProcessError(p.returncode, ' '.join(argv))

def arccp(src_url, dst_url, log = None, timeout = 20):
    return _simple_cmd(['arccp', '-t', str(timeout), src_url, dst_url],
		       log = log)

def arcrm(dst_url, log = None, timeout = 20):
    return _simple_cmd(['arcrm', '-t', str(timeout), dst_url], log = log)

def arcls(url, log = None, timeout = 20):
    p = Popen(['arcls', '-t', str(timeout), '-l', url],
	      stdout = PIPE, stderr = PIPE)
    output, errors = p.communicate()
    entries = []
    for ln in output.split('\n')[1:]:
	if ln:
	    comps = ln.rsplit(None, 7)
	    entries.append(ArclsEntry(*comps))
    if p.returncode != 0:
	if log:
	    log.error('Errors from arcls:\n%s'%errors)
	raise CalledProcessError(p.returncode, 'arcls')
    return entries

def arcclean(jobids, force = False, timeout = 20, log = None):
    argv = ['arcclean', '-t', str(int(timeout))]
    if force:
	argv.append('-f')
    argv += jobids
    _simple_cmd(argv, log = log)

_arcstat_state_re = re.compile(r'\w+\s+\(([^()]+)\)')
def parse_arcstat_state(s):
    if True:
	state = s.split(' ', 1)[0].upper()
    else:
	if s.startswith('Queuing '):
	    return 'QUEUING'
	if s.startswith('Running '):
	    return 'RUNNING'
	mo = re.match(_arcstat_state_re, s)
	if not mo:
	    raise ParseError('Malformed arcstat state %s.'%s)
	state = mo.group(1)
    if not state in all_job_states:
	raise ParseError('Unexpected job state %s'%state)
    return state

_arcstat_conv = {
    'State': parse_arcstat_state,
}

def arcstat(*jobids):
    cmd = 'arcstat ' + ' '.join(map(pipes.quote, jobids))
    fd = os.popen(cmd, 'r')
    jobstats = {}
    lnno = 0

    def parse_error(msg):
	fd.close()
	raise ParseError('Unexpected output from arcstat '
			 'at line %d: %s'%(lnno, msg))
    def check_entry(jobid, jobstat):
	if not 'State' in jobstat:
	    raise ParseError('Missing "State" key for %s.'%jobid)

    jobid, jobstat = None, None
    for ln in fd:
	ln = ln.strip()
	lnno += 1

	if ln == '':
	    if not jobid is None:
		check_entry(jobid, jobstat)
		jobstats[jobid] = jobstat
		jobid, jobstat = None, None
	elif ln == 'No jobs':
	    break
	elif ln.startswith('Job: '):
	    if not jobid is None:
		return parse_error('Missing empty line before "Job: ..."')
	    jobid = ln[5:].strip()
	    jobstat = {}
	elif ln.startswith('Warning:'):
	    pass
	else:
	    if jobid is None:
		return parse_error('Missing "Job: ..." header before %r'%ln)
	    try:
		k, v = ln.split(':', 1)
		if k in _arcstat_conv:
		    jobstat[k] = _arcstat_conv[k](v.strip())
		else:
		    jobstat[k] = v.strip()
	    except ValueError:
		return parse_error('Expecting "<key>: <value>", got %r'%ln)
    fd.close()
    if not jobid is None:
	check_entry(jobstat)
	jobstats[jobid] = jobstat
    return jobstats

def arcprune(max_age = 604800, log = None, timeout = 20):
    t_start = time.time()
    pruned_count = 0
    failed_count = 0
    jobstats = arcstat('-al')
    for jobid, jobstat in jobstats.iteritems():
	if not 'Submitted' in jobstat:
	    continue
	submitted = jobstat['Submitted']
	tm_sub = time.strptime(submitted, '%Y-%m-%d %H:%M:%S')
	t_sub = time.mktime(tm_sub)
	t_left = timeout - time.time() + t_start
	if t_left < 0:
	    if log: log.warn('Timout while pruning jobs.')
	    break
	if t_start - t_sub > max_age:
	    try:
		if log: log.info('Cleaning %s from %s.'%(jobid, submitted))
		arcclean([jobid], log = log, timeout = t_left)
		pruned_count += 1
	    except Exception, xc:
		failed_count = 0
		if log: log.warn('Failed to prune %s: %s'%(jobid, xc))
    return len(jobstats), pruned_count, failed_count

class ArclsEntry(object):
    DIR = 0
    FILE = 1

    def __init__(self, name, typ, size, cdate, ctod, validity, checksum, latency):
	self.filename = name
	if typ == 'dir':
	    self.entry_type = self.DIR
	elif typ == 'file':
	    self.entry_type = self.FILE
	else:
	    self.entry_type = None
	self.size = size
#	Not in Python 2.4:
#	self.ctime = datetime.strptime(cdate + 'T' + ctod, '%Y-%m-%dT%H:%M:%S')
	def drop_NA(s):
	    if s != '(n/a)':
		return s
	self.validity = drop_NA(validity)
	self.checksum = drop_NA(checksum)
	self.latency = drop_NA(latency)
