#------------------------------------------------------------------------------
# Copyright 2008-2012 Istituto Nazionale di Fisica Nucleare (INFN)
# 
# Licensed under the EUPL, Version 1.1 only (the "Licence").
# You may not use this work except in compliance with the Licence. 
# You may obtain a copy of the Licence at:
# 
# http://joinup.ec.europa.eu/system/files/EN/EUPL%20v.1.1%20-%20Licence.pdf
# 
# Unless required by applicable law or agreed to in 
# writing, software distributed under the Licence is 
# distributed on an "AS IS" basis, 
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
# either express or implied. 
# See the Licence for the specific language governing 
# permissions and limitations under the Licence.
#------------------------------------------------------------------------------

import os
import stat
import ConfigParser
import exceptions

class WrongConfigurationFile(exceptions.Exception):
    pass

class WrongConfigurationSettings(exceptions.Exception):
    pass

class GeneralError(exceptions.Exception):
    pass

class LoadConfig(object):
    def __init__(self, config_file=''):
        self.config_file = config_file
        self.parser = ConfigParser.SafeConfigParser()
        self.candidates = {}
        self.load_data = {}

    def __are_options_present(self, section):
        '''Verify the correctness of the options'''
        for option in self.candidates[section].keys():
            if not self.parser.has_option(section, option):
                if option == 'user_cert':
                    if not self.parser.has_option(section, 'user_proxy'):
                        return False
                    else:
                        return True
                elif option == 'user_key':
                    if not self.parser.has_option(section, 'user_proxy'):
                        return False
                    else:
                        return True
                elif option == 'user_proxy':
                    if self.parser.has_option(section, 'user_cert') and self.parser.has_option(section, 'user_key'):
                        return True
                    else:
                        return False
                else:
                    return False

        return True

    def __are_sections_present(self):
        '''Verify the correctness of the sections'''
        for section in self.candidates.keys():
            if not self.parser.has_section(section):
                return False
            elif not self.__are_options_present(section):
                return False

        return True

    def load_data(self):
        '''Load data'''
        try:
            self.parser.read(self.config_file)
            if not self.__are_sections_present():
                raise WrongConfigurationFile('Wrong Configuration file %s. '
                    % self.config_file +
                    'Please verify the file.')

            for section in self.parser.sections():
                self.load_data[section] = {}
                for option in self.parser.options(section):
                    self.load_data[section][option] = self.parser.get(section,
                        option)

        except ConfigParser.ParsingError, err:
            print 'Could not parse: ', err
        except ConfigParser.NoOptionError, err:
            print 'No such option: ', str(err)
        except ConfigParser.NoSectionError, err:
            print 'No such section: ', str(err)

        return self.load_data

class LoadUserConfig(LoadConfig):
    def __init__(self, config_file=''):
        super(LoadUserConfig, self).__init__(config_file=config_file)
        self.candidates = UserConfigCommand().get_candidates()
        self.candidates_user_to_check = ['user_key',
            'user_cert',
            'user_proxy',
            'pub_key']
        self.candidates_server_to_check = ['endpoint',
            'port',
            'ca_path']
        self.candidates_infrastructure_to_check = ['category']
        self.candidates_content_to_check = ['type']
        self.candidates_ns_to_check = ['ns_host', 'ns_port']

    def __is_valid(self, option, value='test'):
        '''Verify the correctness of user configuration file settings'''
        if option == 'ca_path':
            if not os.path.isdir(value):
                return False
        elif option == 'user_key':
            #print 'is_valid', option
            if not (os.path.isfile(value)
                    and stat.S_IMODE(os.stat(value).st_mode) == 0400):
                return False
        elif option == 'user_cert':
            if not (os.path.isfile(value)
                    and stat.S_IMODE(os.stat(value).st_mode) == 0600):
                return False
        elif option == 'user_proxy':
            #print 'is_valid', option
            if not (os.path.isfile(value)
                    and stat.S_IMODE(os.stat(value).st_mode) == 0600):
                return False
        elif option == 'pub_key':
            if not (os.path.isfile(value)
                    and stat.S_IMODE(os.stat(value).st_mode) == 0644):
                return False
        elif option == 'type':
            if value not in ('text/plain', 'text/occi'):
                return False
        elif option == 'category':
            if value not in ('compute','storage','network'):
                return False
        #print 'yes'
        return True

    def load_user_data(self):
        '''Load user data'''

        load = LoadConfig.load_data(self)
        for option in self.candidates_user_to_check:
            try:
                load['user'][option]
            except KeyError:
                if option == 'user_key':
                    if 'user_proxy' in load['user'].keys():
                         continue
                elif option == 'user_cert':
                    if 'user_proxy' in load['user'].keys():
                         continue
                elif option == 'user_proxy':
                    if 'user_cert' in load['user'].keys() and 'user_key' in load['user'].keys():
                         continue
            if not self.__is_valid(option, load['user'][option]):
                raise WrongConfigurationSettings(\
                    'Wrong Configuration Settings for %s' % option)

        for option in self.candidates_server_to_check:
            if not self.__is_valid(option, load['server'][option]):
                raise WrongConfigurationSettings(\
                    'Wrong Configuration Settings for %s' % option)

        for option in self.candidates_infrastructure_to_check:
            if not self.__is_valid(option, load['infrastructure'][option]):
                raise WrongConfigurationSettings(\
                    'Wrong Configuration Settings for %s' % option)

        for option in self.candidates_content_to_check:
            if not self.__is_valid(option, load['content'][option]):
                raise WrongConfigurationSettings(\
                    'Wrong Configuration Settings for %s' % option)

        for option in self.candidates_ns_to_check:
            if not self.__is_valid(option, load['nameserver'][option]):
                raise WrongConfigurationSettings(\
                    'Wrong Configuration Settings for %s' % option)

        return load

class UserConfigCommand(object):
    def __init__(self):
        self.candidates = {
            'user': {'vo': '',
                'user_key': '',
                'user_cert': '',
                'user_proxy': ''},
            'server': {'endpoint': '',
                'port': '',
                'ca_path':''},
            'content': {'type': ''},
            'infrastructure': {'category':''},
            'nameserver':{'ns_host':'',
                'ns_port':8219}
            }
        self.cmd_options = {'--capath': '',
            '--cert': '',
            '--key': ''}

    def get_candidates(self):
        '''Returns user configuration options'''

        return self.candidates

    def get_map(self):
        '''Returns a map between user configuration options and command'''

        map_value = {}

        for option in self.cmd_options.keys():
            if option == '--key':
                map_value['user_key'] = option
                map_value['user_proxy'] = option
            elif option == '--cert':
                map_value['user_cert'] = option
                map_value['user_proxy'] = option
            elif option == '--capath':
                map_value['ca_path'] = option
            

        return map_value
