package de.fzj.unicore.uas.fts.uftp;

import static org.junit.Assert.assertEquals;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import junit.framework.Assert;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.CreationFlagEnumeration;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.DataStagingType;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionDocument;
import org.ggf.schemas.jsdl.x2005.x11.jsdl.JobDefinitionType;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.unigrids.services.atomic.types.GridFileType;
import org.unigrids.services.atomic.types.ProtocolType;
import org.unigrids.x2006.x04.services.tss.SubmitDocument;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.client.FileTransferClient;
import de.fzj.unicore.uas.client.HttpFileTransferClient;
import de.fzj.unicore.uas.client.JobClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.StorageFactoryClient;
import de.fzj.unicore.uas.client.TSFClient;
import de.fzj.unicore.uas.client.TSSClient;
import de.fzj.unicore.uas.client.UFTPConstants;
import de.fzj.unicore.uas.client.UFTPFileTransferClient;
import de.fzj.unicore.wsrflite.Kernel;
import eu.unicore.bugsreporter.annotation.FunctionalTest;
import eu.unicore.uftp.server.UFTPServer;

/**
 * Tests the UFTP integration into the XNJS
 */
public class TestUFTP {

	// job port
	static int jobPort = 62434;

	static int srvPort = 62435;

	static InetAddress host;

	static StorageClient sms;

	static TSSClient tss;

	static Kernel kernel;
	
	protected static String getConfigPath() {
		return "src/test/resources/uas.config";
	}

	@AfterClass
	public static void shutdown() throws Exception {
		stopUFTPDServer();
		kernel.shutdown();
	}

	@BeforeClass
	public static void init() throws Exception {
		host = InetAddress.getByName("localhost");
		startUFTPDServer();

		// start UNICORE
		long start = System.currentTimeMillis();
		// clear data directories
		FileUtils.deleteQuietly(new File("target", "data"));
		FileUtils.deleteQuietly(new File("target", "testfiles"));
		File f = new File("target/testfiles");
		if (!f.exists())f.mkdirs();
		UAS uas = new UAS(getConfigPath());
		uas.startSynchronous();
		System.out.println("Startup time: "
				+ (System.currentTimeMillis() - start) + " ms.");
		kernel = uas.getKernel();
		kernel.getAttribute(UASProperties.class).setProperty(UASProperties.SMS_TRANSFER_FORCEREMOTE, "true");
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		cfg.setProperty(UFTPProperties.PARAM_CLIENT_LOCAL, "true");
		cfg.setProperty(UFTPProperties.PARAM_CLIENT_HOST, "localhost");
		cfg.setProperty(UFTPProperties.PARAM_SERVER_HOST, "localhost");
		cfg.setProperty(UFTPProperties.PARAM_SERVER_PORT, ""+srvPort);
		cfg.setProperty(UFTPProperties.PARAM_COMMAND_PORT, ""+jobPort);
		cfg.setProperty(UFTPProperties.PARAM_COMMAND_HOST, "localhost");
		cfg.setProperty(UFTPProperties.PARAM_COMMAND_SSL_DISABLE, "true");
		// create a storage
		EndpointReferenceType epr = EndpointReferenceType.Factory.newInstance();
		epr.addNewAddress().setStringValue("http://localhost:65321/services/StorageFactory?res=default_storage_factory");
		StorageFactoryClient smf = new StorageFactoryClient(epr, kernel.getClientConfiguration());
		sms = smf.createSMS();
		importTestFile(sms, "test", 16384);
		EndpointReferenceType epr2 = EndpointReferenceType.Factory.newInstance();
		epr2.addNewAddress().setStringValue(
						"http://localhost:65321/services/TargetSystemFactoryService?res=default_target_system_factory");
		TSFClient tsf = new TSFClient(epr2, kernel.getClientConfiguration());
		tss = tsf.createTSS();
	}

	@FunctionalTest(id = "testUFTPStageIn", description = "Tests file staging in using UFTP")
	@Test
	public void testStageIn() throws Exception {
		doStageIn(false);
	}

	@FunctionalTest(id = "testUFTPStageInEncrypt", description = "Tests file staging in using UFTP with encrypted data")
	@Test
	public void testStageInEncrypt() throws Exception {
		doStageIn(true);
	}

	private void doStageIn(boolean encrypt) throws Exception {
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		cfg.setProperty(UFTPProperties.PARAM_ENABLE_ENCRYPTION, "true");
		SubmitDocument in = SubmitDocument.Factory.newInstance();
		in.addNewSubmit().setJobDefinition(getStageInJob());
		JobClient jc = tss.submit(in);
		jc.waitUntilReady(60000);
		jc.start();
		jc.waitUntilDone(15000);
		GridFileType result = jc.getUspaceClient().listProperties(
				"test-staged-in");
		Assert.assertEquals(16384, result.getSize());
		cfg.setProperty(UFTPProperties.PARAM_ENABLE_ENCRYPTION, "false");
	}
	
	@FunctionalTest(id = "testUFTPStageOut", description = "Tests file stage out using UFTP")
	@Test
	public void testStageOut() throws Exception {
		doStageOut(false);
	}

	@FunctionalTest(id = "testUFTPStageOutEncrypt", description = "Tests file stage out using UFTP with encrypted data")
	@Test
	public void testStageOutEncrypt() throws Exception {
		doStageOut(true);
	}
	
	private void doStageOut(boolean encrypt) throws Exception {
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		cfg.setProperty(UFTPProperties.PARAM_ENABLE_ENCRYPTION, String.valueOf(encrypt));
		SubmitDocument in = SubmitDocument.Factory.newInstance();
		in.addNewSubmit().setJobDefinition(getStageOutJob());
		JobClient jc = tss.submit(in);
		jc.waitUntilReady(3000);
		// import a file
		importTestFile(jc.getUspaceClient(), "stage-out-file", 16384);
		jc.start();
		jc.waitUntilDone(15000);
		GridFileType result = sms.listProperties("test-staged-out");
		Assert.assertEquals(16384, result.getSize());
		cfg.setProperty(UFTPProperties.PARAM_ENABLE_ENCRYPTION, "false");
	}
	
	@FunctionalTest(id = "testUFTPImportFile", description = "Tests file import using UFTP")
	@Test
	public void testImportFile() throws Exception {
		doImportFile(false);
	}

	@FunctionalTest(id = "testUFTPImportFileEncrypted", description = "Tests file import using UFTP with data encryption")
	@Test
	public void testImportFileEncrypt() throws Exception {
		doImportFile(true);
	}

	private void doImportFile(boolean encrypt) throws Exception {
		String secret = "test123";
		UFTPFileTransferClient ftc = UFTPFileTransferClient.createImport(
				"test-import", sms, false, "localhost", 2, secret, encrypt);
		Assert.assertNotNull(ftc);
		String rpDoc=ftc.getResourcePropertyDocument();
		assertEquals(encrypt,rpDoc.contains(UFTPConstants.PARAM_ENCRYPTION_KEY));
		File testFile = new File("target/testfiles/data-"+System.currentTimeMillis());
		int size = 1024;
		int n = 100;
		makeTestFile(testFile, size, n);
		InputStream source = new FileInputStream(testFile);
		try {
			ftc.writeAllData(source);
		} finally {
			source.close();
		}
		
		Logger.getLogger("").info("Finished.");
		Thread.sleep(1000);
		// check that file has been written...
		GridFileType gft = sms.listProperties("test-import");
		Assert.assertNotNull(gft);
		Assert.assertEquals(size * n, gft.getSize());
	}
	
	
    @Test
	public void testImportFileUsingStorageClient() throws Exception {
		Map<String,String>params=new HashMap<String,String>();
		params.put(UFTPConstants.PARAM_SECRET, "test123");
		params.put(UFTPConstants.PARAM_CLIENT_HOST, "localhost");
		
		FileTransferClient ftc = sms.getImport("test-import", false, params, ProtocolType.UFTP);
		Assert.assertTrue(ftc instanceof UFTPFileTransferClient);
		
		Assert.assertNotNull(ftc);
		String rpDoc=ftc.getResourcePropertyDocument();
		
		assertEquals(false,rpDoc.contains(UFTPConstants.PARAM_ENCRYPTION_KEY));
		File testFile = new File("target/testfiles/data-"+System.currentTimeMillis());
		int size = 1024;
		int n = 100;
		makeTestFile(testFile, size, n);
		InputStream source = new FileInputStream(testFile);
		try {
			ftc.writeAllData(source);
		} finally {
			source.close();
		}
		
		Logger.getLogger("").info("Finished.");
		Thread.sleep(1000);
		// check that file has been written...
		GridFileType gft = sms.listProperties("test-import");
		Assert.assertNotNull(gft);
		Assert.assertEquals(size * n, gft.getSize());
	}
	
	@FunctionalTest(id = "testUFTPExportFile", description = "Tests file export using UFTP")
	@Test
	public void testExportFile() throws Exception {
		doExportFile(false);
	}

	

	@FunctionalTest(id = "testUFTPExportFileEncrypt", description = "Tests file export using UFTP with encrypted data")
	@Test
	public void testExportFileEncrypt() throws Exception {
		doExportFile(true);
	}

	private void doExportFile(boolean encrypt) throws Exception {
		String secret = "abc_TEST_123";
		// import first via BFT
		HttpFileTransferClient c = (HttpFileTransferClient) sms.getImport(
				"export_test", ProtocolType.BFT);
		File testFile = new File("target/testfiles/data-"
				+ System.currentTimeMillis());
		int size = 1024;
		int n = 100;
		makeTestFile(testFile, size, n);
		c.writeAllData(new FileInputStream(testFile));

		// now do export
		UFTPFileTransferClient ftc = UFTPFileTransferClient.createExport(
				"export_test", sms, "localhost", 1, secret, encrypt);
		Assert.assertNotNull(ftc);
		String rpDoc=ftc.getResourcePropertyDocument();
		assertEquals(encrypt,rpDoc.contains(UFTPConstants.PARAM_ENCRYPTION_KEY));
		File testTarget = new File("target/testfiles/export-"
				+ System.currentTimeMillis());
		OutputStream target = new FileOutputStream(testTarget);
		try {
			ftc.readAllData(target);
		} finally {
			target.close();
		}
		Assert.assertTrue(testTarget.exists());
		Assert.assertEquals(size * n, testTarget.length());
	}
	
	
	private JobDefinitionType getStageInJob() {
		JobDefinitionDocument jdd = JobDefinitionDocument.Factory.newInstance();
		jdd.addNewJobDefinition().addNewJobDescription().addNewApplication()
				.setApplicationName("Date");
		DataStagingType dst = jdd.getJobDefinition().getJobDescription()
				.addNewDataStaging();
		dst.addNewSource().setURI("UFTP:" + sms.getUrl() + "#/test");
		dst.setFileName("test-staged-in");
		dst.setCreationFlag(CreationFlagEnumeration.OVERWRITE);
		return jdd.getJobDefinition();
	}

	private JobDefinitionType getStageOutJob() {
		JobDefinitionDocument jdd = JobDefinitionDocument.Factory.newInstance();
		jdd.addNewJobDefinition().addNewJobDescription().addNewApplication()
				.setApplicationName("Date");
		DataStagingType dst = jdd.getJobDefinition().getJobDescription()
				.addNewDataStaging();
		dst.setFileName("stage-out-file");
		dst.addNewTarget().setURI("UFTP:" + sms.getUrl() + "#/test-staged-out");
		dst.setCreationFlag(CreationFlagEnumeration.OVERWRITE);
		return jdd.getJobDefinition();
	}

	private static UFTPServer server;

	private static void startUFTPDServer() throws Exception {
		System.out.println("Starting UFTPD server.");
		server = new UFTPServer(host, jobPort, host, srvPort);
		Thread serverThread = new Thread(server);
		serverThread.setName("UFTPD-Server-Thread");
		serverThread.start();
		System.out.println("Started UFTPD server.");
	}

	private static void stopUFTPDServer() throws Exception {
		server.stop();
		System.out.println("Stopped UFTPD server.");
	}

	private static void importTestFile(StorageClient sms, String filename,
			int size) throws Exception {
		FileTransferClient ft = sms.getImport(filename, ProtocolType.BFT);
		byte[] buf = new byte[size];
		Random r = new Random();
		r.nextBytes(buf);
		ByteArrayInputStream bis = new ByteArrayInputStream(buf);
		try {
			ft.writeAllData(bis);
		} finally {
			try {
				ft.destroy();
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
	}

	private static void makeTestFile(File file, int chunkSize, int chunks)
			throws IOException {
		FileOutputStream fos = new FileOutputStream(file);
		try {
			Random r = new Random();
			byte[] buf = new byte[chunkSize];
			for (int i = 0; i < chunks; i++) {
				r.nextBytes(buf);
				fos.write(buf);
			}
		} finally {
			fos.close();
		}

	}
}
