/*
 * Copyright (c) 2012 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE.txt file for licensing information.
 */
package de.fzj.unicore.uas.fts.uftp;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.UASProperties;
import eu.unicore.util.Log;
import eu.unicore.util.configuration.ConfigurationException;
import eu.unicore.util.configuration.DocumentationReferenceMeta;
import eu.unicore.util.configuration.DocumentationReferencePrefix;
import eu.unicore.util.configuration.PropertiesHelper;
import eu.unicore.util.configuration.PropertyMD;

public class UFTPProperties extends PropertiesHelper {
	private static final Logger log = Log.getLogger(Log.SERVICES, UFTPProperties.class);
	
	@DocumentationReferencePrefix
	public static final String PREFIX = UASProperties.PREFIX + "uftp.";
	
	
	/**
	 * extra property for configuring the UFTP client side:
	 * <ul>
	 * <li>if set to "true", the Java UFTP client code will be run
	 * directly within the JVM, which will work only if the UNICORE/X has access to
	 * the target file system</li>
	 * <li>if set to <code>false</code>, the Java UFTP client code will be run via the TSI</li>
	 * </ul>
	 */
	public static final String PARAM_CLIENT_LOCAL="client.local";

	/**
	 * property for configuring the path to the client executable (location of 'uftp.sh')
	 */
	public static final String PARAM_CLIENT_EXECUTABLE="client.executable";

	/**
	 * property for configuring the tsi host 
	 */
	//not yet used
	//public static final String PARAM_CLIENT_TSI_HOST="uftp.client.tsihost";

	/**
	 * property for disabling ssl on the command port (useful for testing)
	 */
	public static final String PARAM_COMMAND_SSL_DISABLE="command.sslDisable";

	/**
	 * property for setting the command host
	 */
	public static final String PARAM_COMMAND_HOST="command.host";

	/**
	 * property for setting the command port
	 */
	public static final String PARAM_COMMAND_PORT="command.port";
	
	/**
	 * property for setting the command port socket timeout
	 */
	public static final String PARAM_COMMAND_TIMEOUT="command.socketTimeout";
	
	/**
	 * property for setting the file read/write buffer size in kbytes
	 */
	public static final String PARAM_BUFFERSIZE="buffersize";
	
	/**
	 * parameter for configuring server limit for number of streams 
	 */
	public static final String PARAM_STREAMS_LIMIT="streamsLimit";
	
	/** 
	 * client host 
	 */
	public static final String PARAM_CLIENT_HOST="client.host";

	/**
	 * requested number of parallel data streams
	 */
	public static final String PARAM_STREAMS="streams";

	/**
	 * server host 
	 */
	public static final String PARAM_SERVER_HOST="server.host";

	/**
	 * server port 
	 */
	public static final String PARAM_SERVER_PORT="server.port";
	
	/**
	 * enable encryption by setting this to "true"
	 */
	public static final String PARAM_ENABLE_ENCRYPTION="encryption";


	@DocumentationReferenceMeta
	public static final Map<String, PropertyMD> META = new HashMap<String, PropertyMD>();
	static {
		META.put(PARAM_CLIENT_LOCAL, new PropertyMD("false").
				setDescription("Controls whether, the Java UFTP client code should be run directly within the JVM, which will work only if the UNICORE/X has access to the target file system, or, if set to false, in the TSI."));
		META.put(PARAM_CLIENT_EXECUTABLE, new PropertyMD("uftp.sh").
				setDescription("Configures the path to the client executable (location of 'uftp.sh')."));
		//TODO not yet used anywhere
		//META.put(PARAM_CLIENT_TSI_HOST, new PropertyMD().
		//		setDescription("Configures the TSI host."));
		META.put(PARAM_COMMAND_SSL_DISABLE, new PropertyMD("false").
				setDescription("Allows to disable SSL on the command port (useful for testing)."));
		META.put(PARAM_COMMAND_HOST, new PropertyMD("localhost").
				setDescription("The UFTPD command host."));
		META.put(PARAM_COMMAND_PORT, new PropertyMD("64434").setBounds(1, 65535).
				setDescription("The UFTPD command port."));
		META.put(PARAM_COMMAND_TIMEOUT, new PropertyMD("10").setBounds(0, 300).
				setDescription("The timeout (in seconds) for communicating with the command port."));
		META.put(PARAM_BUFFERSIZE, new PropertyMD("128").setPositive().
				setDescription("File read/write buffer size in kbytes."));
		META.put(PARAM_STREAMS_LIMIT, new PropertyMD("4").setPositive().
				setDescription("Server limit for number of streams."));
		
		//TODO review the rest - in most cases descriptions are not clear for me.
		//also defaults might be wrong as implementation was not consistent. E.g. SERVER_HOST in some places had default and in another no.
		META.put(PARAM_CLIENT_HOST, new PropertyMD((String)null).
				setDescription("Client host. If not set and UFTP client is set to local, then the local interface address will be determined at runtime. If not set and non-local mode is configured, then the server's host (as configured) will be used."));
		META.put(PARAM_STREAMS, new PropertyMD("1").
				setDescription("Requested number of parallel data streams."));
		META.put(PARAM_SERVER_HOST, new PropertyMD("localhost").
				setDescription("UFTPD listen host."));
		META.put(PARAM_SERVER_PORT, new PropertyMD("64434").setBounds(1, 65535).
				setDescription("UFTPD listen port."));
		META.put(PARAM_ENABLE_ENCRYPTION, new PropertyMD("false").
				setDescription("Controls whether encryption should be enabled for server-server transfers."));
	}
	
	public UFTPProperties(Properties properties) throws ConfigurationException {
		super(PREFIX, properties, META, log);
	}
}