package de.fzj.unicore.uas.fts.uftp;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlObject;
import org.unigrids.x2006.x04.services.fts.PropertyDocument;

import de.fzj.unicore.uas.client.UFTPConstants;
import de.fzj.unicore.uas.client.UFTPFileTransferClient;
import de.fzj.unicore.uas.fts.FileTransferImpl;
import de.fzj.unicore.wsrflite.persistence.Persist;
import de.fzj.unicore.wsrflite.utils.TimeoutRunner;
import de.fzj.unicore.wsrflite.xmlbeans.rp.ImmutableResourceProperty;
import eu.unicore.security.Client;
import eu.unicore.uftp.dpc.Utils;
import eu.unicore.uftp.server.UFTPTransferRequest;
import eu.unicore.util.Log;
import eu.unicore.util.httpclient.AuthSSLProtocolSocketFactory;

/**
 * server-side filetransfer ws-resource using the UFTP library
 * 
 * @author schuller
 */
public class UFTPFileTransferImpl extends FileTransferImpl implements UFTPConstants{
	
	private static final Logger logger=Log.getLogger(Log.SERVICES, UFTPFileTransferImpl.class);

	@Persist
	private String clientHost;

	@Persist
	private int streams;

	private byte[] key;
	/**
	 * initialise
	 */
	@Override
	public void initialise(String sname, Map<String, Object> map) throws Exception{
		super.initialise(sname, map);
		logger.info("Creating new UFTP file transfer for client "+getClient());
		checkAccess();
		if(extraParameters==null){
			throw new IllegalArgumentException("Missing parameters for UFTP");	
		}
		clientHost=extraParameters.get(UFTPFileTransferClient.PARAM_CLIENT_HOST);
		if(clientHost==null){
			clientHost=getClient().getSecurityTokens().getClientIP();
			if(clientHost==null){
				throw new IllegalArgumentException("Missing parameter: "+UFTPFileTransferClient.PARAM_CLIENT_HOST);
			}
		}
		String streamsP=extraParameters.get(UFTPFileTransferClient.PARAM_STREAMS);
		streams=streamsP!=null?Integer.parseInt(streamsP) : 2;
		checkNumConnectionsValid();
		String secret=extraParameters.get(UFTPFileTransferClient.PARAM_SECRET);
		String keySpec=extraParameters.get(UFTPFileTransferClient.PARAM_ENABLE_ENCRYPTION);
		if(keySpec!=null && Boolean.parseBoolean(keySpec)){
			logger.info("Data encryption enabled.");
			key=Utils.createKey();
		}
		publishServerParams();
		setupUFTP(secret);
		setReady();
	}

	/**
	 * check and enforce server limit on numconnections
	 */
	protected void checkNumConnectionsValid(){
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		int maxStreams = cfg.getIntValue(UFTPProperties.PARAM_STREAMS_LIMIT);
		streams=Math.min(maxStreams, streams);
	}
	
	/**
	 * put server parameters (host, port, number of streams) into extra RP properties
	 */
	protected void publishServerParams(){
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		String serverHost=cfg.getValue(UFTPProperties.PARAM_SERVER_HOST);
		int serverPort=cfg.getIntValue(UFTPProperties.PARAM_SERVER_PORT);
		
		PropertyDocument pd=PropertyDocument.Factory.newInstance();
		pd.addNewProperty().setName(UFTPFileTransferClient.PARAM_SERVER_HOST);
		pd.getProperty().setValue(serverHost);
		
		PropertyDocument pd2=PropertyDocument.Factory.newInstance();
		pd2.addNewProperty().setName(UFTPFileTransferClient.PARAM_SERVER_PORT);
		pd2.getProperty().setValue(String.valueOf(serverPort));
		
		PropertyDocument pd3=PropertyDocument.Factory.newInstance();
		pd3.addNewProperty().setName(UFTPFileTransferClient.PARAM_STREAMS);
		pd3.getProperty().setValue(String.valueOf(streams));
		ImmutableResourceProperty ip=null;
		if(key!=null){
			String base64=Utils.encodeBase64(key);
			PropertyDocument pd4=PropertyDocument.Factory.newInstance();
			pd4.addNewProperty().setName(UFTPFileTransferClient.PARAM_ENCRYPTION_KEY);
			pd4.getProperty().setValue(base64);
			ip=new ImmutableResourceProperty(new XmlObject[]{pd,pd2,pd3,pd4});
		}
		else{
			ip=new ImmutableResourceProperty(new XmlObject[]{pd,pd2,pd3});
		}
		properties.put(PropertyDocument.type.getDocumentElementName(),ip);
	}
	
	
	/**
	 * check whether the target file can be read or written. Need to
	 * do this here to avoid having to do it in the UFTP server!
	 * 
	 * @throws IOException
	 */
	protected void checkAccess()throws IOException{
		if(isExport)checkReadAccess();
		else checkWriteAccess();
	}
	
	protected void checkReadAccess()throws IOException{
		InputStream is=null;
		try{
			is=createNewInputStream();
			is.read();
		}
		catch(Exception e){
			throw new IOException("Can't read source file.",e);
		}
		finally{
			if(is!=null)is.close();
		}
	}
	
	protected void checkWriteAccess()throws IOException{
		OutputStream os=null;
		try{
			os=createNewOutputStream(true);
			os.write(new byte[0]);
			os.close();
		}
		catch(Exception e){
			throw new IOException("Can't write to target file.",e);
		}
	}
	
	/**
	 * tell the uftpd server that a transfer is coming
	 * @param secret
	 * @throws Exception
	 */
	public void setupUFTP(String secret)throws Exception{
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		String controlHost=cfg.getValue(UFTPProperties.PARAM_COMMAND_HOST);
		int controlPort=cfg.getIntValue(UFTPProperties.PARAM_COMMAND_PORT);
		
		//setup xlogin/group
		String user="nobody";
		String group="nobody";
		Client c=getClient();
		if(c.getXlogin()!=null){
			user=c.getXlogin().getUserName();
			group=c.getXlogin().getGroup();
			logger.debug("Initiating UFTP file transfer for user '"+user+"' group '"+group+"'");
		}
		try{
			String reply=initUFTPJob(controlHost,controlPort,secret,user,group,key);
			if(reply==null || !reply.startsWith("OK")){
				String err="UFTPD server reported an error";
				if(reply!=null){
					err+=": "+reply.trim();
				}
				throw new IOException(err);
			}
		}catch(IOException ex){
			throw new Exception("Problem communicating with the UFTP server",ex);
		}
	}

	/**
	 * sends the UFTP "job" (i.e. announce that a client will connect)
	 * directly to the Java UFTPD via a socket
	 *  
	 * @param host
	 * @param port
	 * @param secret
	 * @throws IOException
	 */
	protected String initUFTPJob(String host, int port, String secret, String user, String group, byte[] key)throws IOException{
		File file=isExport?new File(workdir+"/"+source):new File(workdir+"/"+target);
		boolean append= this.overWrite==false;
		UFTPTransferRequest job=new UFTPTransferRequest(InetAddress.getByName(clientHost),isExport, streams, file, append, secret, user, group, key);
		UFTPProperties cfg = kernel.getAttribute(UFTPProperties.class);
		boolean disableSSL=cfg.getBooleanValue(UFTPProperties.PARAM_COMMAND_SSL_DISABLE);
		int timeout=cfg.getIntValue(UFTPProperties.PARAM_COMMAND_TIMEOUT);
		String uftpdReply=sendTo(host, port, job, disableSSL, timeout);
		if(uftpdReply==null){
			throw new IOException("No reply from UFTPD (timeout)");
		}
		else{
			return uftpdReply;
		}
	}
	
	/**
	 * send job to socket using a timeout
	 *
	 * @param timeout - timeout in seconds
	 * @return reply from uftpd or null if timeout occurred
	 */
	private String sendTo(final String host, final int port, final UFTPTransferRequest job, final boolean disableSSL, int timeout)throws IOException{
		Callable<String>task=new Callable<String>(){
			@Override
			public String call() throws Exception {
				Socket socket=null;
				if(disableSSL){
					socket=new Socket(InetAddress.getByName(host),port);
				}
				else{
					AuthSSLProtocolSocketFactory f=new AuthSSLProtocolSocketFactory(kernel.getClientConfiguration());
					socket=f.createSocket(host, port);
				}
				logger.debug("Sending UFTP request to "+host+":"+port+", SSL="+!disableSSL);
				return job.sendTo(socket);
			}
		};
		TimeoutRunner<String>tr=new TimeoutRunner<String>(task, 
				kernel.getContainerProperties().getThreadingServices(), 
				timeout, TimeUnit.SECONDS);
		try{
			return tr.call();
		}catch(InterruptedException ie){
			throw new IOException(ie);
		}
	}
	
}
