package de.fzj.unicore.uas.fts.uftp;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.unigrids.services.atomic.types.ProtocolType;

import de.fzj.unicore.uas.client.UFTPConstants;
import de.fzj.unicore.uas.client.UFTPFileTransferClient;
import de.fzj.unicore.uas.xnjs.U6FileExportBase;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.util.AsyncCommandHelper;
import de.fzj.unicore.xnjs.util.LogUtil;
import de.fzj.unicore.xnjs.util.ResultHolder;
import eu.unicore.uftp.client.UFTPClient;

public class UFTPExport extends U6FileExportBase implements UFTPConstants{

	private String secret;

	/**
	 * whether the Java UFTP library should be used directly, which requires that
	 * the UNICORE/X has access to the file system<br/>
	 * If set to <code>false</code>, the TSI is used to run the UFTP client   
	 */
	private final boolean localMode;

	private final boolean useEncryption;

	private final UFTPProperties uftpProperties;

	public UFTPExport(Configuration config){
		super(config);
		uftpProperties = kernel.getAttribute(UFTPProperties.class);
		localMode = uftpProperties.getBooleanValue(UFTPProperties.PARAM_CLIENT_LOCAL);
		//TODO this should be specified by the user, not globaly. If fixed remove the config option.
		useEncryption=uftpProperties.getBooleanValue(UFTPProperties.PARAM_ENABLE_ENCRYPTION);
	}

	@Override
	public String getProtocol() {
		return ProtocolType.UFTP.toString();
	}

	@Override
	protected UFTPFileTransferClient getFTClient() throws Exception {
		String url=fileTransferInstanceEpr.getAddress().getStringValue();
		UFTPFileTransferClient c=new UFTPFileTransferClient(url, fileTransferInstanceEpr, sec);
		c.setSecret(secret);
		return c;
	}

	protected Map<String,String>getExtraParameters(){
		Map<String,String>result=new HashMap<String, String>();
		String clientHost=getClientHost();
		result.put(UFTPFileTransferClient.PARAM_CLIENT_HOST, clientHost);
		result.put(UFTPFileTransferClient.PARAM_STREAMS,getNumberOfStreams());
		secret=UUID.randomUUID().toString();
		result.put(UFTPFileTransferClient.PARAM_SECRET, secret);
		result.put(PARAM_ENABLE_ENCRYPTION,String.valueOf(useEncryption));
		return result;
	}

	@Override
	protected void doRun() throws Exception {
		checkReadPermission();
		if(localMode){
			super.doRun();
		}
		else{
			runAsync();
		}
	}

	protected void checkReadPermission()throws Exception{
		InputStream is=getInputStream();
		try{
			is.read();
		}finally{
			if(is!=null)is.close();
		}
	}

	protected String getClientHost(){
		String clientHost=uftpProperties.getValue(UFTPProperties.PARAM_CLIENT_HOST);
		if(clientHost==null){
			if(localMode){
				try{
					clientHost=InetAddress.getLocalHost().getCanonicalHostName();
				}catch(Exception ex){
					clientHost="localhost";
				}
			}
			else{
				//server host should be OK in non-local mode
				clientHost=uftpProperties.getValue(UFTPProperties.PARAM_SERVER_HOST);
			}
		}
		if(clientHost==null){
			throw new IllegalArgumentException("UFTP is not properly configured. Need to set the "
					+PARAM_CLIENT_HOST+" and "+PARAM_SERVER_HOST+" properties.");
		}
		return clientHost;
	}

	protected String getNumberOfStreams(){
		return uftpProperties.getValue(UFTPProperties.PARAM_STREAMS);
	}

	private AsyncCommandHelper ach;

	protected void runAsync()throws Exception{
		String cmd=getCommandLine();
		ach=new AsyncCommandHelper(configuration, cmd, uuid, getParentActionID(), client);
		ach.submit();
		while(!ach.isDone()){
			Thread.sleep(500);
		}
		ResultHolder res=ach.getResult();
		if(res.getExitCode()==null || res.getExitCode()!=0){
			String message="UFTP upload failed with exit code <"+res.getExitCode()+">.";
			try{
				String error=res.getStdErr();
				if(error!=null)message+=" Error details: "+error;
			}catch(IOException ex){
				LogUtil.logException("Could not read UFTP stderr",ex,logger);
			}
			throw new Exception(message);
		}
	}

	private String getCommandLine()throws Exception{
		String uftp = uftpProperties.getValue(UFTPProperties.PARAM_CLIENT_EXECUTABLE);
		UFTPFileTransferClient client=getFTClient();
		String host=client.asString(client.getServerHosts());
		int port=client.getServerPort();
		int streams=client.getStreams();
		String key=client.getEncryptionKey();
		String file=workdir+"/"+localFile;
		int buf = uftpProperties.getIntValue(UFTPProperties.PARAM_BUFFERSIZE);
		return uftp+" "+UFTPClient.makeCommandline(host, port, file, true, secret, streams, key, false, buf);
	}

}
