/*********************************************************************************
 * Copyright (c) 2008 Forschungszentrum Juelich GmbH 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Forschungszentrum Juelich GmbH nor the names of its 
 * contributors may be used to endorse or promote products derived from this 
 * software without specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 ********************************************************************************/
 
package de.fzj.unicore.uas.fts.uftp;

import java.net.URI;

import org.apache.log4j.Logger;
import org.unigrids.services.atomic.types.ProtocolType;
import org.w3.x2005.x08.addressing.EndpointReferenceType;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.uas.xnjs.U6FileTransferBase;
import de.fzj.unicore.wsrflite.utils.WSServerUtilities;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.io.DataStagingCredentials;
import de.fzj.unicore.xnjs.io.IFileTransfer;
import de.fzj.unicore.xnjs.io.IFileTransferCreator;
import eu.unicore.security.Client;

/**
 * creates UDT transfers
 * 
 * @author schuller
 */
public class UFTPCreator implements IFileTransferCreator{
	
	private static final Logger log=LogUtil.getLogger(LogUtil.SERVICES,UFTPCreator.class);
	
	private final Configuration configuration;
	
	public UFTPCreator(Configuration config) {
		this.configuration=config;
	}
	
	public IFileTransfer createFileExport(Client client, String workdir, String source, URI target, DataStagingCredentials creds) {
		log.info("+++ creating file export "+source + " -> "+ target);
		if("uftp".equalsIgnoreCase(target.getScheme())){
			IFileTransfer t=createExport(client,workdir,source,target);
			return t;
		}
		return null;
	}
	
	public IFileTransfer createFileImport(Client client, String workdir, URI source, String target, DataStagingCredentials creds) {
		log.info("+++ creating file import "+source + " -> "+ target);
		if("uftp".equalsIgnoreCase(source.getScheme())){
			IFileTransfer t=createImport(client,workdir,source,target);
			return t;
		}
		return null;
	}
	
	public String getProtocol() {
		return ProtocolType.UDT.toString();
	}

	protected IFileTransfer createExport(Client client, String workdir, String sourceFile, URI target) {
		EndpointReferenceType epr=EndpointReferenceType.Factory.newInstance();
		String tgt=target.getSchemeSpecificPart();
		String targetFile=urlDecode(target.getFragment());
		epr.addNewAddress().setStringValue(tgt);
		WSServerUtilities.addUGSRefparamToEpr(epr);
		try{
			U6FileTransferBase ft=new UFTPExport(configuration);
			ft.setClient(client);
			ft.setWorkdir(workdir);
			ft.setSource(sourceFile);
			ft.setTarget(targetFile);
			ft.setSmsEPR(epr);
			ft.setExport(true);
			return ft;
		}catch(Exception e){
			//can't happen
			LogUtil.logException("Error creating UFTP Export", e, log);
			return null;
		}
	}
	
	protected IFileTransfer createImport(Client client, String workdir, URI source, String targetFile) {
		EndpointReferenceType epr=EndpointReferenceType.Factory.newInstance();
		String src=source.getSchemeSpecificPart();
		epr.addNewAddress().setStringValue(src);
		WSServerUtilities.addUGSRefparamToEpr(epr);
		String sourceFile=urlDecode(source.getFragment());
		try{
			U6FileTransferBase ft=new UFTPImport(configuration);
			ft.setWorkdir(workdir);
			ft.setClient(client);
			ft.setSource(sourceFile);
			ft.setTarget(targetFile);
			ft.setSmsEPR(epr);
			ft.setExport(false);
			return ft;
		}catch(Exception e){
			//can't happen
			LogUtil.logException("Error creating UFTP Import", e, log);
			return null;
		}
	}
	
	/**
	 * replace URI-encoded characters by their unencoded counterparts
	 * @param orig
	 * @return
	 */
	protected String urlDecode(String orig){
		try{
			return orig.replaceAll("%20", " ");
		}catch(Exception e){
			return orig;
		}
	}
}
