package de.fzj.unicore.uas.irods;

import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.ChangeACL;
import de.fzj.unicore.xnjs.io.ChangePermissions;
import de.fzj.unicore.xnjs.io.FileFilter;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.io.XnjsFileImpl;
import de.fzj.unicore.xnjs.io.XnjsFileWithACL;
import de.fzj.unicore.xnjs.io.XnjsStorageInfo;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.irods.jargon.core.connection.IRODSAccount;
import org.irods.jargon.core.exception.JargonException;
import org.irods.jargon.core.pub.DataTransferOperations;
import org.irods.jargon.core.pub.IRODSFileSystem;
import org.irods.jargon.core.pub.io.IRODSFile;
import org.irods.jargon.core.pub.io.IRODSFileFactory;

/**
 * iRODS storage adapter.
 * <p>
 * iRODS is a federated storage technology. This adapter should enable access to
 * such federations from within UNICORE.
 * </p>
 * <p>
 * This is a new version of an adapter, written from scratch. It is planned to 
 * rely on the new Jargon API (irods java api). Currently a mixture of the old
 * and new api. 
 * <br>
 * Issues:
 * <ul>
 * <li>Credentials management</li>
 * <li>ACLs: theoretically possible</li>
 * <li>absolute paths</li>
 * <li>connection management (avoid leaking but closing after each action?)</li>
 * </ul>
 * </p>
 * 
 * @author jrybicki
 */
public class IRodsStorageAdapter implements IStorageAdapter {

    //this could be passed from factory?
    //XXX: credentials management
    String host = "localhost.localdomain";
    int irodsPort = 1247;
    String irodsUserName = "rods";
    String irodsPass = "rods";
    String irodsDefResource = "demoResc";
    String irodsHome = "/tempZone/home/rods";
    String irodsZone = "tempZone";
    private IRODSAccount irodsAccount;
    private IRODSFileSystem fs;
    private String storageRoot;

    public IRodsStorageAdapter(IRODSFileSystem fs) {
        try {
            irodsAccount = IRODSAccount.instance(host, irodsPort,
                    irodsUserName, irodsPass, irodsHome,
                    irodsZone, irodsDefResource);
            this.fs = fs;
        } catch (JargonException ex) {
            Logger.getLogger(IRodsStorageAdapter.class.getName()).log(Level.SEVERE, null, ex);
        }
    }

    @Override
    public void setStorageRoot(String path) {
        storageRoot = path;
    }

    @Override
    public String getStorageRoot() {
        return storageRoot;
    }

    @Override
    public InputStream getInputStream(String path) throws ExecutionException {
        try {
            IRODSFileFactory iRODSFileFactory = fs.getIRODSAccessObjectFactory().getIRODSFileFactory(irodsAccount);
            return new BufferedInputStream(iRODSFileFactory.instanceIRODSFileInputStream(path));
        } catch (JargonException ex) {
            throw new ExecutionException("Error opening file", ex);
        }
    }

    @Override
    public OutputStream getOutputStream(String path, boolean append /*ignored*/) throws ExecutionException {
        try {
            IRODSFileFactory iRODSFileFactory = fs.getIRODSAccessObjectFactory().getIRODSFileFactory(irodsAccount);
            return new BufferedOutputStream(iRODSFileFactory.instanceIRODSFileOutputStream(path));
        } catch (JargonException ex) {
            throw new ExecutionException("Error opening file", ex);
        }
    }

    @Override
    public OutputStream getOutputStream(String string) throws ExecutionException {
        return getOutputStream(string, true);
    }

    @Override
    public void mkdir(String path) throws ExecutionException {
        try {
            IRODSFile newDirectory = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(path);
            boolean mkdirs = newDirectory.mkdirs();
            if (!mkdirs) {
                throw new ExecutionException("Filed to create directory");
            }
            fs.close();
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to create directory", ex);
        }
    }

    @Override
    public void chmod(String path, Permissions prmsns) throws ExecutionException {
        try {
            IRODSFile file = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(path);
            file.setExecutable(prmsns.isExecutable());
            file.setReadable(prmsns.isReadable());
            file.setWritable(prmsns.isWritable());
            fs.close();
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to change file permissions", ex);
        }
    }

    @Override
    public void chmod2(String string, ChangePermissions[] cps, boolean bln) throws ExecutionException {
        throw new UnsupportedOperationException("Not implemented");
    }

    @Override
    public void chgrp(String string, String string1, boolean bln) throws ExecutionException {
        throw new UnsupportedOperationException("Not supported");
    }

    @Override
    public void setfacl(String string, boolean bln, ChangeACL[] cacls, boolean bln1) throws ExecutionException {
        throw new UnsupportedOperationException("Not supported");
    }

    @Override
    public void rm(String path) throws ExecutionException {
        try {
            IRODSFile deleteFileOrDir = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(
                    path);
            deleteFileOrDir.deleteWithForceOption();
            fs.close();
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to remove file or directory: " + path, ex);
        }
    }

    @Override
    public void rmdir(String path) throws ExecutionException {
        rm(path);
    }

    @Override
    public void cp(String source, String target) throws ExecutionException {
        try {
            DataTransferOperations dataTransferOperations = fs.getIRODSAccessObjectFactory().getDataTransferOperations(irodsAccount);
            dataTransferOperations.copy(source, "", target, null, null);
            fs.close();
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to copy file", ex);
        }
    }

    @Override
    public void rename(String source, String target) throws ExecutionException {
        try {
            //XXX: moves file or directory, possibly if moving dirs is not supported by UC we could simplify here
            DataTransferOperations dataTransferOperations = fs.getIRODSAccessObjectFactory().getDataTransferOperations(irodsAccount);
            dataTransferOperations.move(source, target);
            fs.close();
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to rename file", ex);
        }
    }

    @Override
    public XnjsFile[] ls(String string) throws ExecutionException {
        return ls(string, 0, Integer.MAX_VALUE, false);
    }

    @Override
    public XnjsFile[] ls(String base, int offset, int limit, boolean filter) throws ExecutionException {
        return find(base, null, offset, limit);
    }

    @Override
    public XnjsFile[] find(String base, final FileFilter ff, int offset, int limit) throws ExecutionException {
        try {
            IRODSFile dir = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(base);
            //non existing should throw FileNotFound which will be catched/rethrown
            File[] files;
            if (ff != null) {
                //XXX: hack not a solution
                files = dir.listFiles(new java.io.FileFilter() {

                    @Override
                    public boolean accept(File file) {
                        //what shoul be the tsi?
                        return ff.accept(convert(file), null);
                    }
                });
            } else {
                files = dir.listFiles();
            }

            if (offset > files.length) {
                throw new IllegalArgumentException(String.format("Specified offset < %s > is larger than "
                        + "the total number of results < %s >", offset, files.length));
            }
            int numResults = Math.min(files.length, limit);
            XnjsFile[] res = new XnjsFileImpl[numResults];
            for (int i = 0; i < numResults; i++) {
                res[i] = convert(files[i]);
            }
            fs.close();
            return res;
        } catch (JargonException ex) {
            throw new ExecutionException("Error while generating files list", ex);
        }
    }

    @Override
    public XnjsFileWithACL getProperties(String path) throws ExecutionException {
        try {
            IRODSFile file = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(path);
            if (!file.exists()) {
                return null;
            }
            fs.close();
            return convert(file);
        } catch (JargonException ex) {
            throw new ExecutionException("Unable to retrive file properties", ex);
        }
    }

    @Override
    public String getFileSeparator() throws ExecutionException {
        return "/";
    }

    @Override
    public String getFileSystemIdentifier() {
        return "iRODS Storage";
    }

    @Override
    public XnjsStorageInfo getAvailableDiskSpace(String path) {
        XnjsStorageInfo info = new XnjsStorageInfo();
        try {
            IRODSFile dir = fs.getIRODSFileFactory(irodsAccount).instanceIRODSFile(path);
            
            info.setFreeSpace(dir.getFreeSpace());
            info.setTotalSpace(dir.getTotalSpace());
            info.setUsableSpace(dir.getUsableSpace());
            
            fs.close();
        } catch (Exception ex) {
             //irods sometimes does not allow to estimate the free space: fallback
            //xxx: rethrowing does not make sense
            //throw new RuntimeException("Unable to estimate the available space", ex);
            info.setFreeSpace(0);
        }
        return info;
    }

    @Override
    public boolean isACLSupported(String string) throws ExecutionException {
        return false;
    }

    @Override
    public void setUmask(String string) {
        //TODO
    }

    /*
     * XXX: There is surely a method for that already?
     */
    private XnjsFile convert(File f) {
        XnjsFileImpl xFile = new XnjsFileImpl();
        xFile.setPath(f.getPath());
        xFile.setSize(f.length());
        xFile.setDirectory(f.isDirectory());
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(f.lastModified());
        xFile.setLastModified(cal);
        //TODO
        Permissions permissions = new Permissions();
        permissions.setExecutable(true);
        permissions.setReadable(true);
        permissions.setWritable(true);
        xFile.setPermissions(permissions);

        //TODO:
        xFile.setOwnedByCaller(true);
        return xFile;
    }

    /*
     * XXX: has to understand and clean it up
     */
    private XnjsFileWithACL convert(IRODSFile f) {
        XnjsFileImpl xFile = new XnjsFileImpl();
        xFile.setPath(f.getPath());
        xFile.setSize(f.length());
        xFile.setDirectory(f.isDirectory());
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(f.lastModified());
        xFile.setLastModified(cal);
        //TODO
        Permissions permissions = new Permissions();
        permissions.setExecutable(true);
        permissions.setReadable(true);
        permissions.setWritable(true);
        xFile.setPermissions(permissions);

        //TODO:
        xFile.setOwnedByCaller(true);
        return xFile;
    }

    @Override
    public String getUmask() {
        throw new UnsupportedOperationException("Not supported yet.");
    }
}
