package de.fzj.unicore.uas.hadoop;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.unigrids.services.atomic.types.GridFileType;

import de.fzj.unicore.uas.impl.sms.SMSBaseImpl;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.io.ChangeACL;
import de.fzj.unicore.xnjs.io.ChangePermissions;
import de.fzj.unicore.xnjs.io.FileFilter;
import de.fzj.unicore.xnjs.io.IStorageAdapter;
import de.fzj.unicore.xnjs.io.Permissions;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.io.XnjsFileImpl;
import de.fzj.unicore.xnjs.io.XnjsFileWithACL;
import de.fzj.unicore.xnjs.io.XnjsStorageInfo;
import eu.unicore.security.Client;

/**
 * Hadoop-based implementation of the XNJS {@link IStorageAdapter} interface
 *  
 * @author schuller
 */
public class HadoopStorageAdapter implements IStorageAdapter {

	private final FileSystem fs;

	private final Client client;

	//the base directory from which to operate
	private Path storageRoot=new Path("/");
	
	private FsPermission umask=new FsPermission((short) IStorageAdapter.DEFAULT_UMASK);
	private final static FsPermission DEFAULT_DIR_PERMS=new FsPermission((short) IStorageAdapter.DEFAULT_DIR_PERMS);
	private final static FsPermission DEFAULT_FILE_PERMS=new FsPermission((short) IStorageAdapter.DEFAULT_FILE_PERMS);
	
	/**
	 * create a new storage adapter
	 * @param fs - the Hadoop filesystem to use
	 * @param client - the {@link Client}
	 */
	public HadoopStorageAdapter(FileSystem fs, Client client){
		this.fs=fs;
		this.client=client;
	}

	public void chmod(String file, Permissions perm) throws ExecutionException {
		try{
			Path tgt = makePath(file);
			fs.setPermission(tgt, convert(perm));
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public void cp(String source, String target) throws ExecutionException {
		try{
			Path src = makePath(source);
			Path tgt = makePath(target);
			FileUtil.copy(fs, src, fs, tgt, false, false, fs.getConf());
			FileStatus srcStat = fs.getFileStatus(src);
			fs.setPermission(tgt, srcStat.getPermission().applyUMask(umask));
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public XnjsFile[] find(String base, FileFilter options, int offset,
			int limit) throws ExecutionException {
		throw new ExecutionException("Not implemented yet.");
	}

	public InputStream getInputStream(String resource)
	throws ExecutionException {
		try{
			InputStream is=fs.open(makePath(resource));
			return is;
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public OutputStream getOutputStream(String resource)
	throws ExecutionException {
		return getOutputStream(resource, /*overwrite*/ true);
	}

	public OutputStream getOutputStream(String resource, boolean append)
	throws ExecutionException {
		try{
			Path target=makePath(resource);
			OutputStream os=fs.create(target,!append);
			fs.setPermission(target, DEFAULT_FILE_PERMS.applyUMask(umask));
			return os;
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public XnjsFileWithACL getProperties(String file) throws ExecutionException {
		try{
			FileStatus fStatus = fs.getFileStatus(makePath(file));
			return convert(fStatus);
		}catch(FileNotFoundException fex){
			return null;
		}catch(IOException ioe){
			throw new ExecutionException(ioe);
		}
	}

	public XnjsFile[] ls(String base) throws ExecutionException {
		return ls(base,0,Integer.MAX_VALUE, false);
	}

	public XnjsFile[] ls(String base, int offset, int limit, 
			/*TODO this is ignored right now */ boolean filter)
	throws ExecutionException {
		try{
			Path basePath=makePath(base);
			FileStatus[] files=fs.listStatus(basePath);
			if(offset>files.length)throw new IllegalArgumentException("Specified offset <"
					+offset+"> is larger than the total number of results <"+files.length+">");
			int numResults=Math.min(files.length, limit);
			XnjsFile[] res=new XnjsFileImpl[numResults];
			for(int i=0;i<numResults;i++){
				FileStatus f=files[offset+i];
				res[i]=convert(f);
			}
			return res;
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public void mkdir(String dir) throws ExecutionException {
		boolean created=true;
		try{
			Path path = makePath(dir);
			created=fs.mkdirs(path);
			fs.setPermission(path, DEFAULT_DIR_PERMS.applyUMask(umask));
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
		if(!created)throw new ExecutionException();
	}

	public void rename(String source, String target) throws ExecutionException {
		try{
			fs.rename(makePath(source), makePath(target));
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public void rm(String target) throws ExecutionException {
		try{
			fs.delete(makePath(target), false);
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	public void rmdir(String target) throws ExecutionException {
		//same as rm in hadoop
		rm(target);
	}


	public String getFileSeparator() throws ExecutionException {
		return "/";
	}

	/**
	 * create a Hadoop {@link Path} from the given string
	 *
	 * @param path - the path string
	 */
	public Path makePath(String path){
		return new Path(storageRoot,path);
	}

	/**
	 * make the given path relative (i.e. strip off the leading scheme)
	 * 
	 * @param path
	 */
	public String getRelativePath(Path path){
		String rel=path.toUri().getPath();
		return rel.startsWith("/")?rel:"/"+rel;
	}

	public String getFileSystemIdentifier(){
		return "Apache Hadoop "+String.valueOf(fs.getUri());
	}


	public XnjsStorageInfo getAvailableDiskSpace(String path){
		XnjsStorageInfo info=new XnjsStorageInfo();
		return info;
	}

	public void setStorageRoot(String root){
		this.storageRoot=new Path(root);
	}

	public String getStorageRoot(){
		return storageRoot.getName();
	}

	public boolean isACLSupported(){
		return false;
	}
	
	@Override
	public void chgrp(String file, String newGroup, boolean recursive) throws ExecutionException {
		try{
			Path tgt = makePath(file);
			FileStatus status = fs.getFileStatus(tgt);
			fs.setOwner(tgt, status.getOwner(), newGroup);
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	@Override
	public void chmod2(String file, ChangePermissions[] perm, boolean recursive)
			throws ExecutionException {
		try{
			Path tgt = makePath(file);
			FileStatus fStatus = fs.getFileStatus(tgt);
			fs.setPermission(tgt, computeChmodPerms(perm, fStatus));
		}catch(Exception ex){
			throw new ExecutionException(ex);
		}
	}

	@Override
	public boolean isACLSupported(String path) throws ExecutionException {
		return false;
	}

	@Override
	public void setfacl(String file, boolean clearAll, ChangeACL[] changeACL, boolean recursive)
			throws ExecutionException {
		throw new ExecutionException("Not implemented");
	}

	/**
	 * used to convert raw hadoop file info into an {@link XnjsFile}
	 */
	protected XnjsFileWithACL convert(FileStatus f) {
		XnjsFileImpl xFile=new XnjsFileImpl();
		xFile.setPath(getRelativePath(f.getPath()));
		xFile.setSize(f.getLen());
		xFile.setDirectory(f.isDirectory());
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(f.getModificationTime());
		xFile.setLastModified(cal);
		//TODO
		Permissions permissions=new Permissions();
		permissions.setExecutable(true);
		permissions.setReadable(true);
		permissions.setWritable(true);
		xFile.setPermissions(permissions);
		
		xFile.setOwnedByCaller(isOwner(f));
		return xFile;
	}

	private boolean isOwner(FileStatus f) {
		//TODO check
		if(client!=null){
			try{
				String owner = f.getOwner();
				if(client.getXlogin()!=null) {
					return owner.equals(client.getXlogin().getUserName());
				}
			}
			catch(Exception ex){
				LogUtil.logException("error checking file ownership", ex);
			}
		}
		return false;
	}

	/**
	 * used to convert raw hadoop file info into a x{@link GridFileType}
	 */
	protected void convert(FileStatus f, GridFileType gf) {
		gf.setPath(getRelativePath(f.getPath()));
		gf.setSize(f.getLen());
		gf.setIsDirectory(f.isDirectory());
		Calendar cal = Calendar.getInstance();
		cal.setTimeInMillis(f.getModificationTime());
		gf.setLastModified(cal);
		
		gf.setIsOwner(isOwner(f));
		
		//FIXME - is it OK? In UNICORE GridFileType.getPermissions() 
		//should return effective permissions for the caller, not for the file owner.
		FsPermission p = f.getPermission();
		boolean exec=p.getUserAction().implies(FsAction.EXECUTE);
		boolean write=p.getUserAction().implies(FsAction.WRITE);
		boolean read=p.getUserAction().implies(FsAction.READ);
		gf.addNewPermissions().setReadable(read);
		gf.getPermissions().setWritable(write);
		gf.getPermissions().setExecutable(exec);
		
		
		StringBuilder perms = new StringBuilder();
		appendRwxPerms(perms, p.getUserAction());
		appendRwxPerms(perms, p.getGroupAction());
		appendRwxPerms(perms, p.getOtherAction());
		gf.setFilePermissions(perms.toString());
		
		gf.setGroup(f.getGroup());
		gf.setOwner(f.getOwner());
	}

	protected void appendRwxPerms(StringBuilder sb, FsAction action) {
		if (action.implies(FsAction.READ))
			sb.append("r");
		else
			sb.append("-");
		if (action.implies(FsAction.WRITE))
			sb.append("w");
		else
			sb.append("-");
		if (action.implies(FsAction.EXECUTE))
			sb.append("x");
		else
			sb.append("-");
		
	}
	
	protected FsPermission convert(Permissions perm){
		FsAction userAction=FsAction.READ;
		if(perm.isWritable()){
			userAction=userAction.or(FsAction.WRITE);
		}
		if(perm.isExecutable()){
			userAction=userAction.or(FsAction.EXECUTE);
		}
		FsPermission res=new FsPermission(userAction, FsAction.NONE, FsAction.NONE);
		return res;
	}
	
	protected Permissions convert(FsPermission perm){
		Permissions res=new Permissions();
		FsAction userAction=perm.getUserAction();
		res.setReadable(userAction.implies(FsAction.READ));
		res.setWritable(userAction.implies(FsAction.WRITE));
		res.setExecutable(userAction.implies(FsAction.EXECUTE));
		return res;
	}
	
	protected FsPermission computeChmodPerms(ChangePermissions[] perm, FileStatus fs) {
		short res;
		FsPermission working = fs.getPermission();
		for (ChangePermissions pSpec: perm) {
			ChmodParser2 chmodParser = new ChmodParser2(pSpec.getClazzSymbol() +  
					pSpec.getModeOperator() + 
					pSpec.getPermissions());
			res = chmodParser.applyNewPermission(working, fs.isDirectory());
			working = new FsPermission(res);
		}
		return working;
	}

	/**
	 * decode URL characters, replaceing "%20" and "+" by spaces
	 * @see SMSBaseImpl#urlDecode(String)
	 * @param p - the string to decode
	 * @return
	 */
	protected String urlDecode(String p){
		return SMSBaseImpl.urlDecode(p);
	}

	@Override
	public void setUmask(String umask) {
		while (umask.length() < 3)
			umask = "0" + umask;
		this.umask = new FsPermission(umask);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getUmask() {
		return Integer.toOctalString(umask.toShort());
	}
}
