/*
 * Copyright (c) 2000-2005, Intel Corporation All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * (1) Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the disclaimer at the end. Redistributions in
 * binary form must reproduce the above copyright notice, this list of
 * conditions and the following disclaimer in the documentation and/or other
 * materials provided with the distribution.
 * 
 * (2) Neither the name of Intel Corporation nor the names of its contributors
 * may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * DISCLAIMER
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package com.intel.gpe.gridbeans;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.jar.JarFile;
import java.util.zip.ZipEntry;

import org.apache.log4j.Logger;
import org.jdom.adapters.XercesDOMAdapter;
import org.w3c.dom.Element;

import de.fzj.unicore.uas.UASProperties;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.KernelInjectable;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;

/**
 * @author Ralf Ratering
 * @author Alexander Lukichev
 */
public class GridBeanServiceImpl implements GridBeanService, KernelInjectable {
	
    private static final Logger logger = LogUtil.getLogger(LogUtil.SERVICES,GridBeanServiceImpl.class);
    
    private Kernel kernel;
    
    public void setKernel(Kernel kernel){
    	this.kernel=kernel;
    }
    
    public GridBeanDescriptionType getGridBeanDescription(File file) throws Exception  {
        Element element = getDescriptionElement(file);
        GridBeanDescriptionType gridBeanDescription = GridBeanDescriptionType.Factory.newInstance();
        gridBeanDescription.setSize((int) file.length());
        gridBeanDescription.setName(ElementUtil.getChildValueString(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.NAME, ""));
        gridBeanDescription.setAuthor(ElementUtil.getChildValueString(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.AUTHOR, ""));
        gridBeanDescription.setVersion(ElementUtil.getChildValueFloat(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.VERSION, 1.0f));
        gridBeanDescription.setApplication(ElementUtil.getChildValueString(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.APPLICATION, ""));
        gridBeanDescription.setDescription(ElementUtil.getChildValueString(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.DESCRIPTION, ""));
        gridBeanDescription.setPluginVersion(ElementUtil.getChildValue(
                element, GridBeanConstants.NAMESPACE, GridBeanConstants.PLUGIN_VERSION, 1));
        return gridBeanDescription;
    }


	public GetGridBeanResponseDocument GetGridBean(GetGridBeanDocument request) throws BaseFault {
		
		String gridBeanPath = getGridBeanPath();
		if (gridBeanPath == null) {
			String msg = "GridBean directory is not set";
			logger.warn(msg);
			GridBeanNotGotFaultType fault = GridBeanNotGotFaultType.Factory.newInstance();
			fault.setTimestamp(Calendar.getInstance());
			fault.addNewDescription().setStringValue(msg);
			throw new BaseFault(msg, fault);
		}
		else {
			
			File gridBeanDir = new File(gridBeanPath);
			
			if (!gridBeanDir.exists() || !gridBeanDir.isDirectory()) {
				String msg = "GridBean directory does not exist ("+gridBeanPath+")";
				logger.error(msg);
				GridBeanNotGotFaultType fault = GridBeanNotGotFaultType.Factory.newInstance();
				fault.setTimestamp(Calendar.getInstance());
				fault.addNewDescription().setStringValue(msg);
				throw new BaseFault(msg, fault);				
			}
			
			String name = request.getGetGridBean().getName();
			File file = new File(gridBeanDir, name + GridBeanConstants.GB_POSTFIX);
			if (!file.exists()) {
				String msg = "GridBean does not exist ("+name+")";
				logger.debug(msg);
				GridBeanNotGotFaultType fault = GridBeanNotGotFaultType.Factory.newInstance();
				fault.setTimestamp(Calendar.getInstance());
				fault.addNewDescription().setStringValue(msg);
				throw new BaseFault(msg, fault);				
			}
			logger.debug("Found resource " + name + " at " + file);
			
			GetGridBeanResponseDocument response = GetGridBeanResponseDocument.Factory.newInstance();
			byte[] result = null;
			try {
				result = readBytesFromFile(file);
				response.addNewGetGridBeanResponse().setContents(result);
			}
			catch (IOException e) {
				LogUtil.logException("Could not load GridBean " + name, e, logger);
			}
			
			return response;
		}
	}

	public GetGridBeanInfoResponseDocument GetGridBeanInfo(GetGridBeanInfoDocument request) throws BaseFault {
		String gridBeanPath = getGridBeanPath();
		if (gridBeanPath == null) {
			String msg = "GridBean directory is not set";
			logger.error(msg);
			GridBeanInfoNotGotFaultType fault = GridBeanInfoNotGotFaultType.Factory.newInstance();
			fault.setTimestamp(Calendar.getInstance());
			fault.addNewDescription().setStringValue(msg);
			throw new BaseFault(msg, fault);
		}
        else {
			File gridBeanDir = new File(gridBeanPath);
			
			if (!gridBeanDir.exists() || !gridBeanDir.isDirectory()) {
				String msg = "GridBean directory does not exist ("+gridBeanPath+")";
				logger.error(msg);
				GridBeanInfoNotGotFaultType fault = GridBeanInfoNotGotFaultType.Factory.newInstance();
				fault.setTimestamp(Calendar.getInstance());
				fault.addNewDescription().setStringValue(msg);
				throw new BaseFault(msg, fault);				
			}
			
			String name = request.getGetGridBeanInfo().getName();
			File file = new File(gridBeanDir, name + GridBeanConstants.GB_POSTFIX);
			if (!file.exists()) {
				String msg = "GridBean does not exist ("+name+")";
				logger.error(msg);
				GridBeanInfoNotGotFaultType fault = GridBeanInfoNotGotFaultType.Factory.newInstance();
				fault.setTimestamp(Calendar.getInstance());
				fault.addNewDescription().setStringValue(msg);
				throw new BaseFault(msg, fault);				
			}
			logger.debug("Found resource " + name + " at " + file);
            
            GetGridBeanInfoResponseDocument response = GetGridBeanInfoResponseDocument.Factory.newInstance();
	        response.addNewGetGridBeanInfoResponse();
            GridBeanDescriptionType gridBeanInfo = null;
            try {
            	gridBeanInfo = getGridBeanDescription(file);
                response.getGetGridBeanInfoResponse().setGridBeanInfo(gridBeanInfo);
            }
            catch (Exception e) {
				String msg = "Cannot get GridBean ("+name+")";
				logger.error(msg);
				GridBeanInfoNotGotFaultType fault = GridBeanInfoNotGotFaultType.Factory.newInstance();
				fault.setTimestamp(Calendar.getInstance());
				fault.addNewDescription().setStringValue(msg);
				throw new BaseFault(msg, e, fault);				
            }
            
            return response;
        }
	}

	public ListGridBeansResponseDocument ListGridBeans(ListGridBeansDocument request) throws BaseFault {
		String gridBeanPath = getGridBeanPath();
		if (gridBeanPath == null) {
			String msg = "GridBean directory is not set";
			logger.error(msg);
			GridBeansNotListedFaultType fault = GridBeansNotListedFaultType.Factory.newInstance();
			fault.setTimestamp(Calendar.getInstance());
			fault.addNewDescription().setStringValue(msg);
			throw new BaseFault(msg, fault);
		}
		File gridBeanDir = new File(gridBeanPath);
		
		if (!gridBeanDir.exists() || !gridBeanDir.isDirectory()) {
			String msg = "GridBean directory does not exist ("+gridBeanPath+")";
			logger.error(msg);
			GridBeansNotListedFaultType fault = GridBeansNotListedFaultType.Factory.newInstance();
			fault.setTimestamp(Calendar.getInstance());
			fault.addNewDescription().setStringValue(msg);
			throw new BaseFault(msg, fault);				
		}
		File[] files = gridBeanDir.listFiles();
		List<GridBeanDescriptionType> gridBeans = 
			new ArrayList<GridBeanDescriptionType>(files.length);
		
		for (File file : files) {
			if (file.getName().endsWith(GridBeanConstants.GB_POSTFIX)) {
				try {
					gridBeans.add(getGridBeanDescription(file));
					
				}
				catch (Exception e) {
					LogUtil.logException(e.getMessage(), e, logger);
					continue;
				}
			}
		}
		
		ListGridBeansResponseDocument response = ListGridBeansResponseDocument.Factory.newInstance();
		response.addNewListGridBeansResponse().setAvailableGridBeansArray(
				gridBeans.toArray(new GridBeanDescriptionType[0]));
		return response;   
		
	  }
	
	private Element getDescriptionElement(File file)throws Exception{
		JarFile jarFile = new JarFile(file);
    	ZipEntry entry = jarFile.getEntry("META-INF/gridbean.xml");
    	if (entry == null) {
    		throw new Exception("Cannot find GridBean description for " + file.getAbsolutePath());
    	}
    	InputStream is = jarFile.getInputStream(entry);
    	org.w3c.dom.Document doc = new XercesDOMAdapter().getDocument(is,false);
    	Element root=doc.getDocumentElement();
    	return root;
	}
	
	private static byte[] readBytesFromFile(File file) throws IOException {
		InputStream is = new FileInputStream(file);
		long length = file.length();
		if (length > Integer.MAX_VALUE) {
			throw new IOException("File size " + length + " is too large.");
		}
		byte[] bytes = new byte[(int) length];
		int offset = 0;
		int numRead = 0;
		while (offset < bytes.length && (numRead = is.read(bytes, offset, bytes.length - offset)) >= 0) {
			offset += numRead;
		}
		if (offset < bytes.length) {
			throw new IOException("Could not completely read file " + file.getName());
		}
		is.close();
		return bytes;
	}
	
	protected String getGridBeanPath(){
		UASProperties uasProperties = kernel.getAttribute(UASProperties.class);
		String res=uasProperties.getValue(UASProperties.GRIDBEAN_DIR);
		return res;
	}
	
}