package com.intel.gpe.gridbeans;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Utility methods for XML processing.
 * 
 * @author Dmitry Petrov
 */
public class ElementUtil {

    public static Element simpleElt(Document doc, String ns, String tag, String value) {
        Element elt = doc.createElementNS(ns, tag);
        elt.appendChild(doc.createTextNode(value));
        return elt;
    }

    public static Element simpleCDATA(Document doc, String ns, String tag, String value) {
        Element elt = doc.createElementNS(ns, tag);
        elt.appendChild(doc.createCDATASection(value));
        return elt;
    }
    /**
     * Get the first child with the specified namespace and local part.
     * @param elt - the parent element
     * @param ns - the namespace of the child
     * @param tag - the local part of the child name
     * @return the found first child, <code>null</code> if nothing found
     */
    public static Element getFirstChildByNameNS(Element elt, String ns, String tag) {
        NodeList children = elt.getChildNodes();
        for (int i=0; i<children.getLength(); i++) {
            if (!(children.item(i) instanceof Element))
                continue;
            Element childi = (Element) children.item(i);
            if (ns.equals(childi.getNamespaceURI()) && tag.equals(childi.getLocalName())) {
                return childi;
            }
        }
        // no child found
        return null;
    }
    /**
     * Get the list of children with the specified namespace and local part
     * @param elt - the parent element
     * @param ns - the namespace
     * @param name - the local part
     * @return the list of children
     * @author Alexander Lukichev
     */
    public static List<Element> getChildElements(Element elt, String ns, String name) {
        List<Element> result = new ArrayList<Element>();
        NodeList children = elt.getChildNodes();
        for (int i=0; i<children.getLength(); i++) {
            if (!(children.item(i) instanceof Element))
                continue;
            Element childi = (Element) children.item(i);
            if (ns.equals(childi.getNamespaceURI()) && name.equals(childi.getLocalName())) {
                result.add(childi);
            }
        }
        return result;
    }
    
    public static void addOrReplace(Element parent, Element elem) {
        Element old = getFirstChildByNameNS(parent, elem.getNamespaceURI(), elem.getLocalName());
        if (old != null) {
            parent.replaceChild(elem, old);
        } else {
            parent.appendChild(elem);
        }    
    }

    public static String getChildValueString(Element elt, String ns, String tag, String _default) {
        Element child = getFirstChildByNameNS(elt, ns, tag);
        if (child != null) {
            if (child.getFirstChild() != null) {
                return child.getFirstChild().getNodeValue();
            }
            else {
                // handle standard serializer behaviour for whitespace-only strings
                return ""; // sorry, no better idea what it could've been
            }
        }
        else {
            return _default;
        }
    }

    public static Double getChildValueDouble(Element elt, String ns, String tag, Double _default) {
        Element child = getFirstChildByNameNS(elt, ns, tag);
        if (child != null && child.getFirstChild() != null) {
            return Double.valueOf(child.getFirstChild().getNodeValue());
        }
        else {
            return _default;
        }
    }
    
    public static Float getChildValueFloat(Element elt, String ns, String tag, Float _default) {
        Element child = getFirstChildByNameNS(elt, ns, tag);
        if (child != null && child.getFirstChild() != null) {
            return Float.valueOf(child.getFirstChild().getNodeValue());
        }
        else {
            return _default;
        }
    }

    public static double getChildValue(Element elt, String ns, String tag, double _default) {
        Element child = getFirstChildByNameNS(elt, ns, tag);
        if (child != null && child.getFirstChild() != null) {
            return Double.parseDouble(child.getFirstChild().getNodeValue());
        }
        else {
            return _default;
        }
    }

    public static int getChildValue(Element elt, String ns, String tag, int _default) {
        Element child = getFirstChildByNameNS(elt, ns, tag);
        if (child != null && child.getFirstChild() != null) {
            try{
            	return Integer.parseInt(child.getFirstChild().getNodeValue());
            }catch(NumberFormatException nfe){
            }
        }
        return _default;
    }

    public static String getTextValue(Element elt) {
        if (elt.getFirstChild()==null)
            return "";
        else
            return elt.getFirstChild().getNodeValue();
    }

    public static String getAttributeValue(Element elt, String attrName, String _default) {
        String attr = elt.getAttribute(attrName);
        // MessageElement does not follow interface specification as it is defined in JavaDoc
        // (returns null instead of "" if attribute is absent).
        if (attr==null || attr=="") {
            return _default;
        } else {
            return attr;
        }
    }
    /**
     * Read document from input stream
     * @param in - the input stream
     * @return
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     */
    public static Document readDocument(InputStream in) throws SAXException, IOException, ParserConfigurationException {
    	DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
    	dbf.setNamespaceAware(true);
    	return dbf.newDocumentBuilder().parse(in);
    }
}