/*
 * Copyright (c) 2010 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE file for licencing information.
 *
 * Created on 06-09-2010
 * Author: K. Benedyczak <golbi@mat.umk.pl>
 */
package eu.unicore.uas.security.gridmapfile;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.security.auth.x500.X500Principal;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.util.Pair;
import edu.virginia.vcgr.genii.security.X500PrincipalUtilities;
import eu.unicore.security.util.Log;
import eu.unicore.uas.security.file.FileAttributeSource;


/**
 * Utility class to parse file with attributes. Format is described in {@link FileAttributeSource}.
 * StAX parser is used.
 * @author golbi
 */
public class GridMapFileParser
{

	
	
	private static final Logger logger = Log.getLogger(Log.SECURITY, GridMapFileAuthoriser.class);
	static final private Pattern ATTRIBUTE_PATTERN = Pattern.compile("([^=]+)=([^=]+)");
	
	private File gridMapFile;

	private static final String COMMENT_CHARS = "#";
	// keywords that need to be replaced
	private static final String EMAIL_KEYWORD_1 = "e";
	private static final String EMAIL_KEYWORD_2 = "email"; 
	private static final String USERID_KEYWORD = "userid";
	
	// Keywords to be replaced with.
	private static final String EMAIL_KEYWORD = "emailaddress";
	private static final String UID_KEYWORD = "uid";
	
	private int numDuplicates;
	private int numEmpty;
	private int numError;

	public GridMapFileParser(File gridMapFile)
	{
		this.gridMapFile = gridMapFile;
	}

	public Map<String,List<String>> parse()
	{
		
		Map<String,List<String>> result = new HashMap<String, List<String>>();
	
		DataInputStream in = null;
		try{
			in = new DataInputStream(new FileInputStream(gridMapFile));
			// Get the object of DataInputStream
			BufferedReader br = new BufferedReader(new InputStreamReader(in));
			String line;
			//Read File Line By Line
			while ((line = br.readLine()) != null)   {
				try {
					line = line.trim();
					if ( (line.length() == 0) ||
							( COMMENT_CHARS.indexOf(line.charAt(0)) != -1) ) {
							numEmpty++;
						continue;
					}

					Pair<String, String[]> mapping = parseLine(line);
					if(mapping != null)
					{
						String key = mapping.getM1();

						List<String> logins = result.get(key);
						if(logins == null)
						{
							logins = new ArrayList<String>();
							result.put(key,logins);
						}
						String[] ids = mapping.getM2();
						for(String id : ids)
						{
							if(logins.contains(id))
							{
								numDuplicates++;
							}
							else
							{
								logins.add(id);
							}
						}
					}
					else 
					{
						numError ++;
					}
					
				} catch (Exception e) {
					logger.warn("Problem while parsing line "+line+" from grid-mapfile "+gridMapFile.getAbsolutePath()+": "+e.getMessage(),e);
					numError ++;
				}
			}

		} catch (Exception e){//Catch exception if any
			logger.warn("Problem while parsing grid-mapfile "+gridMapFile.getAbsolutePath()+": "+e.getMessage(),e);
		}
		finally
		{
			//Close the input stream
			try {
				if(in != null) in.close();	
			} catch (Exception e2) {
				// do nothing
			}
		}
		
		
		return result;
	}

	public static Pair<String, String[]> parseLine(String line) throws IOException
	{

		QuotedStringTokenizer tokenizer;
		StringTokenizer idTokenizer;

		tokenizer = new QuotedStringTokenizer(line);

		String globusID = null;

		if (tokenizer.hasMoreTokens()) {
			globusID = tokenizer.nextToken();
		} else {
			throw new IOException("Line does not contain Grid id!");

		}

		String userIDs = null;

		if (tokenizer.hasMoreTokens()) {
			userIDs = tokenizer.nextToken();
		} else {
			throw new IOException("Line does not contain user id!");

		}

		idTokenizer = new StringTokenizer(userIDs, ",");
		String [] ids = new String [ idTokenizer.countTokens() ];
		int i = 0;
		while(idTokenizer.hasMoreTokens()) {
			ids[i++] = idTokenizer.nextToken();
		}

		String normalizedDN = normalizeDN(globusID);
		if(normalizedDN == null) throw new IOException("Invalid distinguished name: "+globusID);
		return new Pair<String,String[]>(normalizedDN,ids);
	}


	public static String normalizeDN(String globusID) {

		if (globusID == null) {
			return null;
		}


		StringBuffer buff = new StringBuffer();
		String[] tokens = globusID.split("/");
		if(tokens.length == 0) return null;
		for(String s : tokens)
		{
			Matcher m = ATTRIBUTE_PATTERN.matcher(s);
			
			if(m.matches())
			{
				String att = m.group(1);
				String value = m.group(2);
				// replace invalid attribute names
				if(att.equalsIgnoreCase(EMAIL_KEYWORD_1) 
						|| att.equalsIgnoreCase(EMAIL_KEYWORD_2))
				{
					s = EMAIL_KEYWORD +"="+value;
				}
				else if(att.equalsIgnoreCase(USERID_KEYWORD))
				{
					s = UID_KEYWORD +"="+value;
				}
				if(value.contains(","))
				{
					s = att+"="+ value.replaceAll(",", Matcher.quoteReplacement(new String("\\"+",")));
				}
				
				if(buff.length()>0) buff.append(",");
				buff.append(s);
				
			}
			
		}
		String result = buff.toString();
		try {
			X500Principal x500 = new X500Principal(result);
			result = X500PrincipalUtilities.getCanonicalizedString(x500);	
			return result;
		} catch (Exception e) {
			logger.warn("Found invalid distinguished user name in grid-mapfile: "+result,e);
			return null;
		}

		
	}

	public int getNumDuplicates() {
		return numDuplicates;
	}

	public int getNumEmpty() {
		return numEmpty;
	}

	public int getNumError() {
		return numError;
	}


	

}
