/*
 * Copyright (c) 2010 ICM Uniwersytet Warszawski All rights reserved.
 * See LICENCE file for licencing information.
 *
 * Created on 06-09-2010
 * Author: K. Benedyczak <golbi@mat.umk.pl>
 */
package eu.unicore.uas.security.gridmapfile;

import java.io.File;
import java.io.IOException;
import java.security.KeyStore;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.wsrflite.ResourcePool;
import de.fzj.unicore.wsrflite.security.IAttributeSource;
import de.fzj.unicore.wsrflite.utils.FileWatcher;
import edu.virginia.vcgr.genii.security.X500PrincipalUtilities;
import eu.unicore.genii.GenesisConstants;
import eu.unicore.genii.security.TruststoreUtil;
import eu.unicore.security.SecurityTokens;
import eu.unicore.security.SubjectAttributesHolder;
import eu.unicore.security.XACMLAttribute;
import eu.unicore.security.util.Log;


/**
 * Retrieves local login name(s) from a Globus grid-mapfile. For security
 * reasons, this must be used in a combination with a trust store containing CAs
 * for valid user certificates. 
 * @author demuth
 *
 */
public class GridMapFileAuthoriser implements IAttributeSource 
{
	private static final Logger logger = Log.getLogger(Log.SECURITY, GridMapFileAuthoriser.class);

	//config options
	private File mapFile = new File("conf", "grid-mapfile");

	private String name;
	private String status = "OK";
	private Map<String,List<String>> map;

	private static KeyStore trustStore;

	@Override
	public void init(String name) throws Exception
	{
		this.name = name;
		parse();
		reloadTrustStore();
		FileWatcher modifiedWatcher = new FileWatcher(mapFile, new Runnable(){
			@Override
			public void run() {
				parse();
			}});
		ResourcePool.getScheduledExecutorService().scheduleWithFixedDelay(modifiedWatcher, 2, 2, TimeUnit.MINUTES);

	}

	@Override
	public SubjectAttributesHolder getAttributes(SecurityTokens tokens,
			SubjectAttributesHolder otherAuthoriserInfo)
	throws IOException
	{

		Map<String, String[]> retAll = new HashMap<String, String[]>();
		Map<String, String[]> retFirst = new HashMap<String, String[]>();
		List<XACMLAttribute> retXACML = new ArrayList<XACMLAttribute>();
		CertPath user = tokens.getUser();
		if(user == null) user = tokens.getConsignor();

		if(user != null)
		{
			List<? extends Certificate> certs = user.getCertificates();
			X509Certificate[] certArray = new X509Certificate[certs.size()];

			int i = 0;
			for(Certificate cert : certs)
			{
				certArray[i++] = (X509Certificate) cert;
			}

			String subject = X500PrincipalUtilities.getCanonicalizedString(tokens.getEffectiveUserName());


			if(TruststoreUtil.isFromTrustedAuthority(trustStore,certArray))
			{
				List<String> xlogins = map.get(subject);
				if(xlogins != null)
				{
					String[] loginArray = xlogins.toArray(new String[xlogins.size()]);
					retFirst.put(IAttributeSource.ATTRIBUTE_XLOGIN,loginArray);	
					retAll.put(IAttributeSource.ATTRIBUTE_XLOGIN,loginArray);
					retFirst.put(IAttributeSource.ATTRIBUTE_ROLE,new String[]{"user"});	
					retAll.put(IAttributeSource.ATTRIBUTE_ROLE,new String[]{"user"});
				}
			}
		}
		return new SubjectAttributesHolder(retXACML, retFirst, retAll);
	}


	private void parse()
	{		
		try
		{
			if(mapFile == null)
			{
				logger.error("Could not parse grid-mapfile as it is not specified. Please set it in your attribute source configuration (look for properties named uas.security.attributes.*");
				status = "FAILED";
				return;
			}
			if(!mapFile.exists())
			{
				logger.error("Could not parse grid-mapfile as it does not exist. Please change it in your attribute source configuration (look for properties named uas.security.attributes.*");
				status = "FAILED";
				return;
			}
			if(mapFile.isDirectory())
			{
				logger.error("Could not parse grid-mapfile as it is a directory. Please change it in your attribute source configuration (look for properties named uas.security.attributes.*");
				status = "FAILED";
				return;
			}
			if(!mapFile.canRead())
			{
				logger.error("Could not parse grid-mapfile as it cannot be read. Please check permissions of the file "+mapFile.getAbsolutePath());
				status = "FAILED";
				return;
			}

			GridMapFileParser parser = new GridMapFileParser(mapFile);
			map = parser.parse();

			logger.info("User attributes were loaded from the file " + 
					mapFile);

		} catch (Exception e)
		{
			status = "FAILED";
			logger.error("The updated attributes list is INVALID: " + e.getMessage());
		}

	}



	@Override
	public String getStatusDescription()
	{
		return "File Attribute Source [" + name + "]: " + 
		status + ", using map file " + mapFile.getAbsolutePath();
	}

	@Override
	public String getName()
	{
		return name;
	}

	public void setFile(String uudbFile) {
		this.mapFile = new File(uudbFile);
	}


	@Override
	public String[] getAcceptedVOs()
	{
		return null;
	}




	private void reloadTrustStore()
	{

		String dir = UAS.getProperty(GenesisConstants.PROP_TRUSTED_ID_PROVIDER_DIR);
		if(dir == null) dir = GenesisConstants.DEFAULT_TRUSTED_ID_PROVIDER_DIR;
		try {
			trustStore = TruststoreUtil.loadTruststoreFromDirectory(dir);
		} catch (Exception e) {
			logger.error("Unable to setup trusted identity authorities: "+e.getMessage(), e);
		}

	}
}








