package eu.unicore.genii.security;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.Random;
import java.util.UUID;

import javax.security.auth.x500.X500Principal;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.bouncycastle.x509.X509V3CertificateGenerator;

import de.fzj.unicore.persist.ConfigurationSource;
import de.fzj.unicore.uas.UAS;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.security.ISecurityProperties;
import de.fzj.unicore.wsrflite.security.SecurityManager;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.ems.Action;
import de.fzj.unicore.xnjs.ems.InternalManager;
import de.fzj.unicore.xnjs.ems.ProcessingContext;
import de.fzj.unicore.xnjs.ems.SubAction;
import de.fzj.unicore.xnjs.simple.BasicManager;
import de.fzj.unicore.xnjs.tsi.TSI;
import de.fzj.unicore.xnjs.util.LogUtil;
import edu.virginia.vcgr.genii.client.GenesisIIConstants;
import edu.virginia.vcgr.genii.client.context.ContextFileSystem;
import edu.virginia.vcgr.genii.client.context.ICallingContext;
import edu.virginia.vcgr.genii.client.security.GenesisIISecurityException;
import edu.virginia.vcgr.genii.client.security.authz.AuthZSecurityException;
import edu.virginia.vcgr.genii.client.security.x509.KeyAndCertMaterial;
import edu.virginia.vcgr.genii.security.credentials.GIICredential;
import edu.virginia.vcgr.genii.security.credentials.TransientCredentials;
import edu.virginia.vcgr.genii.security.credentials.assertions.BasicConstraints;
import edu.virginia.vcgr.genii.security.credentials.assertions.DelegatedAssertion;
import edu.virginia.vcgr.genii.security.credentials.assertions.DelegatedAttribute;
import edu.virginia.vcgr.genii.security.credentials.assertions.SignedAssertion;
import edu.virginia.vcgr.genii.security.credentials.identity.UsernamePasswordIdentity;
import edu.virginia.vcgr.genii.security.credentials.identity.X509Identity;
import eu.unicore.genii.GenesisConstants;
import eu.unicore.security.Client;
import eu.unicore.security.util.KeystoreChecker;

public class TrustDelegationUtils implements GenesisConstants {

	protected static final Logger logger=LogUtil.getLogger(LogUtil.JOBS,TrustDelegationUtils.class);

	private static final String PROP_GENII_DIR_CREATED = "GEN_II_DIR_CREATED_FLAG";

	private static final Object CREATE_GENII_STATE_DIR_LOCK = new Object();

	


	/**
	 * Creates a directory containing credentials for GenII on the TSI host.
	 * Returns the absolute path of the created dir for convenience. 
	 * @param configuration
	 * @param action
	 * @return
	 * @throws Exception
	 */
	public static CreateStateDirResult createGenIIStateDirIfNotExists(Configuration configuration, Action action) throws Exception{
		CreateStateDirResult result = new CreateStateDirResult();
		InternalManager manager = configuration.getInternalManager();
		Action parentAction = action;
		// go to top level action
		while(parentAction instanceof SubAction)
		{
			parentAction = manager.getAction(((SubAction) parentAction).getParentID());
		}
		if(parentAction == null) throw new Exception("Can't create Genesis II security context, parent action not found.");
		Client client = action.getClient();
		ICallingContext context =(ICallingContext) client.getSecurityTokens().getContext().get(GENII_GAMLASSERTION_REF);
		if(context==null){
			// can't operate without the context
			throw new Exception("No GAML assertion available in security attributes! Can't create Genesis II security context.");
		}

		synchronized(CREATE_GENII_STATE_DIR_LOCK)
		{
			
			parentAction = manager.getAction(parentAction.getUUID());
			if(parentAction == null) throw new Exception("Can't create Genesis II security context, parent action not found.");
			ProcessingContext pContext = parentAction.getProcessingContext();
			Object o = pContext.get(PROP_GENII_DIR_CREATED);
			if(o == null)
			{
				// set some files
				File temp = getTempDirectory();

				File stateDir = new File(temp,GENII_STATE_DIR);
				stateDir.mkdirs();
				try {	
					// create the Genesis II state directory
					ISecurityProperties secProps = Kernel.getKernel().getSecurityProperties();
					KeyAndCertMaterial delegatee = generateUnprivilegedCertificate(secProps);
					X509Certificate serverCert = SecurityManager.getServerCert();
					PrivateKey serverKey = getServerPrivateKey();
					KeyAndCertMaterial delegator = new KeyAndCertMaterial(new X509Certificate[]{serverCert}, serverKey);
					File contextFile = new File(stateDir, USER_CONTEXT_FILENAME);
					File transientFile = new File(stateDir, USER_TRANSIENT_FILENAME);
					delegateContext(context, delegatee, delegator);


					//Set key material
					context.setActiveKeyAndCertMaterial(delegatee);

					// Retrieve and authenticate other accumulated 
					// message-level credentials (e.g., GII delegated assertions, etc.)
					@SuppressWarnings("unchecked")
					ArrayList<GIICredential> bearerCredentials =
						(ArrayList<GIICredential>) context
						.getTransientProperty(GIICredential.CALLER_CREDENTIALS_PROPERTY);			

					// Finally add all of our callerIds to the calling-context's 
					// outgoing credentials 
					TransientCredentials transientCredentials = 
						TransientCredentials.getTransientCredentials(context); 
					transientCredentials._credentials.addAll(bearerCredentials);

					ContextFileSystem.store(contextFile, transientFile, context);
					result.contextFileSize = contextFile.length();
					result.transientFileSize = transientFile.length();
					// copy it over to the TSI machine
					copyGenIIStateDirToTSI(configuration, stateDir, action);

				} finally {
					// now delete temporary files
					FileUtils.deleteDirectory(temp);
				}


				if(manager instanceof BasicManager)
				{
					BasicManager bm = (BasicManager) manager;
					parentAction = bm.getActionForUpdate(parentAction.getUUID());
					try {
						pContext = parentAction.getProcessingContext();
						pContext.put(PROP_GENII_DIR_CREATED,true);
						parentAction.setDirty();	
					} finally {
						bm.getActionStore().put(parentAction.getUUID(), parentAction);	
					}
				}

			}

			String parent = action.getExecutionContext().getWorkingDirectory();
			TSI tsi = configuration.getTargetSystemInterface(client);
			String tsiFilesep = tsi.getFileSeparator(); 
			result.stateDirPath = parent + tsiFilesep + GENII_STATE_DIR;
			return result;
		}




	}

	protected static File getTempDirectory()
	throws IOException
	{
		String dirString = Kernel.getKernel().getProperty(ConfigurationSource.DB_DIRECTORY);
		File temp = null;
		if(dirString != null && dirString.trim().length() > 0)
		{
			temp = new File(dirString);	
		}
		if(temp == null || !temp.exists() || !temp.canWrite())
		{
			temp = File.createTempFile("gffs", "");

			if(!(temp.delete()))
			{
				throw new IOException("Could not delete temp file: " + temp.getAbsolutePath());
			}

			if(!(temp.mkdir()))
			{
				throw new IOException("Could not create temp directory: " + temp.getAbsolutePath());
			}
			temp = new File(temp,Long.toString(System.nanoTime()));

			if(!(temp.mkdir()))
			{
				throw new IOException("Could not create temp directory: " + temp.getAbsolutePath());
			}
		}
		else
		{
			temp = new File(temp,"genii_"+UUID.randomUUID().toString());
			temp.mkdir();
		}
		return (temp);
	}


	protected static void copyGenIIStateDirToTSI(Configuration configuration, File genIIDir, Action action)throws Exception{

		Client client = action.getClient();
		TSI tsi = configuration.getTargetSystemInterface(client);
		String tsiFilesep = tsi.getFileSeparator();
		String parent = action.getExecutionContext().getWorkingDirectory();
		copyFolderToTSI(tsi, genIIDir, genIIDir.getParentFile().getAbsolutePath(), parent, tsiFilesep);

	}

	protected static void copyFolderToTSI(TSI tsi, File f, String ourBasePath, String tsiBasePath, String tsiFileSep) throws Exception
	{
		String relative = f.getAbsolutePath().substring(ourBasePath.length());
		relative.replace(File.separator, tsiFileSep);
		tsi.mkdir(tsiBasePath+tsiFileSep+relative);
		for(File child : f.listFiles())
		{
			if(child.isDirectory())
			{
				copyFolderToTSI(tsi, child, ourBasePath+File.separator+child.getName(), tsiBasePath+tsiFileSep+child.getName(), tsiFileSep);
			}
			else
			{
				OutputStream os = null;
				InputStream is = null;
				try {
					os = tsi.getOutputStream(tsiBasePath+relative+tsiFileSep+child.getName());
					is = new BufferedInputStream(new FileInputStream(child));
					copy(is, os);
				} finally {
					if(is != null)
					{
						try {
							is.close();
						} catch (Exception e) {

						}
					}
					if(os != null)
					{
						try {
							os.close();
						} catch (Exception e) {

						}
					}
				}
			}
		}
	}

	//copy all data from an input stream to an output stream
	protected static void copy(InputStream in, OutputStream out)throws Exception{
		long transferredBytes=0;
		int bufferSize=16384;
		byte[] buffer = new byte[bufferSize];
		int len=0;
		while (true)
		{
			len=in.read(buffer,0,bufferSize);
			if (len<0)
				break;
			out.write(buffer,0,len);
			transferredBytes+=len;
		}

	}

	private static void delegateContext(ICallingContext ctxt, 
			KeyAndCertMaterial delegatee, KeyAndCertMaterial delegator) 
	throws GenesisIISecurityException{

		// get a copy of the original credentials
		@SuppressWarnings("unchecked")
		ArrayList<GIICredential> oldCredentials = (ArrayList<GIICredential>) ctxt
		.getTransientProperty(GIICredential.CALLER_CREDENTIALS_PROPERTY);



		ArrayList<GIICredential> delegatedCredentials =
			new ArrayList<GIICredential>();

		if (delegator != null && delegator._clientCertChain != null) 
			oldCredentials.add(new X509Identity(delegator._clientCertChain));

		try
		{

			Iterator<GIICredential> itr = oldCredentials.iterator();
			while (itr.hasNext())
			{
				GIICredential cred = itr.next();

				if (cred instanceof UsernamePasswordIdentity)
				{
					// We do not delegate Username/Password just add to new calling context
					delegatedCredentials.add(cred);
				}
				else if (cred instanceof SignedAssertion)
				{
					// check if we are authorized to use this assertion in our
					// outgoing messages
					SignedAssertion signedAssertion = (SignedAssertion) cred;
					if (signedAssertion.getAuthorizedIdentity()[0].equals(
							delegator._clientCertChain[0]))
					{

						// If we have key material for the delegatee, delegate 
						if ( delegatee._clientCertChain != null)
						{
							DelegatedAttribute delegatedAttribute =
								new DelegatedAttribute(
										new BasicConstraints(
												System.currentTimeMillis() - GenesisIIConstants.CredentialGoodFromOffset,
												GenesisIIConstants.CredentialExpirationMillis, 
												GenesisIIConstants.MaxDelegationDepth),
												signedAssertion, 
												delegatee._clientCertChain);


							// delegate assertions 
							signedAssertion = new DelegatedAssertion(delegatedAttribute, delegator._clientPrivateKey);
							delegatedCredentials.add(signedAssertion);
						}
					}
				}
			}

		}
		catch (GeneralSecurityException e)
		{
			throw new GenesisIISecurityException(
					"Could not delegate calling context: "
					+ e.getMessage(), e);
		}

		ctxt.setTransientProperty(GIICredential.CALLER_CREDENTIALS_PROPERTY, 
				delegatedCredentials);
	}


	/**
	 * Checks for equality and throw an exception if the certs are not equal.
	 * This has been put here in order to be able to mock it easily...
	 * @param transport
	 * @param authorized
	 * @throws AuthZSecurityException
	 */
	public static void checkTransportCertEqualsAuthorizedCert(Certificate transport, Certificate authorized) throws AuthZSecurityException
	{
		if (transport != null && !transport.equals(authorized)) {
			String authorizedString = "null";
			if(authorized != null)
			{
				if(authorized instanceof X509Certificate)
				{
					authorizedString = ((X509Certificate) authorized).getSubjectDN().getName();
				}
				else authorizedString = authorized.toString();
			}
			String transportString = "null";

			if(transport instanceof X509Certificate)
			{
				transportString = ((X509Certificate) transport).getSubjectDN().getName();
			}
			else transportString = transport.toString();
			throw new AuthZSecurityException(
					"The authorised service \""
					+ authorizedString
					+ "\" does not match the incoming message sender which is "+transportString);
		}
	}


	private static PrivateKey getServerPrivateKey() throws Exception
	{
		ISecurityProperties secProps = Kernel.getKernel().getSecurityProperties();
		String keystoreName=secProps.getKeystore();
		String keystoreType=secProps.getKeystoreType();
		if(keystoreType==null)keystoreType="jks";
		String keystorePassword=secProps.getKeystorePassword();
		String keyPassword = secProps.getKeystoreKeyPassword();
		String keystoreAlias=secProps.getKeystoreAlias();
		try{
			KeyStore keyStore;
			logger.debug("Reading from keystore: " + keystoreName);
			keyStore = KeyStore.getInstance(keystoreType);
			File f = new File(keystoreName);
			keyStore.load(new FileInputStream(f), keystorePassword.toCharArray());
			logger.debug("Keystore: " + keystoreName + " successfully loaded");
			if(keystoreAlias==null){
				keystoreAlias = KeystoreChecker.findAlias(keyStore);
				if(keystoreAlias==null){
					throw new IllegalArgumentException("Keystore "+keystoreName+" does not contain any key entries!");
				}
				logger.debug("No alias supplied, loading  <"+keystoreAlias+">");
			}
			else {
				logger.debug("Loading  <"+keystoreAlias+">");
			}
			Certificate[] path=keyStore.getCertificateChain(keystoreAlias);
			if(path==null)throw new IllegalArgumentException("Alias <"+keystoreAlias+"> cannot be found in keystore. Please check your configuration.");
			PrivateKey privateKey=(PrivateKey)keyStore.getKey(keystoreAlias, keyPassword.toCharArray());
			if(privateKey==null)throw new IllegalArgumentException("Alias <"+keystoreAlias+"> does not denote a key entry. Please check your configuration.");
			return privateKey;

		}catch(Exception e){ 
			//always deadly, so we will throw a runtime exeption
			logger.fatal("Could not load certificate(s) from keystore.",e);
			throw new RuntimeException(e);
		}

	}




	/**
	 *  
	 * @return PEM Encoded String of a Certificate
	 * @throws Exception 
	 */
	protected static KeyAndCertMaterial generateUnprivilegedCertificate(ISecurityProperties secProps) throws Exception{
		long startTime = System.currentTimeMillis()- GenesisIIConstants.CredentialGoodFromOffset;
		long endTime = startTime + 30*3600*1000;
		String keyLengthProp= UAS.getProperty(PROP_DELEGATION_KEYLENGTH);
		if(keyLengthProp == null) keyLengthProp = "2048";
		int keyLength = Integer.parseInt(keyLengthProp);
		String signatureAlgorithm="SHA1withRSA";

		X509V3CertificateGenerator certGen = new X509V3CertificateGenerator();
		KeyPairGenerator kpg=KeyPairGenerator.getInstance(secProps.getPrivateKey().getAlgorithm());
		kpg.initialize(keyLength);
		KeyPair pair = kpg.generateKeyPair();

		Random rand=new Random();
		X500Principal subjectDN = new X500Principal("CN="+UUID.randomUUID().toString()+"-unprivilegedcert");

		certGen.setSerialNumber(new BigInteger(20,rand));
		certGen.setIssuerDN(secProps.getCertificateChain()[0].getSubjectX500Principal());
		certGen.setSubjectDN(subjectDN);
		certGen.setNotBefore(new Date(startTime));
		certGen.setNotAfter(new Date(endTime));
		certGen.setPublicKey(pair.getPublic());
		certGen.setSignatureAlgorithm(signatureAlgorithm);

		X509Certificate certificate = certGen.generate(secProps.getPrivateKey());

		certificate.checkValidity(new Date());
		certificate.verify(secProps.getPublicKey().getPublicKey());
		KeyAndCertMaterial result = new KeyAndCertMaterial(new X509Certificate[]{certificate}, pair.getPrivate());
		return result;
	}


}
