package eu.unicore.genii.security;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Serializable;
import java.security.KeyStore;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.codehaus.xfire.MessageContext;
import org.codehaus.xfire.handler.AbstractHandler;
import org.codehaus.xfire.handler.Phase;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;
import org.xml.sax.InputSource;

import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.wsrflite.security.util.AttributeHandlingCallback;
import edu.virginia.vcgr.genii.client.context.CallingContextImpl;
import edu.virginia.vcgr.genii.client.context.CallingContextUtilities;
import edu.virginia.vcgr.genii.client.context.ICallingContext;
import edu.virginia.vcgr.genii.client.security.authz.AuthZSecurityException;
import edu.virginia.vcgr.genii.client.ser.ObjectDeserializer;
import edu.virginia.vcgr.genii.context.ContextType;
import edu.virginia.vcgr.genii.security.credentials.GIICredential;
import edu.virginia.vcgr.genii.security.credentials.assertions.DelegatedAssertion;
import edu.virginia.vcgr.genii.security.credentials.assertions.SignedAssertion;
import eu.unicore.genii.GenesisConstants;
import eu.unicore.security.Client;
import eu.unicore.security.Role;
import eu.unicore.security.SecurityTokens;
import eu.unicore.security.xfireutil.AuthInHandler;

public class GAMLAssertionInHandler extends AbstractHandler implements GenesisConstants {

	private static final Logger logger = LogUtil.getLogger(LogUtil.SECURITY,GAMLAssertionInHandler.class);

	private static KeyStore trustStore;
	private static Object LOCK = new Object();

	private final Kernel kernel;

	private static final String defaultDir = DEFAULT_TRUSTED_ID_PROVIDER_DIR;

	public GAMLAssertionInHandler(Kernel kernel){
		this.kernel=kernel;
		setPhase(Phase.POLICY);
		after(AuthInHandler.class.getName());
		addAttributeCallback();
		synchronized (LOCK) {
			if(trustStore == null)
			{
				reloadTrustStore();
			}
		}
	}

	private void reloadTrustStore()
	{

		String dir = kernel.getProperty(PROP_TRUSTED_ID_PROVIDER_DIR, defaultDir);
		try {
			trustStore = TruststoreUtil.loadTruststoreFromDirectory(dir);
		} catch (Exception e) {
			logger.error("Unable to setup trusted identity authorities: "+e.getMessage(), e);
		}

	}

	//add callback that extracts attributes when creating a client
	private void addAttributeCallback(){
		kernel.getSecurityManager().addCallback(getCallback());
	}


	/**
	 * This method retrieves GenesisII's GAML Assertion from the SOAP Header, generates an unprivileged certificate, 
	 * and puts this information into the the security context for running jobs with GenesisII based data stagings.  
	 * 
	 */
	public void invoke(MessageContext ctx) throws Exception
	{	

		SecurityTokens securityTokens = (SecurityTokens) ctx.getProperty(
				SecurityTokens.KEY);
		if (securityTokens == null)
		{
			throw new Exception("No security info in headers. Wrong configuration: " +
					AuthInHandler.class.getCanonicalName() + " handler" +
			" must be configure before this handler.");

		}

		// get the SOAP header
		Element header = ctx.getInMessage().getHeader();
		if (header == null){
			return ;	
		} 

		// get gaml header extensions
		@SuppressWarnings("rawtypes")
		List extensions = header.getChildren(GENII_GAMLASSERTION, GENII_NS);
		if(extensions.size()==0){
			logger.debug("No GAML Assertion header found.");
			return;
		}

		try {

			Element gamlElement = (Element) extensions.get(0);		
			ByteArrayOutputStream bos=new ByteArrayOutputStream();
			new XMLOutputter(Format.getPrettyFormat()).output(gamlElement, bos);

			String gamlAsString = bos.toString(); 
			ByteArrayInputStream in = new ByteArrayInputStream(gamlAsString.getBytes());
			InputSource source = new InputSource(in);

			ContextType ct = (ContextType) ObjectDeserializer.deserialize(
					source, ContextType.class);
			ICallingContext callingContext =
				CallingContextUtilities.setupCallingContextAfterCombinedExtraction(
						new CallingContextImpl(ct));
			
			@SuppressWarnings("unchecked")
			Collection<Serializable> signedAssertions = (Collection<Serializable>)
			callingContext.getTransientProperty(GIICredential.CALLER_CREDENTIALS_PROPERTY);
			
			validateAssertions(securityTokens,signedAssertions);

			boolean foundUser = determineUser(signedAssertions,securityTokens);
			if(!foundUser)
			{
				
				// don't throw an exception here..
				// this might be a read-only operation by some Genesis agent
				// e.g. the Grid queue
				if(logger.isDebugEnabled()){
					String message = "Trust delegation was issued by untrusted/unknown user. These are the delegation chains: "+signedAssertions;
					logger.debug(message);		
				}
				
			}


			if(logger.isDebugEnabled()){
				logger.debug("Extracted GAML header:\n"+gamlAsString);		
			}

			// get the security context
			SecurityTokens tokens=((SecurityTokens)ctx.getProperty(SecurityTokens.KEY));		

			if (tokens==null){
				//should never happen in production (always have AuthInHandler)
				logger.warn("No security tokens found, new ones created...");			
				tokens=new SecurityTokens();
				ctx.setProperty(SecurityTokens.KEY, tokens);
			}

			tokens.getContext().put(GENII_GAMLASSERTION_REF, callingContext);
		} catch (Throwable t) {
			Exception e;
			if(t instanceof Exception) e = (Exception) t;
			else e = new Exception(t);
			logger.info("Problem while reading incoming GAML assertion: "+e.getMessage(),e);
			throw e;
		}
	}


	private boolean determineUser(Collection<Serializable> signedAssertions, SecurityTokens tokens)
	{
		CertPath oldConsignor = tokens.getConsignor();

		if (signedAssertions != null) 
		{

			Iterator<Serializable> itr = signedAssertions.iterator();

			while (itr.hasNext()) 
			{
				GIICredential cred = (GIICredential) itr.next();
				if (cred instanceof SignedAssertion)
				{
					SignedAssertion signedAssertion = (SignedAssertion) cred;
					if(signedAssertion.getAttribute() == null) continue;
					X509Certificate[] identity = signedAssertion.getAttribute().getAssertingIdentityCertChain();

					
					
					if(TruststoreUtil.isFromTrustedAuthority(trustStore,identity))
					{
						try {
							CertificateFactory cf = CertificateFactory.getInstance("X.509");
							CertPath certPath = cf.generateCertPath(Arrays.asList(identity));
							tokens.setConsignor(certPath);

							Client client = kernel.getSecurityManager().createAndAuthoriseClient(tokens);
							if(client.getRole() != null && !Role.ROLE_ANONYMOUS.equals(client.getRole().getName()))
							{
								tokens.setUser(certPath);
								return true;
							}
						} catch (Exception e) {
							logger.warn(e.getMessage(), e);
							continue;
						}

					}

				}
			}
		}

		if(oldConsignor != null) tokens.setConsignor(oldConsignor);
		return false;
	}




	private void validateAssertions(SecurityTokens securityTokens, Collection<Serializable> signedAssertions) throws Exception
	{

		Certificate serverCert = kernel.getSecurityManager().getServerCert();
		// Deserialize the encoded caller-credentials
		if (signedAssertions != null) 
		{

			Certificate consignorCert = securityTokens.getConsignorCertificate();

			Iterator<Serializable> itr = signedAssertions.iterator();
			while (itr.hasNext()) 
			{
				GIICredential cred = (GIICredential) itr.next();
				if (cred instanceof SignedAssertion)
				{
					// Holder-of-key token
					SignedAssertion signedAssertion = (SignedAssertion) cred;

					// Check validity and verify integrity
					signedAssertion.checkValidity(new Date());
					signedAssertion.validateAssertion();


					// If the assertion is pre-authorized for us, unwrap one
					// layer
					if ((serverCert != null)
							&& (signedAssertion.getAuthorizedIdentity()[0]
							                                            .equals(serverCert)))
					{
						if (!(signedAssertion instanceof DelegatedAssertion))
						{
							throw new AuthZSecurityException(
							"Found unknown assertion type in SOAP header. This is a Genesis II - Unicore 6 interop problem.");
						}
						signedAssertion =
							((DelegatedAssertion) signedAssertion).unwrap();
					}

					// Verify that the request message signer is the same as the
					// one of the holder-of-key certificates
					TrustDelegationUtils.checkTransportCertEqualsAuthorizedCert(consignorCert, signedAssertion.getAuthorizedIdentity()[0]);

				}
			}
		}
	}



	private static AttributeHandlingCallback aac;

	private synchronized AttributeHandlingCallback getCallback(){
		if(aac==null){
			aac=new GAMLAssertionAttributeCallback();
		}
		return aac;
	}




	/**
	 * callback functions
	 */
	private static class GAMLAssertionAttributeCallback implements AttributeHandlingCallback{
		/**
		 * gets the gaml assertion and an unprivileged cert from the sec tokens and returns it as an attribute
		 * which will be stored in the client
		 */
		public Map<String,Serializable> extractAttributes(SecurityTokens tokens) {
			Map<String,Serializable> result = new HashMap<String,Serializable>();

			String gaml= (String)tokens.getContext().get(GENII_GAMLASSERTION);

			if(gaml!=null){
				result.put(GENII_GAMLASSERTION_REF , gaml);
			}
			return result;
		}
	}


}
