package de.pc2.unicore.edgi.xnjs.g3bridge;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/*
 * @brief serve as java-wrapper for the file-up- and -download-mechanism
 * 
 * remotely downloaded files are provides by an http-server
 *    => a local file is accessible via http-get
 */
public class HttpFileService {
	
	private File path_localprefix;
	private URL url_httpprefix;

	public HttpFileService(File path_localprefix, URL url_httpprefix) throws IOException{
		this(path_localprefix, url_httpprefix, false);
	}
	
	public HttpFileService(File path_localprefix, URL url_httpprefix, boolean auto_mkdir) throws IOException {
		this.path_localprefix = path_localprefix;
		this.url_httpprefix = url_httpprefix;
	
		if (auto_mkdir){
			if (!this.path_localprefix.exists()) {
				if (!this.path_localprefix.mkdirs()){
					throw new IOException("directory " + this.path_localprefix.getPath() + " can't be created.");
				}
			}
		}
		
		if (!this.path_localprefix.isDirectory()) {
			throw new IOException("directory " + this.path_localprefix.getPath() + " not exists.");
		} else if (!this.path_localprefix.canWrite()) {
			throw new IOException("directory " + this.path_localprefix.getPath() + " is not writable.");
		}
	}
	
	public File getPath_localprefix() {
		return this.path_localprefix;
	}
	
	public URL getUrl_httpprefix() {
		return this.url_httpprefix;
	}
	
	public static class Info {
		public File localFileName = null;
		public URL remoteURL = null;
		public String md5hash = null;
		
		public Info() {}
	}
	
	public HttpFileService.Info provideFile(File sourceFile, String relProvidedPath) throws IOException {
		return provideFile(sourceFile, relProvidedPath, true);
	}
	/*
	 * @brief provides a file via Http (by copying it to the correct directory)
	 * @param absFilePath the absolute path of the source file
	 * @param relProvidedPath the relative path of the remotely provided file
	 * @returns the remote url to access the file
	 */
	public HttpFileService.Info provideFile(File sourceFile, String relProvidedPath, boolean computeMd5) throws IOException {
		if (!sourceFile.isFile()) {
			throw new IOException("unexcepted path '"+sourceFile.getAbsolutePath()+"': not a path to a file.");
		} else if (!sourceFile.canRead()) {
			throw new IOException("access rights '"+sourceFile.getAbsolutePath()+"': can read file.");
		}
		
		HttpFileService.Info result = new HttpFileService.Info();
		File target = new File(this.path_localprefix, relProvidedPath);
		result.localFileName = target;
		result.remoteURL = new URL(this.url_httpprefix + "/" + relProvidedPath);
		if (!target.exists()){
			File parent = target.getParentFile();
			if (parent != null && !parent.exists()){
				if (!parent.mkdirs()) {
					throw new IOException("failed to create subdirectories for: "+parent.getAbsolutePath());
				}
			}
			if (!target.createNewFile()){
				throw new IOException("failed to create file, subdirectories exists. "+target.getAbsolutePath());
			}
		} 
		if (!target.canWrite()){
			throw new IOException("access rights '"+target.getAbsolutePath()+"': can not write target file.");
		}
		
		FileChannel targetChannel = new FileOutputStream(target).getChannel();
		FileInputStream sourceInputStream = new FileInputStream(sourceFile);
		FileChannel sourceChannel = sourceInputStream.getChannel();
		targetChannel.transferFrom(sourceChannel, 0, sourceChannel.size());
		
		if (computeMd5) {

			MessageDigest md5;
			try {
				md5 = MessageDigest.getInstance("MD5");
				sourceInputStream = new FileInputStream(sourceFile);
				sourceChannel = sourceInputStream.getChannel();
				long chunksize = Math.min(sourceChannel.size(), 1024*1024*30); //30mb
				for (long position = 0; position < sourceChannel.size();)  
				{  
				    long curr_chunksize = Math.min(sourceChannel.size()-position, chunksize);  
				    ByteBuffer buffer = sourceChannel.map(FileChannel.MapMode.READ_ONLY, position, curr_chunksize);  
				    md5.update(buffer);  
				    position += curr_chunksize;  
				}
				
				// md5 byte array -> hex string
				BigInteger bigInt = new BigInteger(1,md5.digest());
				StringBuilder hashtext = new StringBuilder();
				String hash = bigInt.toString(16);
				// Now we need to zero pad it if you actually want the full 32 chars.
				//for (int count = 32-hashtext.length(); count > 0; count--) {
				//	hashtext.append("0");
				//}
				hashtext.append(hash);
				result.md5hash = hashtext.toString();
			} catch (NoSuchAlgorithmException e) {
				e.printStackTrace();
			}  
		}
				
		return result;
	}
	
	
	public boolean delete(String relProvidedPath) throws IOException {
		String path = this.path_localprefix + "/" + relProvidedPath;
		File file = new File(path);
		if (!file.isFile()) {
			throw new IOException("unexcepted path '"+relProvidedPath+"': not a path to a file.");
		} else if (!file.canRead()) {
			throw new IOException("access rights '"+relProvidedPath+"': can read file.");
		}		
		
		return file.delete();
	}
	
}
