package de.pc2.unicore.edgi.xnjs;

import java.io.IOException;
import java.net.MalformedURLException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Logger;

import de.fzj.unicore.wsrflite.Kernel;
import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.ems.Action;
import de.fzj.unicore.xnjs.ems.ActionStatus;
import de.fzj.unicore.xnjs.ems.ExecutionContext;
import de.fzj.unicore.xnjs.ems.ExecutionException;
import de.fzj.unicore.xnjs.ems.InternalManager;
import de.fzj.unicore.xnjs.ems.Manager;
import de.fzj.unicore.xnjs.ems.event.ContinueProcessingEvent;
import de.fzj.unicore.xnjs.io.XnjsFile;
import de.fzj.unicore.xnjs.management.Dependency;
import de.fzj.unicore.xnjs.management.Lifecycle;
import de.fzj.unicore.xnjs.tsi.ApplicationInfo;
import de.fzj.unicore.xnjs.util.LogUtil;
import de.pc2.unicore.edgi.idb.ApplicationRepository;
import de.pc2.unicore.edgi.idb.ApplicationRepositoryException;
import de.pc2.unicore.edgi.monitoring.ReportException;
import de.pc2.unicore.edgi.monitoring.XMLLogReport;
import de.pc2.unicore.edgi.xnjs.UtilAction.StagingEntry;
import de.pc2.unicore.edgi.xnjs.g3bridge.G3BridgeClient;
import de.pc2.unicore.edgi.xnjs.g3bridge.G3BridgeException;
import de.pc2.unicore.edgi.xnjs.g3bridge.JobDescription;
import de.pc2.unicore.edgi.xnjs.g3bridge.JobDescription.Status;
import de.pc2.unicore.edgi.xnjs.g3bridge.LogicalFileDescription;
import de.pc2.unicore.edgi.xnjs.io.passthrough.PassthroughCapabilities;
import eu.unicore.security.Client;


/**
 * Handles interactions with the desktop grid 3G Bridge
 */
@Lifecycle(isSingleton=true)
@Dependency(classes={ApplicationRepository.class})
public class DesktopGridManager {

	private static final Logger log = LogUtil.getLogger(LogUtil.TSI, DesktopGridManager.class);

	private final Configuration configuration;

	private final G3BridgeClient g3bridge ;

	private final Map<String, JobDescription> dgInfo = new HashMap<String, JobDescription>();

	protected String desktop_grid_queue_name = "Null";
	protected String local_filespace = null;
	protected String remote_filespace = null;
	protected long interval_check_subactions_sec = 5;
	private XMLLogReport monitoring;
	private MonitoringHelperThread monitoringThread ;

	protected String optionStdout = null;
	protected String optionStderr = null;
	protected boolean optionUserDN = false;

	private final ApplicationRepository appAR;

	public DesktopGridManager(Configuration conf) throws G3BridgeException, ExecutionException, MalformedURLException {
		this.configuration = conf;
		UtilConfigWrapper helper = new UtilConfigWrapper(conf, log);

		desktop_grid_queue_name = helper.getProperty("DGTSI.desktop_grid_queue_name", desktop_grid_queue_name);
		local_filespace = helper.getProperty("XNJS.filespace", null);
		remote_filespace = helper.getProperty("DGTSI.remote", null);
		optionStdout = helper.getProperty("DGTSI.stdout", "");
		optionStderr = helper.getProperty("DGTSI.stderr", "");
		optionUserDN = helper.getPropertyBool("DGTSI.allow_proxy_userdn", false);
		if (optionStdout != null && optionStdout.length() == 0) optionStdout = null;
		if (optionStderr != null && optionStderr.length() == 0) optionStderr = null;

		// mkeller:
		// I'm not sure about this registration point
		// but DesktopGridManager is singleton-like so registration happens once
		PassthroughCapabilities.register(conf); 

		Kernel kernel=conf.getComponentInstanceOfType(Kernel.class);
		g3bridge=setupClient(helper, kernel);

		setupMonitoring();

		this.appAR = configuration.getComponentInstanceOfType(ApplicationRepository.class);

		int interval_dgpolling = helper.getPropertyInt("DGTSI.interval_dgpolling", 300);
		Updater x = new DesktopGridManager.Updater(this);
		configuration.getScheduledExecutor().scheduleWithFixedDelay(x, interval_dgpolling, interval_dgpolling, TimeUnit.SECONDS);
		log.info("Will update job states every <"+interval_dgpolling+"> seconds.");
	}

	private void setupMonitoring()throws ExecutionException{
		try {
			//TODO unify this stuff
			this.monitoring = MonitoringHelperThread.createLogReportFromConfiguration(configuration);
			this.monitoringThread = MonitoringHelperThread.createFromConfiguration(configuration, this.monitoring);
			if (this.monitoringThread != null) {
				long interval=monitoringThread.getReport().getReportingInterval();
				configuration.getScheduledExecutor().scheduleWithFixedDelay(monitoringThread, interval, interval, TimeUnit.SECONDS);
			}
		} catch (ReportException e) {
			log.fatal("error during monitoring system init: ", e);
		}		
	}

	private G3BridgeClient setupClient(UtilConfigWrapper helper, Kernel kernel)throws G3BridgeException,ExecutionException{
		String url_monitor = helper.getProperty("DGTSI.url_3gbridge_monitor", "");
		String version_monitor = helper.getProperty("DGTSI.versioncheck_3gbridge_monitor","");
		String url_submitter = helper.getProperty("DGTSI.url_3gbridge_submitter", null);
		String version_submitter = helper.getProperty("DGTSI.versioncheck_3gbridge_submitter", "");
		G3BridgeClient client = new G3BridgeClient(url_submitter, url_monitor, kernel);

		if (version_submitter.length() > 0) {
			String version = client.version_submitter();
			if (!version.contains(version_submitter)) {
				throw new G3BridgeException("Version of 3gbridge mismatched,\n  requirement: '"+version_submitter+"'  (see xnjs.conf)"+
						"\n  url = '"+url_submitter.toString()+"', version = '"+version+"'");
			}
		}

		if (url_monitor != null) {
			if (version_monitor.length() > 0) {
				String version = client.version_monitor();
				if (!version.contains(version_monitor)) {
					throw new G3BridgeException("Version of 3gbridge mismatched,\n  requirement: '"+version_monitor+"'  (see xnjs.conf)"+
							"\n  url = '"+url_monitor.toString()+"', version = '"+version+"'");
				}
			}
		}
		return client;
	}

	public void submit(ApplicationInfo appDescription, Action ucjob) throws ExecutionException, G3BridgeException {
		Client client = ucjob.getClient();
		DesktopGridTSI tsi = (DesktopGridTSI)configuration.getTargetSystemInterface(client);
		ExecutionContext ec = ucjob.getExecutionContext();
		HashMap<String, String> envs = ec.getEnvironment();
		//TODO: Think about replacing the IDB<->AppRep-Couple with custom IncarnateImpl.

		ApplicationRepository.Impl dg_impl = null;
		ApplicationRepository.App dg_app = null;
		String shortcut = appDescription.getExecutable();
		try {
			dg_impl = this.appAR.getImplByShortcut(shortcut);
			dg_app = this.appAR.getAppByShortcut(shortcut);
			if (log.isDebugEnabled()) {
				log.debug("for shortcut '"+shortcut+"': \n"+this.appAR.debugString(dg_impl, dg_app));
			}
		} catch (ApplicationRepositoryException are) { 
			throw new ExecutionException("Application '"+shortcut+"' could not be mapped to an desktop-grid application-repository entry.", are);
		} catch (Exception are) { 
			throw new ExecutionException("Fatal: Application '"+shortcut+"' mapping to DG application-repository entry failed.", are);
		}

		if (ec.isInteractive()) {
			throw new ExecutionException("this TSI/the system can't handle interactive jobs on Desktop Grids.");
		}

		StringBuffer args = new StringBuffer();
		for (String s: appDescription.getArguments()) {
			args.append(s); 
			args.append(" ");
		}
		if (envs.containsKey("ARGUMENTS")) {
			// workaround for rich-client generic grid submission
			args.append(envs.get("ARGUMENTS"));
		}

		UtilAction ucjobutil = new UtilAction(ucjob);
		ArrayList<StagingEntry> stage_entries = ucjobutil.getStagingInEntries_dgconverted(local_filespace, remote_filespace);
		ArrayList<LogicalFileDescription> inputs = new ArrayList<LogicalFileDescription>();
		ArrayList<String> outputs = new ArrayList<String>();
		//List<DataStageInInfo> sin = ucjob.getStageIns();
		//List<DataStageInInfo> sout = ucjob.getStageOuts();


		/*
		//
		// "input files"  - add app-files specified in the AR 
		//
		for (String file_url: dg_impl.files) {
			String[] split = file_url.split("/");
			String fn = split[split.length-1];
			inputs.add(new LogicalFileDescription(fn, file_url));
		}
		 */

		//
		// input files   -  data input   
		//
		if (stage_entries != null) {
			for (StagingEntry e: stage_entries) {
				LogicalFileDescription ldf = null;
				if (e.uric != null) {
					// url passthrough => e.uric contains all info (path,md5,size)
					ldf = new LogicalFileDescription(
							e.filename, e.uric.uri.toString(),
							e.uric.md5, e.uric.size);
				} else {
					// no passthrough => info (path,md5,size) from local file
					XnjsFile xf = tsi.getProperties(e.file.getAbsolutePath());
					assert xf != null;
					String md5 = "";
					// TODO: UPGRADE: compute md5 every time 
					try {
						md5 = XnjsFileDB.computeMD5(e.file.getAbsolutePath());
					} catch (NoSuchAlgorithmException e1) {
						log.error("failed to compute info about local file (e.g. md5)", e1);
					} catch (IOException e1) {
						log.error("failed to compute info about local file (e.g. md5)", e1);
					}
					ldf = new LogicalFileDescription(
							e.filename, e.uri,
							md5, xf.getSize());
				}
				inputs.add(ldf);									
			}
		}
		//
		// output files
		//
		stage_entries = ucjobutil.getStagingEntries(false);
		if (stage_entries != null) {
			for (StagingEntry e: stage_entries) {
				outputs.add(e.filename);
			}
		}
		if (this.optionStdout != null) { 
			ec.setStdout(this.optionStdout);
			outputs.add(this.optionStdout);
		} 
		if (this.optionStderr != null) { 
			ec.setStderr(this.optionStderr);
			outputs.add(this.optionStderr);
		}

		//
		// set env 
		//
		List<String> dgenvs = new ArrayList<String>();

		if (this.optionUserDN) {
			dgenvs.add("PROXY_USERDN=" + ucjob.getClient().getDistinguishedName());
		} else {
			dgenvs.add("PROXY_USERDN=local_disabled_feature");
		}
		String tags = "";

		//
		// JobDescription - put everything together 
		//
		List<JobDescription> dgjobs = new ArrayList<JobDescription>();
		dgjobs.add(new JobDescription(
				dg_app.name, // alg
				desktop_grid_queue_name, //  grid
				args.toString(),  // args
				inputs, outputs, 
				dgenvs, tags)
				);

		ucjob.addLogTrace("submitting job.");
		this.g3bridge.submit(dgjobs);
		if (dgjobs.get(0).id == null) {
			ucjob.addLogTrace("submitting failed, see logs of U/X site.");
		}
		ucjob.setBSID(dgjobs.get(0).id);
		ucjob.addLogTrace("job submitted, desktop-grid-ID (3gBridgeID) = '" + ucjob.getBSID() + "'");
		//ucjob.setStatus(ActionStatus.RUNNING);
		log.debug("DEBUG current status " + ucjob.getStatusAsString());

		StringBuffer input_grid_info = new StringBuffer();
		input_grid_info.append("UNICORE/");
		if (client.getXlogin().getGroup() != null){
			input_grid_info.append(client.getXlogin().getGroup()); //that's good enough because group-namess are usually similar to VO-names
		}

		this.monitoring.jobEntry(ucjob.getUUID(), dgjobs.get(0).alg, input_grid_info.toString());
		this.monitoring.jobSubmission(ucjob.getUUID(), ucjob.getStatusAsString(), 
				this.appAR.getDgId(), ucjob.getBSID());

		ucjob.getProcessingContext().put(JobDescription.class, dgjobs.get(0));
	}

	/**
	 * returns dg-job-representation from internal mapping, 
	 * new mapping added for valid ucjob.bsid  => mapping got lost in edge cases (restart of xnjs) 
	 * @param ucjob
	 * @return JobDescription of dg-job of corresponding uc-job, null if ucjob.bsid is empty (before submit or failed submit)
	 */
	private JobDescription getJobDescriptionFromAction(Action ucjob) {
		if (ucjob.getBSID() == null) {
			return null;
		}
		JobDescription dgjd = ucjob.getProcessingContext().get(JobDescription.class);

		if (dgjd == null) {
			// updateStatus is called either before submit or in the last application run.
			// add XNJS handled job.
			List<JobDescription> dgjobs = new ArrayList<JobDescription>();
			dgjobs.add(new JobDescription(ucjob.getBSID()));
			this.g3bridge.updateStatus(dgjobs);
			dgjd = dgjobs.get(0);
			ucjob.getProcessingContext().put(JobDescription.class, dgjd);
		}

		return dgjd;
	}


	public void post_monitor_change_status(Action ucjob, String alg) {
		String statusstr = ucjob.getStatusAsString();
		if (statusstr.equals("DONE") || statusstr.equals("POSTPROCESSING")) {
			statusstr = "FINISHED";
		}		
		this.monitoring.jobChangeStatus(ucjob.getUUID(), alg, statusstr);
	}

	/**
	 * updates the status of a single UNICORE action
	 **/
	public void updateStatus(Action ucjob, JobDescription.Status newstatus) throws G3BridgeException {
		JobDescription dgjd = null;
		synchronized (dgInfo) {
			dgjd = dgInfo.get(ucjob.getUUID());	
		}
		if (dgjd == null)return;

		ucjob.getProcessingContext().put(JobDescription.class, dgjd);
		
		JobDescription.Status dgstatus = dgjd.status;

		if(log.isDebugEnabled()){
			log.debug("updateStatus (evoke trigger on change) for '"+ucjob.getStatusAsString()+
					"' / (dg:)'"+dgstatus+"'  " + logmsgprefix(ucjob));
		}

		if (dgstatus == JobDescription.Status.UNSET) {
			ucjob.setStatus(ActionStatus.RUNNING);
			post_monitor_change_status(ucjob, dgjd.alg);
		} else if (dgstatus == JobDescription.Status.UNKNOWN) {
			// desktop grid doesn't know the job
			if (ucjob.getStatus() == ActionStatus.RUNNING){
				ucjob.addLogTrace("lost desktop grid job (UNKNOWN) => fail");
				ucjob.fail("lost desktop grid job.");
				post_monitor_change_status(ucjob, dgjd.alg);
			} else {
				log.warn("unkown dgjob-status, ucjob-status '"+ucjob.getStatus()+"' " + logmsgprefix(ucjob));
			}
		} else if (dgstatus == JobDescription.Status.INIT) {
			if (ucjob.getStatus() != ActionStatus.RUNNING) {
				ucjob.addLogTrace("new desktop grid / 3gbridge status = INIT  => uc: RUNNING");
				ucjob.setStatus(ActionStatus.RUNNING);
				post_monitor_change_status(ucjob, dgjd.alg);
			}
		} else if (dgstatus == JobDescription.Status.RUNNING) {
			if (ucjob.getStatus() != ActionStatus.RUNNING) {
				ucjob.addLogTrace("new desktop grid / 3gbridge status = RUNNING  => uc: RUNNING");
				ucjob.setStatus(ActionStatus.RUNNING);
				post_monitor_change_status(ucjob, dgjd.alg);
			}
		} else if (dgstatus == JobDescription.Status.FINISHED) {
			//initiate stageout
			ucjob.getProcessingContext().put(DesktopGridJobProcessor.STAGEOUT_CONTROL, DesktopGridJobProcessor.STAGEOUT_INIT);

		} else if (dgstatus == JobDescription.Status.ERROR) {

			List<JobDescription> dgjobs = new ArrayList<JobDescription>();
			dgjobs.add(dgjd);
			g3bridge.updateOutput(dgjobs);			
			log.error("3gbridge report ERROR state for this job  " + logmsgprefix(ucjob));
			ucjob.addLogTrace("desktop grid/3g bridge report ERROR state for this job = fail; try to download stageout files.");

			//still initiate download, but keep in mind that it has failed
			ucjob.getProcessingContext().put(DesktopGridJobProcessor.STAGEOUT_CONTROL, DesktopGridJobProcessor.STAGEOUT_INIT);
			ucjob.getProcessingContext().put(DesktopGridJobProcessor.FAIL_AFTER_STAGEOUT, 
					Boolean.TRUE);

			// ERROR on dg site
			//  => delete dg-job at 3gbridge
			List<JobDescription> dgjobs2 = new ArrayList<JobDescription>();
			dgjobs2.add(dgjd);
			g3bridge.delete(dgjobs2);
			log.debug("updatestatus (ERROR): forget about " + logmsgprefix(ucjob));

		} else if (dgstatus == JobDescription.Status.TEMPFAILED) {			
			log.warn("3gbridge report TEMPFAILED state for this job.  " + logmsgprefix(ucjob));
			ucjob.addLogTrace("report TEMPFAILED state for this job => waiting for selfhealing");
		}

	}


	public void delete(Action ucjob) throws G3BridgeException {
		JobDescription dgjd = this.getJobDescriptionFromAction(ucjob);
		if (dgjd == null || dgjd.status == JobDescription.Status.UNKNOWN) {
			return;
		}
		deleteOnDG(dgjd);
		ucjob.fail("job is delete due to user command.");
		log.debug("delete: remove entry at desktop grid / 3gbridge " + logmsgprefix(ucjob));		
	}

	public void deleteOnDG(JobDescription dgjd)throws G3BridgeException{
		List<JobDescription> dgjobs = new ArrayList<JobDescription>();
		dgjobs.add(dgjd);
		g3bridge.delete(dgjobs);
	}

	/**
	 * Updates the status of all DG jobs. Changes the "dgInfo" map.
	 *  
	 * @return a map containing *changed* DG job stati (key is the UNICORE action ID)
	 */
	public Map<String, Status> updateStatuses() throws G3BridgeException,ExecutionException {
		Manager mgr=configuration.getEMSManager();
		InternalManager internal=configuration.getInternalManager();

		String[]jobIDs=mgr.list(null);

		Map<String, Status> statuschanges = new HashMap<String, JobDescription.Status>();
		
		if (jobIDs.length == 0) {
			// bypass the remaining function, nothing to do without running jobs.
			return statuschanges;
		}

		// 1) check for known jobs and changes in status
		
		List<JobDescription> dgjobs = new ArrayList<JobDescription>();
		List<JobDescription.Status> oldstatuses = new ArrayList<JobDescription.Status>();
		StringBuilder sb = new StringBuilder();

		for (String ucid: jobIDs) {
			Action job=internal.getAction(ucid);
			JobDescription x = getJobDescriptionFromAction(job);
			dgjobs.add(x);
			oldstatuses.add(x.status);
			sb.append("\n      ucid='"+ucid+"', dgid='"+x.id+
					"' ( dgstatus="+x.status+")");
		}
		if(log.isDebugEnabled()){
			log.debug("updateStatuses: periodic check of desktop grid / 3gbridge entries: "+ sb.toString());
		}
		g3bridge.updateStatus(dgjobs);

		// TODO remove from list if unknown

		// 1b) check changes
		// 1c) update output for finished 
		ArrayList<JobDescription> dgjob_finished = new ArrayList<JobDescription>();
		for (int i = 0; i < dgjobs.size(); i++) {
			JobDescription dgjob = dgjobs.get(i);
			String ucid = jobIDs[i];
			JobDescription.Status oldstatus = oldstatuses.get(i);

			//
			// for dg site unkown jobs:
			//    remove them for dgInfo => they don't need to be considered anymore (for bugs/errors, action-updates will recreate the entry)
			//
			if (dgjob.status == JobDescription.Status.UNKNOWN) {
				if (oldstatus == JobDescription.Status.UNKNOWN || 
						oldstatus == JobDescription.Status.UNSET) {
					continue;
				}					
			}

			//
			//  for jobs with changed status: 
			//    build a list of them for return
			//    build a special list of just finished jobs to query their output 
			//
			if (oldstatus != dgjob.status) {
				statuschanges.put(ucid, dgjob.status);
				log.info("updateStatuses: status changed " 
						+ oldstatus+"' -> '"+ dgjob.status+"' " 
						+ logmsgprefix(ucid, dgjob.id)+" (dg:)'"
						);				
				if (dgjob.status == JobDescription.Status.FINISHED) {
					dgjob_finished.add(dgjob);
				}
			}


		}
		g3bridge.updateOutput(dgjob_finished);

		//update our map now
		synchronized(dgInfo){
			dgInfo.clear();
			for (int i = 0; i < dgjobs.size(); i++) {
				JobDescription dgjob = dgjobs.get(i);
				String ucid = jobIDs[i];
				dgInfo.put(ucid, dgjob);
			}
		}

		return statuschanges;
	}


	static public String logmsgprefix(Action ucjob){
		return logmsgprefix(ucjob.getUUID(), ucjob.getBSID());
	}

	static public String logmsgprefix(String ucid, String dgid) {
		return "[ucid='"+ucid+"'/dgid='"+dgid+"']: ";
	}

	/**
	 * periodically get statuses from DG and fire continue events on status changes
	 */
	public static class Updater implements Runnable {

		private final DesktopGridManager dgm;

		private final InternalManager eh;

		public Updater(DesktopGridManager dgm) {
			this.dgm = dgm;
			this.eh = dgm.configuration.getInternalManager();
		}


		public void run() {
			try {
				Map<String, Status> ucids_dgstatus = dgm.updateStatuses();
				for (Entry<String, Status> id_status: ucids_dgstatus.entrySet()) {
					log.debug("DesktopGridManager.Updater: desktop grid / 3gbridge status change detected ucid: "+id_status.getKey());
					this.eh.handleEvent(new ContinueProcessingEvent(id_status.getKey()));
				}

			} catch (Throwable e) {
				log.warn("DesktopGridManager.Updater: Problem updating DG job states", e);
			} finally {
			}

		}
	};

}
