package de.pc2.unicore.edgi.xnjs;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;

import org.apache.log4j.Logger;

import de.fzj.unicore.xnjs.Configuration;
import de.fzj.unicore.xnjs.ems.ActionStatus;
import de.fzj.unicore.xnjs.ems.ProcessingException;
import de.fzj.unicore.xnjs.io.IFileTransfer;
import de.fzj.unicore.xnjs.io.IFileTransfer.OverwritePolicy;
import de.fzj.unicore.xnjs.io.IFileTransferEngine;
import de.fzj.unicore.xnjs.jsdl.JSDLProcessor;
import de.fzj.unicore.xnjs.management.Dependency;
import de.fzj.unicore.xnjs.util.LogUtil;
import de.pc2.unicore.edgi.xnjs.g3bridge.G3BridgeException;
import de.pc2.unicore.edgi.xnjs.g3bridge.JobDescription;
import de.pc2.unicore.edgi.xnjs.g3bridge.LogicalFileDescription;

/**
 * In addition to the usual job processing, this processor downloads results after the 
 * desktop grid execution, before setting the status to POSTPROCESSING 
 *  
 * @author schuller
 */
@Dependency(classes={DesktopGridManager.class})
public class DesktopGridJobProcessor extends JSDLProcessor {

	private static final Logger logger=LogUtil.getLogger(LogUtil.JOBS,DesktopGridJobProcessor.class);

	public static final String STAGEOUT_CONTROL=DesktopGridJobProcessor.class.getName()+"_stageout";

	public static final String STAGEOUT_INIT="INIT";
	
	public static final String STAGEOUT_RUNNING="RUNNING";

	public static final String STAGEOUT_DONE="DONE";

	public static final String FAIL_AFTER_STAGEOUT=DesktopGridJobProcessor.class.getName()+"_FAIL_after_stageout";

	public DesktopGridJobProcessor(Configuration config) {
		super(config);
	}

	@Override
	protected void handleQueued() throws ProcessingException {
		System.out.println("*** QUEUED "+action.getUUID());
		super.handleQueued();
	}
	
	@Override
	protected void handleRunning() throws ProcessingException {
		System.out.println("*** RUNNING "+action.getUUID());
		super.handleRunning();
		String stageout=(String)action.getProcessingContext().get(STAGEOUT_CONTROL);
		if(STAGEOUT_INIT.equals(stageout)){
			action.setWaiting(false);
			startStageoutFromDG();
		}
		else if(STAGEOUT_RUNNING.equals(stageout)){
			try{
				checkStageoutsFromDG();
			}catch(G3BridgeException e){
				throw new ProcessingException("Error handling stage-outs from desktop grid", e);
			}
		}
	}
	
	private void dgFinished(){
		action.addLogTrace("Desktop grid execution FINISHED");
		action.setStatus(ActionStatus.POSTPROCESSING);
		action.getProcessingContext().put(STAGEOUT_CONTROL, STAGEOUT_DONE);
	}

	private void startStageoutFromDG(){
		JobDescription dgjd=getDGJobDescription();

		if (dgjd.outputs == null || dgjd.outputs.size() == 0) {
			// not outputs => no stageout
			dgFinished();
			DesktopGridManager dg = configuration.getComponentInstanceOfType(DesktopGridManager.class);
			dg.post_monitor_change_status(action, dgjd.alg);
			
		}
		else
		{
			// we have outputs, so start the download(s)
			UtilSubaction ucjobUtil = new UtilSubaction(configuration, action, "DGStageOut");
			IFileTransferEngine fte = configuration.getFileTransferEngine();
			final OverwritePolicy overwrite = OverwritePolicy.OVERWRITE;
			String ucwd = action.getExecutionContext().getWorkingDirectory();
			for (LogicalFileDescription lfd: dgjd.outputs) {
				URI source = null;
				try {
					source = new URI(lfd.url);
				} catch (URISyntaxException e) {
					action.addLogTrace("exception while converting 3gbridge-value to uri: '"+lfd.url+"': " + e.toString());
					continue;
				} 

				if (source != null) {
					IFileTransfer tf = null;
					try {
						tf = fte.createFileImport(action.getClient(), ucwd, source, 
								lfd.logicalName, overwrite, null);
					} catch (IOException e) {
						action.addLogTrace("exception while creating filetransfer '"+source+"'->'"+ucwd+"' '"+lfd.logicalName+"': " + e.toString());
					}

					if (tf != null) {
						ucjobUtil.execSubaction(tf);
						logger.debug("uc:"+action.getUUID()+": execute filetransfer '"+tf.getSource()+"' -> '"+tf.getTarget()+"': '"+tf.getUniqueId()+"'");
						action.addLogTrace("execute filetransfer '"+tf.getSource()+"' -> '"+tf.getTarget()+"': '"+tf.getUniqueId()+"'");
					}
				}
			}
		}

	}

	private void checkStageoutsFromDG()throws G3BridgeException{
		UtilSubaction ucjobUtil = new UtilSubaction(configuration, action, "DGStageOut");
		if(!ucjobUtil.hadSubactions()){
			dgFinished();
		}
		else {
			// check file transfer status
			ArrayList<IFileTransfer> fts = ucjobUtil.getSubactionsAsIFileTransfer();
			for (IFileTransfer ft: fts) {
				if (ft.getStatus() == IFileTransfer.Status.ABORTED) {
					action.addLogTrace("file transfer aborted : '"+ft.getUniqueId()+"'");
					logger.debug("file transfer aborted : '"+ft.getUniqueId()+"'");
					ucjobUtil.abort(ft.getUniqueId());
				} else if (ft.getStatus() == IFileTransfer.Status.FAILED) {
					action.addLogTrace("file transfer failed : '"+ft.getUniqueId()+"'");
					logger.debug("file tranfer failed : '"+ft.getUniqueId()+"'");
					ucjobUtil.abort(ft.getUniqueId());
				} else if (ft.getStatus() == IFileTransfer.Status.DONE) {
					action.addLogTrace("file transfer done : '"+ft.getUniqueId()+"'");
					logger.debug("file transfer done : '"+ft.getUniqueId()+"'");
					ucjobUtil.abort(ft.getUniqueId());
				} else {
					logger.debug("file transfer ongoing."+ft.getUniqueId());
				}
			}

			if (!ucjobUtil.pendingSubactions()) {
				dgFinished();
				if(action.getProcessingContext().get(FAIL_AFTER_STAGEOUT)!=null){
					//DG job was already failed
					deleteDGJob();
				}
			}
		} 
	}

	private JobDescription getDGJobDescription(){
		return action.getProcessingContext().get(JobDescription.class);
	}

	private void deleteDGJob()throws G3BridgeException{
		JobDescription dgjd=getDGJobDescription();
		DesktopGridManager dg = configuration.getComponentInstanceOfType(DesktopGridManager.class);
		dg.post_monitor_change_status(action, dgjd.alg);
		dg.deleteOnDG(dgjd);
	
	}
}
