package de.pc2.unicore.edgi.monitoring;

import java.io.File;
import java.io.FileWriter;
import java.net.MalformedURLException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import de.fzj.unicore.xnjs.util.LogUtil;

/**
 * This class represents generalized report entry with all available fields.
 */
class ReportEntry {
    public String dt = null;
    public String event = null;
    public String job_id = null;
    public String application = null;
    public String status = null;
    public String input_grid_name = null;
    public String output_grid_name = null;
    public String output_job_id = null;

    public ReportEntry() {
    }
    
    public ReportEntry(
    		String dt, 
    		String event, String job_id, String app, 
    		String status, String input_grid_name, 
    		String output_grid_name) {
    	this.dt = dt;
    	this.event = event;
    	this.job_id = job_id;
    	this.application = app;
    	this.status = status;
    	this.input_grid_name = input_grid_name;
    	this.output_grid_name = output_grid_name;
    }
    
    static public ReportEntry create(
    		String event, String job_id
    ) {
    	ReportEntry result = new ReportEntry();
    	result.dt = ReportEntry.dateToString(new Date());
    	result.event = event;
    	result.job_id = job_id;
    	return result;
    }

    static public String dateToString(Date d) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return format.format(d);
    }
    
    public String toString() {
    	return this.dt + "-" + this.event + "-" + this.job_id;
    }
    
    public String toDebugString() {
    	StringBuilder sb = new StringBuilder();
    	sb.append(this.dt);
    	sb.append(" - ");
    	sb.append(this.event);
    	sb.append(" - ");
    	sb.append(this.job_id);
    	sb.append(" (app: ");
    	sb.append(this.application);
    	sb.append(" )");
    	sb.append(" status=");
    	sb.append(this.status);
    	sb.append(", input_grid=");
    	sb.append(this.input_grid_name);
    	sb.append(", output_grid=");
    	sb.append(this.output_grid_name);
		return sb.toString();
    }
}


/**
 *
 * @amir 
 */
public class XMLLogReport {
	final static Logger log = LogUtil.getLogger(LogUtil.TSI, XMLLogReport.class);
	final static public String version = "2.1"; // xml log event format version
	final static public String PATH_DISABLE = "disable";

    private long reportingInterval = 600; // Seconds
    private long cleanupAgeHour = 24*7; // a week
    private String reportsPath = null;
    private Vector<ReportEntry> list = null;

    public XMLLogReport() throws ReportException {
    	this(600, 24*7, "");
    }
    
    public XMLLogReport(long reportingInterval, long cleanupAgeHour, String reportsPath) throws ReportException {
        list = new Vector<ReportEntry>();
        this.reportingInterval = reportingInterval;
        setReportsPath(reportsPath);
    }
    
    public String statusMessage(String remote_url) {
    	StringBuilder sb = new StringBuilder();
    	sb.append("MONITORING XMLLogReport: ");
    	sb.append(!this.isdisabled());
    	sb.append("\n");
    	sb.append("     interval_save=");
    	sb.append(this.reportingInterval);
    	sb.append("sec, interval_cleanup=");
    	sb.append(this.cleanupAgeHour);
    	sb.append("hours");
    	sb.append("\n     local path=");
    	sb.append(this.reportsPath);
    	sb.append("\n     remote url=");
    	sb.append(remote_url==null?"<not configured>":remote_url);
    	sb.append("\n     (This message can be send to EDGI Monitoring System admins to add this site.)");
    	return sb.toString();
    }
    
    public boolean isdisabled() {
    	return this.reportsPath.equals(XMLLogReport.PATH_DISABLE);
    }
    
    public void setCleanupAge(long hour) throws ReportException {
    	if (hour < 0) {
    		throw new ReportException("invalid argument, has to be positive.");
    	}
    	this.cleanupAgeHour = hour;
    }
    
    public long getCleanupAge() {
    	return this.cleanupAgeHour;
    }

    public void setReportingInterval(long sec) throws ReportException {
    	if (sec < 0) {
    		throw new ReportException("invalid argument, has to be positive.");
    	}
    	this.reportingInterval = sec;
    }
    
    public long getReportingInterval() {
    	return this.reportingInterval;
    }

    public void setReportsPath(String reportsPath) throws ReportException {
    	if (reportsPath.equals(XMLLogReport.PATH_DISABLE)){
    		this.reportsPath = reportsPath;
    	} else {
	        try {
				this.reportsPath = relativPath(reportsPath).getAbsolutePath();
			} catch (MalformedURLException e) {
				e.printStackTrace();
			}
	        if (!lazyCreateDir(this.reportsPath)) {
	       		throw new ReportException("error while creating directory: '"+this.reportsPath+"'");
	        }
    	}
    }
    
    public void setReportsPath(File reportsPath) throws ReportException {
    }
    
    public String getReportsPath() {
    	return this.reportsPath;
    }
    
    private void addEvent(ReportEntry logentry) {
        if (this.isdisabled()) {
        	return;
        }
    	this.list.addElement(logentry);
    	if (log.isDebugEnabled()) {
    		log.debug(logentry.toDebugString());
    	}
    }

    /**
     * Job submission event. Creates entry and stores to temporary array. Use this method
     * if some job is submitted from your infrastructure to some another infrastructure or software.
     */
    public void jobSubmission(
    		String job_id, 
    		String status, 
    		String output_grid,
    		String output_job_id
    ) {
        ReportEntry log = ReportEntry.create("job_submission", job_id);
        log.status = status;
        log.output_grid_name = output_grid;
        log.output_job_id = output_job_id;
        this.addEvent(log);
    }

    /**
     * Job entry event. Creates entry and stores to temporary array. Use this method
     * if some job is submitted to your infrastructure by some another infrastructure or software.
     */
    public void jobEntry(
    		String job_id, 
    		String application, 
    		String input_grid
    ) {
        ReportEntry log = ReportEntry.create("job_entry", job_id);
        log.application = application;
        log.input_grid_name = input_grid;
        this.addEvent(log);
    }

    /**
     * Job status event. Creates entry and stores to temporary array. Use this method
     * when your infrastructure changed the status of some job or became aware of job
     * status change on some other infrastructure.
     */
    public void jobChangeStatus(
    		String job_id, 
    		String application, 
    		String status
    ) {
        ReportEntry log = ReportEntry.create("job_status", job_id);
        log.application = application;
        log.status = status;
        this.addEvent(log);
    }

    /**
     * Use this method if you want to take care of report creation by yourself.
     * WARNING: if you are already running report creation thread, manual method won't work (or
     * should not work)
     * NOTE: to use this method, make sure you call previously method setReportsPath
     */
    public void output() throws ReportException {
        if (this.reportsPath != null) {
            createReport();
        } else {
            throw new ReportException("reportPath variable must be set before creating reports!");
        }
    }
    
    public void cleanup() throws ReportException {
    	cleanup(this.cleanupAgeHour);
    }
    
    /**
     * delete entries/files older than an interval given in hours
     * @param age_hour
     * @throws ReportException
     */
    public void cleanup(long age_hour) throws ReportException {
    	if (this.isdisabled()){
    		return;
    	}
    	if (this.reportsPath == null) {
        	throw new ReportException("reportPath variable must be set before creating reports!");
    	}    	
    	File file = new File(this.getReportsPath());
    	assert file.exists();
    	assert file.isDirectory();
    	long threshold = System.currentTimeMillis() - (age_hour*60*60*1000);
    	for (File fp: file.listFiles()) {
    		if (fp.lastModified() <= threshold) {
    			// this file is older than 'age' hours
    			fp.delete();
    			log.debug("delete old file: " + fp.getAbsolutePath());
    		}
    	}
    }

    /*
     * Method for report creation. Creates a new thread which finishes after report is created.
     * This job is being done on new thread for not to block other parts of software.
     */
    synchronized void createReport() throws ReportException {
    	if (this.isdisabled()){
    		return;
    	}
    	if (this.list.size() == 0) {
    		// avoid creating empty file
    		return;
    	}
		FileWriter file_writer = null;
		long millis = System.currentTimeMillis();
    	File file = new File(this.reportsPath, "report_" + String.valueOf(millis) + ".xml");
    	try {
    		// first try to open the file, this can also fail -> so this.list-items get not removed
    		file_writer = new FileWriter(file);
    	} catch (Exception e) {
            throw new ReportException("error while creating new XML document: '"+file.getAbsolutePath()+"'", e);
        }

        Element root = new Element("report");
        root.setAttribute("timestamp", String.valueOf(millis));
        root.setAttribute("timezone", "GMT");
        root.setAttribute("version", XMLLogReport.version);
    
        int count = 0;
        while (this.list.size() > 0) {

            ReportEntry entry = (ReportEntry) this.list.remove(0);
            count += 1;
            
            Element metric_data = new Element("metric_data");

            if (entry.dt != null) {
                Element element = new Element("dt");
                element.setText(entry.dt);
                metric_data.addContent(element);
            }
            
            if (entry.event != null) {
                Element element = new Element("event");
                element.setText(entry.event);
                metric_data.addContent(element);
            }
            
            if (entry.job_id != null) {
                Element element = new Element("job_id");
                element.setText(entry.job_id);
                metric_data.addContent(element);
            }
            
            if (entry.application != null) {
                Element element = new Element("application");
                element.setText(entry.application);
                metric_data.addContent(element);
            }
            
            if (entry.status != null) {
                Element element = new Element("status");
                if (entry.status.endsWith("$")) {
                	entry.status.replace("$","");
                }
                element.setText(entry.status);
                metric_data.addContent(element);
            }
            
            if (entry.input_grid_name != null) {
                Element element = new Element("input_grid_name");
                element.setText(entry.input_grid_name);
                metric_data.addContent(element);
            }     
            
            if (entry.output_grid_name != null) {
                Element element = new Element("output_grid_name");
                element.setText(entry.output_grid_name);
                metric_data.addContent(element);
            }     
            
            if (entry.output_job_id != null) {
                Element element = new Element("job_id_bridge");
                element.setText(entry.output_job_id);
                metric_data.addContent(element);
            }     
            ///--

            root.addContent(metric_data);
        }

        Document doc = new Document(root);

        try {
            XMLOutputter serializer = new XMLOutputter();
            serializer.setFormat(Format.getPrettyFormat());
            serializer.output(doc, file_writer);

            file_writer.flush();
            file_writer.close();
        } catch (Exception e) {
            throw new ReportException("Error while writing XML document '" + file.getAbsolutePath() + "'!", e);
        }
        
        log.debug("Write "+String.valueOf(count)+" events to "+file.getAbsolutePath());

    }

    File relativPath(String relativ) throws MalformedURLException {
		assert relativ != null;
		// remove leading "file://" prefix - absolute starts with "file:///"
		if (relativ.startsWith("file://")){
			relativ = relativ.substring(0, 7);
		}
		int idx = relativ.indexOf("://");
		if (idx != -1) {
			throw new MalformedURLException("url is not local file: '"+relativ+"'");
		}
		if (relativ.charAt(0) != File.separatorChar) {
			return new File(System.getProperty("user.dir"), relativ);
		}
		return new File(relativ);
	}
 
    boolean lazyCreateDir(String path) {
		File file=new File(path);
		if (!file.exists()) {
			return file.mkdirs();
		}
		return true;
	}
   
}



