package de.fzj.unicore.uas.impl.dsms.operation.helper;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Stack;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import pl.edu.icm.dsms.catalogue.CatalogueClient;
import pl.edu.icm.dsms.catalogue.exception.FileExistsFault;
import pl.edu.icm.dsms.catalogue.exception.FileNotExistsFault;
import pl.edu.icm.dsms.catalogue.exception.InvalidPathFault;
import pl.edu.icm.dsms.catalogue.exception.IsADirectoryFault;
import pl.edu.icm.dsms.catalogue.model.Lda;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.impl.dsms.OperationContext;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;

public class LookupHelper {
	private static final int THREADS_LIMIT = 4;

	private static final Logger logger = Log.getLogger(LogUtil.SERVICES, LookupHelper.class);

	private final OperationContext c;

	private final ClientHelper clientHelper;

	private final CatalogueClient cat;

	public LookupHelper(ClientHelper clientHelper, OperationContext context) throws BaseFault {
		this.clientHelper = clientHelper;
		this.c = context;
		this.cat = clientHelper.getCatalogueClient();
	}

	/**
	 * Lookup for a file and try to auto-synchronize SMS and dCatalogue
	 * 
	 * @param cat - catalogue client
	 * @param path - logical path
	 * @param enableSync2 - if true, syncLookup tries to remove from dCatalogue file which doesn't exist in
	 * sms
	 * @throws Exception
	 */
	public List<Lda> lookup(String path) throws Exception {
		List<Lda> ldas = Collections.emptyList();
		Exception fileNotFoundException = null;
		boolean ssynchronized = false;

		// sync 1 - file in SMS but not in the dCatalogue
		try {
			ldas = cachedLookup(path);
		} catch (FileNotExistsFault e) {
			fileNotFoundException = e;
		} catch (InvalidPathFault e) {
			fileNotFoundException = e;
		}
		if (fileNotFoundException != null) {
			if (c.isSync1()) {
				List<StorageClientWithFile> results = checkIfFileExistsOnSms(path);
				ldas = addToCatalogue(results, path);
				ssynchronized = true;
			} else {
				throw fileNotFoundException;
			}
		}

		// sync 2 - file in dCat but deleted from sms
		if (c.isSync2() && !ssynchronized && ldas.size() == 1 && !ldas.get(0).isDirectory()) {
			Lda lda = ldas.get(0);
			StorageClient storage = clientHelper.getStorageClient(lda);
			try {
				storage.listProperties(lda.getPhysicalName());
			} catch (FileNotFoundException e) {
				if (!cat.isDir(path, c.getdSmsId())) {
					cat.remove(path, c.getdSmsId());
					throw FileNotExistsFault.createFault();
				}
			}
		}
		return ldas;
	}

	private List<Lda> addToCatalogue(List<StorageClientWithFile> results, String path) throws Exception {
		if (results.isEmpty()) {
			throw FileNotExistsFault.createFault();
		}

		StorageClientWithFile firstResult = results.get(0);
		if (results.size() == 1 && !firstResult.getFile().getIsDirectory()) {
			return Arrays.asList(cat.add(path, c.getdSmsId(), firstResult.getClient().getEPR()));
		}

		List<Lda> ldas = new ArrayList<Lda>();
		for (StorageClientWithFile result : results) {
			if (result.getFile().getIsDirectory()) {
				cat.addDir(path, c.getdSmsId());
				String tempFile = path + "/tempFile";
				cat.add(tempFile, c.getdSmsId(), result.getClient().getEPR());
				cat.remove(tempFile, c.getdSmsId());
				ldas.add(new Lda(result.getFile().getPath(), path, true, result.getClient().getEPR()));
			} else {
				throw FileExistsFault.createFault("File " + path + " exists on multiple SMSes");
			}
		}
		return ldas;
	}

	private List<Lda> cachedLookup(String path) throws Exception {
		Lda lda = c.getCache().get(path, c.getdSmsId());
		List<Lda> ldas = null;
		if (lda == null) {
			try {
				ldas = Arrays.asList(cat.lookup(path, c.getdSmsId()));
				c.getCache().put(path, c.getdSmsId(), ldas.get(0));
			} catch (IsADirectoryFault e) {
				ldas = cat.lookupDir(path, c.getdSmsId());
			}
		} else {
			ldas = Arrays.asList(lda);
		}
		return ldas;
	}

	/**
	 * Check if file given by path exists on any SMS. If exists, file will be added to dCatalogue and LDA is
	 * returned.
	 * 
	 * @param cat
	 * @param path
	 * @param dSmsId
	 * @param client
	 * @return LDA if file exists, null otherwise.
	 * @throws Exception
	 */
	public List<StorageClientWithFile> checkIfFileExistsOnSms(String path) throws Exception {
		Stack<Lda> ldas = new Stack<Lda>();
		logger.debug("Looking for file " + path + " on SMSes:");
		for (Lda l : cat.lookupDir("/", c.getdSmsId())) {
			ldas.add(l);
			logger.debug(l.getSmsAddress());
		}

		ExecutorService executor = c.getKernel().getContainerProperties().getThreadingServices().getExecutorService();
		List<Future<StorageClientWithFile>> futures = new ArrayList<Future<StorageClientWithFile>>();
		for (int i = 0; i < THREADS_LIMIT; i++) {
			futures.add(executor.submit(new CheckFileOnSmsCallable(ldas, cat, path, c.getdSmsId(), c
					.getdSmsRoot(), clientHelper)));
		}

		List<StorageClientWithFile> results = new ArrayList<StorageClientWithFile>();
		for (Future<StorageClientWithFile> future : futures) {
			try {
				if (future.get() != null) {
					results.add(future.get());
				}
			} catch (Exception e) {
				logger.error("Can't connect to SMS", e);
			}
		}

		return results;
		/*
		 * if (result != null) { return result; } else { throw FileNotExistsFault.createFault(); }
		 */
	}
}
