package de.fzj.unicore.uas.impl.dsms.operation.helper;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;

import org.unigrids.services.atomic.types.GridFileType;

import pl.edu.icm.dsms.catalogue.CatalogueClient;
import pl.edu.icm.dsms.catalogue.exception.FileNotExistsFault;
import pl.edu.icm.dsms.catalogue.model.Lda;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.impl.dsms.OperationContext;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;

public class AddHelper {

	private final ClientHelper clientHelper;

	private final LookupHelper lookupHelper;

	private final OperationContext c;

	private final CatalogueClient catalogue;

	public AddHelper(ClientHelper clientHelper, LookupHelper lookupHelper, OperationContext context)
			throws BaseFault {
		this.clientHelper = clientHelper;
		this.lookupHelper = lookupHelper;
		this.c = context;
		this.catalogue = clientHelper.getCatalogueClient();
	}

	/**
	 * Add file to Catalogue and return LDA.
	 * 
	 * @param logicalFilename
	 * @param dSmsId
	 * @param catalogue
	 * @param client
	 * @return Object containing StorageClient and physical filename.
	 * @throws Exception
	 */
	public FilenameAndClient addFile(String logicalFilename) throws Exception {
		FilenameAndClient filenameAndClient;
		try {
			filenameAndClient = replaceExistingFile(logicalFilename);
		} catch (FileNotExistsFault e) {
			Lda lda = catalogue.add(logicalFilename, c.getdSmsId());
			filenameAndClient = addToExistingStorage(lda);
			c.getCache().put(logicalFilename, c.getdSmsId(), lda);
		}

		makeParentDirs(filenameAndClient.getFilename(), filenameAndClient.getClient());
		return filenameAndClient;
	}

	private FilenameAndClient replaceExistingFile(String logicalFilename) throws Exception {
		Lda lda = lookupHelper.lookup(logicalFilename).get(0);
		StorageClient storageClient = clientHelper.getStorageClient(lda);
		return new FilenameAndClient(lda.getPhysicalName(), storageClient);
	}

	private FilenameAndClient addToExistingStorage(Lda lda) throws Exception {
		StorageClient storageClient = clientHelper.getStorageClient(lda);
		return new FilenameAndClient(lda.getPhysicalName(), storageClient);
	}

	/**
	 * Make parent directories for filename on given storage.
	 * 
	 * @param filename
	 * @param storage
	 * @throws BaseFault
	 * @throws FileNotFoundException
	 */
	public void makeParentDirs(String filename, StorageClient storage) throws BaseFault,
			FileNotFoundException {
		String parentDir = filename.substring(0, filename.lastIndexOf('/'));
		try {
			GridFileType d = storage.listProperties(parentDir);
			if (d != null && d.getIsDirectory()) {
				return;
			}
		} catch (FileNotFoundException e) {
		} // ok, we'll create it

		String[] pathAr = filename.split("/");
		List<String> pathList = new ArrayList<String>();
		for (String s : pathAr) {
			if (!s.isEmpty()) {
				pathList.add(s);
			}
		}
		if (!c.isCreateRoot()) {
			try {
				storage.listProperties(c.getdSmsRoot());
			} catch (FileNotFoundException e) {
				throw new FileNotFoundException("dSMS root directory " + c.getdSmsRoot()
						+ " doesn't exist on storage " + storage.getEPR().getAddress().getStringValue());
			}
		}

		String path = "";
		for (int i = 0, length = pathList.size(); i < length - 1; i++) {
			path += "/" + pathList.get(i);
			try {
				storage.listProperties(path);
			} catch (FileNotFoundException e) {
				storage.createDirectory(path);
			}
		}
	}

	public static class FilenameAndClient {
		private StorageClient client;

		private String filename;

		public FilenameAndClient(String filename, StorageClient client) {
			this.client = client;
			this.filename = filename;
		}

		public StorageClient getClient() {
			return client;
		}

		public String getFilename() {
			return filename;
		}
	}
}
