package de.fzj.unicore.uas.impl.dsms.meta;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import pl.edu.icm.dsms.catalogue.CatalogueClient;
import pl.edu.icm.dsms.catalogue.CatalogueUtils;
import pl.edu.icm.dsms.catalogue.model.Lda;

import de.fzj.unicore.uas.client.MetadataClient;
import de.fzj.unicore.uas.client.StorageClient;
import de.fzj.unicore.uas.client.TaskClient;
import de.fzj.unicore.uas.impl.dsms.OperationContext;
import de.fzj.unicore.uas.impl.dsms.operation.helper.ClientHelper;
import de.fzj.unicore.uas.impl.dsms.operation.helper.LookupHelper;
import de.fzj.unicore.uas.metadata.ExtractionStatistics;
import de.fzj.unicore.uas.metadata.MetadataManager;
import de.fzj.unicore.uas.metadata.SearchResult;
import de.fzj.unicore.uas.util.LogUtil;
import de.fzj.unicore.wsrflite.xmlbeans.BaseFault;
import eu.unicore.util.Log;

public class DSMSMetadataManager implements MetadataManager {
	private static final Logger logger = Log.getLogger(LogUtil.SERVICES, DSMSMetadataManager.class);

	private final ClientHelper clientHelper;

	private final LookupHelper lookupHelper;

	private final OperationContext c;
	
	private final CatalogueClient catalogue;

	public DSMSMetadataManager(OperationContext context) throws BaseFault {
		this.clientHelper = new ClientHelper(context);
		this.lookupHelper = new LookupHelper(clientHelper, context);
		this.c = context;
		this.catalogue = clientHelper.getCatalogueClient();
	}

	@Override
	public void createMetadata(String resourceName, Map<String, String> metadata) throws IOException {
		try {
			String path = CatalogueUtils.normalizePath(resourceName);
			Lda lda = lookupHelper.lookup(path).get(0);
			StorageClient smsClient = clientHelper.getStorageClient(lda);
			MetadataClient metadataClient = smsClient.getMetadataClient();
			metadataClient.createMetadata(lda.getPhysicalName(), metadata);
		} catch (Exception e) {
			logger.error("Can't create metadata", e);
			throw new IOException("Can't create metadata", e);
		}
	}

	@Override
	public void updateMetadata(String resourceName, Map<String, String> metadata) throws Exception {
		try {
			String path = CatalogueUtils.normalizePath(resourceName);
			Lda lda = lookupHelper.lookup(path).get(0);
			StorageClient smsClient = clientHelper.getStorageClient(lda);
			MetadataClient metadataClient = smsClient.getMetadataClient();
			metadataClient.updateMetadata(lda.getPhysicalName(), metadata);
		} catch (Exception e) {
			throw new IOException("Can't update metadata", e);
		}
	}

	@Override
	public List<SearchResult> searchMetadataByContent(String searchString, boolean isAdvancedSearch)
			throws Exception {
		try {
			List<Lda> ldas = catalogue.lookupDir("/", c.getdSmsId());
			Set<String> addrs = new HashSet<String>();
			Set<Lda> uniqueLdas = new HashSet<Lda>();
			for (Lda l : ldas) {
				if (!addrs.contains(l.getSmsAddress())) {
					addrs.add(l.getSmsAddress());
					uniqueLdas.add(l);
				}
			}
			List<SearchResult> results = new ArrayList<SearchResult>();
			for (Lda lda : uniqueLdas) {
				StorageClient sms = clientHelper.getStorageClient(lda);
				Collection<String> partialResult = sms.getMetadataClient().search(searchString,
						isAdvancedSearch);
				if (partialResult == null) {
					continue;
				}
				for (String filename : partialResult) {
					String normalizedPath = CatalogueUtils.normalizePath(filename);
					String normalizedRoot = CatalogueUtils.normalizePath(c.getdSmsRoot());
					if (!normalizedPath.startsWith(normalizedRoot)) {
						continue;
					}
					String logical = CatalogueUtils.getLogicalName(filename, c.getdSmsRoot());
					SearchResult r = new SearchResult();
					r.setResourceName(logical);
					results.add(r);
				}
			}
			return results;
		} catch (Exception e) {
			throw new IOException("Can't update metadata", e);
		}
	}

	@Override
	public Map<String, String> getMetadataByName(String resourceName) throws Exception {
		try {
			String path = CatalogueUtils.normalizePath(resourceName);
			Lda lda = lookupHelper.lookup(path).get(0);
			StorageClient smsClient = clientHelper.getStorageClient(lda);
			MetadataClient metadataClient = smsClient.getMetadataClient();
			return metadataClient.getMetadata(lda.getPhysicalName());
		} catch (Exception e) {
			throw new IOException("Can't update metadata", e);
		}
	}

	@Override
	public void removeMetadata(String resourceName) throws Exception {
		try {
			String path = CatalogueUtils.normalizePath(resourceName);
			Lda lda = lookupHelper.lookup(path).get(0);
			StorageClient smsClient = clientHelper.getStorageClient(lda);
			MetadataClient metadataClient = smsClient.getMetadataClient();
			metadataClient.deleteMetadata(lda.getPhysicalName());
		} catch (Exception e) {
			throw new IOException("Can't update metadata", e);
		}
	}

	@Override
	public void renameResource(String source, String target) throws Exception {
		Map<String, String> metadata = getMetadataByName(source);
		createMetadata(target, metadata);
		removeMetadata(source);
	}

	@Override
	public void copyResourceMetadata(String source, String target) throws Exception {
		Map<String, String> metadata = getMetadataByName(source);
		createMetadata(target, metadata);
	}

	@Override
	public Future<ExtractionStatistics> startAutoMetadataExtraction(String base, int depthLimit)
			throws Exception {
		try {
			List<Lda> ldas = catalogue.lookupDir(base, c.getdSmsId());
			Set<String> addrs = new HashSet<String>();
			Set<Lda> uniqueLdas = new HashSet<Lda>();
			for (Lda l : ldas) {
				if (!addrs.contains(l.getSmsAddress())) {
					addrs.add(l.getSmsAddress());
					uniqueLdas.add(l);
				}
			}
			List<TaskClient> clients = new LinkedList<TaskClient>();
			for (Lda lda : uniqueLdas) {
				StorageClient smsClient = clientHelper.getStorageClient(lda);
				MetadataClient metadataClient = smsClient.getMetadataClient();
				clients.add(metadataClient.startMetadataExtraction(base, depthLimit));
			}
			return Executors.newSingleThreadExecutor().submit(new CombinedTaskClient(clients));
		} catch (Exception e) {
			throw new IOException("Can't update metadata", e);
		}
	}
}
